<?php

/**
 * SSO Azure AD Client Functions
 * 
 * Functions for communicating with Microsoft Graph API
 * 
 * @package SSO
 */

// require_once(dirname(__FILE__) . '/../createprovider.php');
// require_once(dirname(__FILE__) . '/sso-utilities.php');

/**
 * AJAX - Get users from Azure AD
 * 
 * @param string 'client_id' -- OAuth client ID
 * @param int 'page_size' -- number of items per page (default: 100)
 * @param string 'search' -- search term for filtering users
 * @param string 'next_link' -- full URL from @odata.nextLink for pagination
 * 
 * @echo json array -- [status, data with users array]
 */
function get_users_AAD(){
    check_nagios_session_protector();
    
    $client_id = grab_request_var('client_id', '');
    $page_size = grab_request_var('page_size', 100);
    $search = grab_request_var('search', '');
    $search_mode = grab_request_var('search_mode', 'startswith');
    $next_link = grab_request_var('next_link', '');
    $page_number = grab_request_var('page_number', null);
    
    // Validate search_mode
    if (!in_array($search_mode, ['startswith', 'contains'])) {
        $search_mode = 'startswith';
    }
    
    // Convert page_number to integer if provided
    if ($page_number !== null) {
        $page_number = intval($page_number);
        if ($page_number < 0) {
            $page_number = 0;
        }
    }
    
    if(empty($client_id)){
        echo json_encode(['status' => 'failed', 'message' => _('Client ID is required')]);
        exit;
    }
    
    try {
        // Get OAuth credentials (always needed for authentication, even when using next_link)
        $credentials = get_oauth_credentials($client_id);
        if(!$credentials){
            echo json_encode(['status' => 'failed', 'message' => _('OAuth credentials not found')]);
            exit;
        }
        
        // Create Microsoft Graph API client
        $graph_client = create_graph_client($credentials);
        
        // Initialize session token cache if not exists
        if (!isset($_SESSION['sso_pagination_tokens'])) {
            $_SESSION['sso_pagination_tokens'] = [];
        }
        
        // Create cache key for this query (client_id + search + search_mode)
        $cache_key = md5($client_id . '|' . $search . '|' . $search_mode);
        
        // Initialize cache entry if not exists
        if (!isset($_SESSION['sso_pagination_tokens'][$cache_key])) {
            $_SESSION['sso_pagination_tokens'][$cache_key] = [
                'tokens' => [0 => null], // Page 0 always uses null (no token)
                'last_updated' => time()
            ];
        }
        
        $token_cache = &$_SESSION['sso_pagination_tokens'][$cache_key];
        
        // Helper function to extract $skipToken from a nextLink URL
        $extract_skiptoken = function($next_link_url) {
            if (empty($next_link_url)) {
                return null;
            }
            // Parse the URL to extract $skiptoken parameter
            $parsed_url = parse_url($next_link_url);
            if (isset($parsed_url['query'])) {
                parse_str($parsed_url['query'], $query_params);
                if (isset($query_params['$skiptoken'])) {
                    return $query_params['$skiptoken'];
                }
            }
            return null;
        };
        
        // Helper function to build a request URL with $skiptoken
        $build_request_with_skiptoken = function($base_url, $skiptoken, $page_size, $search, $search_mode) {
            $query_params = ['$top' => $page_size];
            
            if (!empty($search)) {
                $sanitized_search = addslashes($search);
                if ($search_mode === 'contains') {
                    $query_params['$filter'] = "contains(displayName,'$sanitized_search') or contains(mail,'$sanitized_search') or contains(userPrincipalName,'$sanitized_search')";
                } else {
                    $query_params['$filter'] = "startswith(displayName,'$sanitized_search') or startswith(mail,'$sanitized_search') or startswith(userPrincipalName,'$sanitized_search')";
                }
            } else {
                $query_params['$orderby'] = 'displayName';
            }
            
            if ($skiptoken !== null) {
                $query_params['$skiptoken'] = $skiptoken;
            }
            
            return $base_url . '?' . http_build_query($query_params);
        };
        
        // Regular user loading with pagination support
        // Priority: page_number > next_link > default (page 0)
        if ($page_number !== null) {
            // Page number-based navigation
            $target_page = $page_number;
            
            // Check if we have a token for this page
            $token_for_page = isset($token_cache['tokens'][$target_page]) ? $token_cache['tokens'][$target_page] : null;
            
            // If we don't have a token, navigate sequentially from closest cached page
            if ($token_for_page === null && $target_page > 0) {
                // Find the highest cached page number <= target page
                $start_page = 0;
                $start_token = null;
                
                for ($i = $target_page - 1; $i >= 0; $i--) {
                    if (isset($token_cache['tokens'][$i])) {
                        $start_page = $i;
                        $start_token = $token_cache['tokens'][$i];
                        break;
                    }
                }
                
                // Navigate sequentially from start_page to target_page
                $current_nav_page = $start_page;
                $current_nav_token = $start_token;
                
                while ($current_nav_page < $target_page) {
                    if ($current_nav_page === 0 && $current_nav_token === null) {
                        // Fetch page 0 to get token for page 1
                        $query_params = ['$top' => $page_size];
                        
                        if (!empty($search)) {
                            $sanitized_search = addslashes($search);
                            if ($search_mode === 'contains') {
                                $query_params['$filter'] = "contains(displayName,'$sanitized_search') or contains(mail,'$sanitized_search') or contains(userPrincipalName,'$sanitized_search')";
                            } else {
                                $query_params['$filter'] = "startswith(displayName,'$sanitized_search') or startswith(mail,'$sanitized_search') or startswith(userPrincipalName,'$sanitized_search')";
                            }
                        } else {
                            $query_params['$orderby'] = 'displayName';
                        }
                        
                        $headers = [];
                        if (!empty($search)) {
                            $headers['ConsistencyLevel'] = 'eventual';
                        }
                        
                        if (!empty($headers)) {
                            $nav_response = $graph_client->get('https://graph.microsoft.com/v1.0/users?' . http_build_query($query_params), [
                                'headers' => $headers
                            ]);
                        } else {
                            $nav_response = $graph_client->get('https://graph.microsoft.com/v1.0/users?' . http_build_query($query_params));
                        }
                        
                        $nav_data = $nav_response->getBody();
                        $nav_users_data = json_decode($nav_data, true);
                        
                        // Store token for page 0
                        $token_cache['tokens'][0] = null;
                        
                        // Extract token for page 1
                        if (isset($nav_users_data['@odata.nextLink'])) {
                            $next_token = $extract_skiptoken($nav_users_data['@odata.nextLink']);
                            $token_cache['tokens'][1] = $next_token;
                            $current_nav_token = $next_token;
                        } else {
                            throw new Exception('Cannot navigate to target page - reached end of results.');
                        }
                        $current_nav_page = 1;
                    } else {
                        // Check if we already have the token for the next page
                        if (isset($token_cache['tokens'][$current_nav_page + 1])) {
                            $current_nav_page++;
                            $current_nav_token = $token_cache['tokens'][$current_nav_page];
                        } else if ($current_nav_token !== null) {
                            // Navigate forward using the token
                            $request_url = $build_request_with_skiptoken(
                                'https://graph.microsoft.com/v1.0/users',
                                $current_nav_token,
                                $page_size,
                                $search,
                                $search_mode
                            );
                            
                            $headers = [];
                            if (!empty($search)) {
                                $headers['ConsistencyLevel'] = 'eventual';
                            }
                            
                            if (!empty($headers)) {
                                $nav_response = $graph_client->get($request_url, ['headers' => $headers]);
                            } else {
                                $nav_response = $graph_client->get($request_url);
                            }
                            
                            $nav_data = $nav_response->getBody();
                            $nav_users_data = json_decode($nav_data, true);
                            
                            $current_nav_page++;
                            // Store the token we used to get to this page
                            $token_cache['tokens'][$current_nav_page] = $current_nav_token;
                            
                            // Extract token for the next page
                            if (isset($nav_users_data['@odata.nextLink'])) {
                                $next_token = $extract_skiptoken($nav_users_data['@odata.nextLink']);
                                $token_cache['tokens'][$current_nav_page + 1] = $next_token;
                                $current_nav_token = $next_token;
                            } else {
                                if ($current_nav_page < $target_page) {
                                    throw new Exception('Cannot navigate to target page - reached end of results.');
                                }
                                $current_nav_token = null;
                            }
                        } else {
                            throw new Exception('Unable to navigate to target page - missing pagination token.');
                        }
                    }
                }
                
                // Now we should have the token for the target page
                $token_for_page = isset($token_cache['tokens'][$target_page]) ? $token_cache['tokens'][$target_page] : null;
            }
            
            // Fetch the target page
            if ($target_page === 0) {
                // Page 0 - no token needed
                $query_params = ['$top' => $page_size];
                
                if (!empty($search)) {
                    $sanitized_search = addslashes($search);
                    if ($search_mode === 'contains') {
                        $query_params['$filter'] = "contains(displayName,'$sanitized_search') or contains(mail,'$sanitized_search') or contains(userPrincipalName,'$sanitized_search')";
                    } else {
                        $query_params['$filter'] = "startswith(displayName,'$sanitized_search') or startswith(mail,'$sanitized_search') or startswith(userPrincipalName,'$sanitized_search')";
                    }
                } else {
                    $query_params['$orderby'] = 'displayName';
                }
                
                $headers = [];
                if (!empty($search)) {
                    $headers['ConsistencyLevel'] = 'eventual';
                }
                
                if (!empty($headers)) {
                    $response = $graph_client->get('https://graph.microsoft.com/v1.0/users?' . http_build_query($query_params), [
                        'headers' => $headers
                    ]);
                } else {
                    $response = $graph_client->get('https://graph.microsoft.com/v1.0/users?' . http_build_query($query_params));
                }
            } else {
                // Use $skiptoken to fetch the page
                $request_url = $build_request_with_skiptoken(
                    'https://graph.microsoft.com/v1.0/users',
                    $token_for_page,
                    $page_size,
                    $search,
                    $search_mode
                );
                
                $headers = [];
                if (!empty($search)) {
                    $headers['ConsistencyLevel'] = 'eventual';
                }
                
                if (!empty($headers)) {
                    $response = $graph_client->get($request_url, ['headers' => $headers]);
                } else {
                    $response = $graph_client->get($request_url);
                }
            }
            
            // Update cache timestamp
            $token_cache['last_updated'] = time();
            
        } else if(!empty($next_link)){
            // Use the next_link URL directly - it already contains all query parameters
            $response = $graph_client->get($next_link);
        } else {
            $query_params = [
                '$top' => $page_size
            ];
            
            // When using $filter with 'or' conditions, $orderby is not supported by Microsoft Graph API
            // So we only add $orderby when there's no search filter
            if(!empty($search)){
                // Sanitize search term to prevent injection
                $sanitized_search = addslashes($search);
                
                if($search_mode === 'contains'){
                    $query_params['$filter'] = "contains(displayName,'$sanitized_search') or contains(mail,'$sanitized_search') or contains(userPrincipalName,'$sanitized_search')";
                } else {
                    // Default to startswith
                    $query_params['$filter'] = "startswith(displayName,'$sanitized_search') or startswith(mail,'$sanitized_search') or startswith(userPrincipalName,'$sanitized_search')";
                }
            } else {
                $query_params['$orderby'] = 'displayName';
            }
            
            // Get users from Microsoft Graph
            // ConsistencyLevel: eventual header is required when using $filter with search operations
            $headers = [];
            if(!empty($search)){
                $headers['ConsistencyLevel'] = 'eventual';
            }
            
            if(!empty($headers)){
                $response = $graph_client->get('https://graph.microsoft.com/v1.0/users?' . http_build_query($query_params), [
                    'headers' => $headers
                ]);
            } else {
                $response = $graph_client->get('https://graph.microsoft.com/v1.0/users?' . http_build_query($query_params));
            }
        }
        
        $data = $response->getBody();
        $users_data = json_decode($data, true);
        
        // Format users for frontend
        $users = [];
        if(isset($users_data['value'])){
            foreach($users_data['value'] as $user){
                $users[] = [
                    'user_id' => $user['id'],
                    'name' => $user['displayName'],
                    'email' => $user['mail'],
                    'user_id_aad' => $user['id'],
                    'displayName' => $user['displayName'],
                    'userPrincipalName' => $user['userPrincipalName'],
                    'givenName' => $user['givenName'] ?? '',
                    'surname' => $user['surname'] ?? '',
                    'jobTitle' => $user['jobTitle'] ?? '',
                    'department' => $user['department'] ?? '',
                    'officeLocation' => $user['officeLocation'] ?? ''
                ];
            }
        }
        
        // Extract next_link if available
        $next_link_result = isset($users_data['@odata.nextLink']) ? $users_data['@odata.nextLink'] : null;
        $has_more = !empty($next_link_result);
        
        // Update token cache with the next page token if we used page_number
        if ($page_number !== null && $next_link_result !== null) {
            $next_page = $page_number + 1;
            $next_token = $extract_skiptoken($next_link_result);
            $token_cache['tokens'][$next_page] = $next_token;
            $token_cache['last_updated'] = time();
        } else if (empty($next_link) && $page_number === null && $next_link_result !== null) {
            // If we fetched page 0 without page_number, store token for page 1
            $next_token = $extract_skiptoken($next_link_result);
            $token_cache['tokens'][1] = $next_token;
            $token_cache['last_updated'] = time();
        }
        
        $result = [
            'status' => 'success',
            'data' => [
                'users' => $users,
                'nextLink' => $next_link_result,
                'hasMore' => $has_more,
                'pageSize' => $page_size,
                'totalFetched' => count($users)
            ]
        ];
        
        echo json_encode($result);
        exit;
        
    } catch (Exception $e) {
        echo json_encode(['status' => 'failed', 'message' => $e->getMessage()]);
        exit;
    }
}

/**
 * AJAX - Get user count from Azure AD
 * 
 * @param string 'client_id' -- OAuth client ID
 * @param string 'search' -- search term for filtering users
 * @param string 'search_mode' -- search mode: 'startswith' or 'contains' (default: 'startswith')
 * 
 * @echo json array -- [status, data with count number]
 */
function get_usercount_AAD(){
    check_nagios_session_protector();
    
    $client_id = grab_request_var('client_id', '');
    $search = grab_request_var('search', '');
    $search_mode = grab_request_var('search_mode', 'startswith');
    
    // Validate search_mode
    if (!in_array($search_mode, ['startswith', 'contains'])) {
        $search_mode = 'startswith';
    }
    
    if(empty($client_id)){
        echo json_encode(['status' => 'failed', 'message' => _('Client ID is required')]);
        exit;
    }
    
    try {
        // Get OAuth credentials
        $credentials = get_oauth_credentials($client_id);
        if(!$credentials){
            echo json_encode(['status' => 'failed', 'message' => _('OAuth credentials not found')]);
            exit;
        }
        
        // Create Microsoft Graph API client
        $graph_client = create_graph_client($credentials);
        
        // Use $count=true to get total count without loading all users
        $query_params = [
            '$count' => 'true',
            '$top' => 1  // Only get 1 user to minimize data transfer
        ];
        
        if(!empty($search)){
            // Sanitize search term to prevent injection
            $sanitized_search = addslashes($search);
            
            if($search_mode === 'contains'){
                $query_params['$filter'] = "contains(displayName,'$sanitized_search') or contains(mail,'$sanitized_search') or contains(userPrincipalName,'$sanitized_search')";
            } else {
                // Default to startswith
                $query_params['$filter'] = "startswith(displayName,'$sanitized_search') or startswith(mail,'$sanitized_search') or startswith(userPrincipalName,'$sanitized_search')";
            }
        }
        
        $headers = [
            'ConsistencyLevel' => 'eventual'
        ];
        
        $response = $graph_client->get('https://graph.microsoft.com/v1.0/users?' . http_build_query($query_params), [
            'headers' => $headers
        ]);
        $data = $response->getBody();
        $users_data = json_decode($data, true);
        
        $total_count = 0;
        if(isset($users_data['@odata.count'])){
            $total_count = $users_data['@odata.count'];
        } else {
            // Fallback: if count is not available, estimate from returned data
            if(isset($users_data['value'])) {
                $returned_count = count($users_data['value']);
                if(isset($users_data['@odata.nextLink'])) {
                    // If there's a nextLink, this is just the first page
                    $total_count = $returned_count + 1; // At least this many, likely more
                } else {
                    $total_count = $returned_count;
                }
            }
        }
        
        echo json_encode(['status' => 'success', 'data' => $total_count]);
        exit;
        
    } catch (Exception $e) {
        echo json_encode(['status' => 'failed', 'message' => $e->getMessage()]);
        exit;
    }
}

/**
 * AJAX - Get Azure AD groups
 * 
 * @param string 'client_id' -- OAuth client ID
 * @param int 'page' -- page number for pagination
 * @param int 'page_size' -- number of items per page
 * @param string 'search' -- search term for filtering groups
 * 
 * @echo json array -- [status, data with groups array]
 */
function get_groups_AAD(){
    check_nagios_session_protector();
    
    $client_id = grab_request_var('client_id', '');
    $page = grab_request_var('page', 1);
    $page_size = grab_request_var('page_size', 100);
    $search = grab_request_var('search', '');
    
    if(empty($client_id)){
        echo json_encode(['status' => 'failed', 'message' => _('Client ID is required')]);
        exit;
    }
    
    try {
        $credentials = get_oauth_credentials($client_id);
        if(!$credentials){
            echo json_encode(['status' => 'failed', 'message' => _('OAuth credentials not found')]);
            exit;
        }
        $graph_client = create_graph_client($credentials);
        $query_params = [
            '$top' => $page_size
        ];
        
        // When using $filter with 'or' conditions, $orderby is not supported by Microsoft Graph API
        // Only add $orderby when there's no search filter (no 'or' conditions)
        if(!empty($search)){
            $query_params['$filter'] = "startswith(displayName,'$search') or startswith(description,'$search')";
        } else {
           $query_params['$orderby'] = 'displayName';
        }
        
        // Get groups from Microsoft Graph
        $response = $graph_client->get('https://graph.microsoft.com/v1.0/groups?' . http_build_query($query_params));
        $data = $response->getBody();
        $groups_data = json_decode($data, true);
        
        // Format groups for frontend with size validation
        $groups = [];
        if(isset($groups_data['value'])){
            foreach($groups_data['value'] as $group){
                // Get member count for each group using the count endpoint
                $member_count = 0;
                try {
                    // Get member count using the proper Graph API approach
                    // Based on: https://learn.microsoft.com/en-us/graph/api/group-list-members?view=graph-rest-1.0&tabs=http#example-2-get-only-a-count-of-all-membership
                    $headers = [
                        'ConsistencyLevel' => 'eventual'
                    ];
                    $members_response = $graph_client->get("https://graph.microsoft.com/v1.0/groups/{$group['id']}/members?\$count=true", [
                        'headers' => $headers
                    ]);
                    $members_data = json_decode($members_response->getBody(), true);
                    
                    if(isset($members_data['@odata.count'])){
                        $member_count = $members_data['@odata.count'];
                    } else {
                        // Fallback: if count is not available, try without $top=0
                        $fallback_response = $graph_client->get("https://graph.microsoft.com/v1.0/groups/{$group['id']}/members?\$count=true", [
                            'headers' => $headers
                        ]);
                        $fallback_data = json_decode($fallback_response->getBody(), true);
                        
                        if(isset($fallback_data['@odata.count'])){
                            $member_count = $fallback_data['@odata.count'];
                        } else {
                            // Last resort: estimate based on returned data
                            if(isset($fallback_data['value'])) {
                                $returned_count = count($fallback_data['value']);
                                if(isset($fallback_data['@odata.nextLink'])) {
                                    // If there's a nextLink, this is just the first page
                                    $member_count = $returned_count + 1; // At least this many, likely more
                                } else {
                                    $member_count = $returned_count;
                                }
                            } else {
                                $member_count = 0;
                            }
                        }
                    }
                } catch (Exception $e) {
                    log_sso_error("Error getting member count for group {$group['id']}: " . $e->getMessage());
                    $member_count = 0;
                }
                
                // Add size validation and warnings
                $size_validation = validate_group_size($member_count);
                
                $groups[] = [
                    'id' => $group['id'],
                    'displayName' => $group['displayName'],
                    'description' => $group['description'] ?? '',
                    'mail' => $group['mail'] ?? '',
                    'mailEnabled' => $group['mailEnabled'] ?? false,
                    'securityEnabled' => $group['securityEnabled'] ?? false,
                    'groupTypes' => $group['groupTypes'] ?? [],
                    'member_count' => $member_count,
                    'size_validation' => $size_validation
                ];
            }
        }
        
        $result = [
            'status' => 'success',
            'data' => [
                'groups' => $groups,
                'total' => count($groups),
                'page' => 1,
                'page_size' => $page_size,
                'note' => 'Microsoft Graph API does not support $skip parameter. Showing first ' . $page_size . ' results.'
            ]
        ];
        
        echo json_encode($result);
        exit;
        
    } catch (Exception $e) {
        echo json_encode(['status' => 'failed', 'message' => $e->getMessage()]);
        exit;
    }
}

/**
 * AJAX - Get Azure AD groups by specific type
 * 
 * @param string 'client_id' -- OAuth client ID
 * @param string 'group_type' -- Type of group to filter (microsoft365, security, distribution, dynamic, cloud, onpremises)
 * @param int 'page' -- page number for pagination
 * @param int 'page_size' -- number of items per page
 * @param string 'search' -- search term for filtering groups
 * 
 * @echo json array -- [status, data with groups array]
 */
function get_groups_AAD_by_type(){
    check_nagios_session_protector();
    
    $client_id = grab_request_var('client_id', '');
    $group_type = grab_request_var('group_type', '');
    $page = grab_request_var('page', 1);
    $page_size = grab_request_var('page_size', 100);
    $search = grab_request_var('search', '');
    
    if(empty($client_id) || empty($group_type)){
        echo json_encode(['status' => 'failed', 'message' => _('Client ID and group type are required')]);
        exit;
    }
    
    try {
        // Get OAuth credentials
        $credentials = get_oauth_credentials($client_id);
        if(!$credentials){
            echo json_encode(['status' => 'failed', 'message' => _('OAuth credentials not found')]);
            exit;
        }
        
        // Create Microsoft Graph API client
        $graph_client = create_graph_client($credentials);
        
        // Build query parameters with type-specific filters
        $query_params = [
            '$top' => $page_size,
            '$orderby' => 'displayName'
        ];
        
        // Add type-specific filters based on Microsoft Graph API
        $type_filters = [];
        switch($group_type) {
            case 'microsoft365':
                $type_filters[] = "groupTypes/any(c:c eq 'Unified')";
                break;
            case 'security':
                $type_filters[] = "securityEnabled eq true";
                break;
            case 'distribution':
                $type_filters[] = "mailEnabled eq true";
                break;
            case 'dynamic':
                $type_filters[] = "groupTypes/any(c:c eq 'DynamicMembership')";
                break;
            case 'cloud':
                $type_filters[] = "onPremisesSyncEnabled eq null";
                break;
            case 'onpremises':
                $type_filters[] = "onPremisesSyncEnabled eq true";
                break;
            default:
                echo json_encode(['status' => 'failed', 'message' => _('Invalid group type')]);
                exit;
        }
        
        // Combine type filter with search filter
        $all_filters = [];
        if(!empty($type_filters)) {
            $all_filters[] = '(' . implode(' and ', $type_filters) . ')';
        }
        if(!empty($search)){
            $all_filters[] = "(startswith(displayName,'$search') or startswith(description,'$search'))";
        }
        
        if(!empty($all_filters)) {
            $query_params['$filter'] = implode(' and ', $all_filters);
        }
        
        // Get groups from Microsoft Graph
        $response = $graph_client->get('https://graph.microsoft.com/v1.0/groups?' . http_build_query($query_params));
        $data = $response->getBody();
        $groups_data = json_decode($data, true);
        
        // Format groups for frontend with size validation
        $groups = [];
        if(isset($groups_data['value'])){
            foreach($groups_data['value'] as $group){
                // Get member count for each group using the count endpoint
                $member_count = 0;
                try {
                    $headers = [
                        'ConsistencyLevel' => 'eventual'
                    ];
                    $members_response = $graph_client->get("https://graph.microsoft.com/v1.0/groups/{$group['id']}/members?\$count=true", [
                        'headers' => $headers
                    ]);
                    $members_data = json_decode($members_response->getBody(), true);
                    
                    if(isset($members_data['@odata.count'])){
                        $member_count = $members_data['@odata.count'];
                    } else {
                        $fallback_response = $graph_client->get("https://graph.microsoft.com/v1.0/groups/{$group['id']}/members?\$count=true", [
                            'headers' => $headers
                        ]);
                        $fallback_data = json_decode($fallback_response->getBody(), true);
                        
                        if(isset($fallback_data['@odata.count'])){
                            $member_count = $fallback_data['@odata.count'];
                        } else {
                            if(isset($fallback_data['value'])) {
                                $returned_count = count($fallback_data['value']);
                                if(isset($fallback_data['@odata.nextLink'])) {
                                    $member_count = $returned_count + 1;
                                } else {
                                    $member_count = $returned_count;
                                }
                            } else {
                                $member_count = 0;
                            }
                        }
                    }
                } catch (Exception $e) {
                    log_sso_error("Error getting member count for group {$group['id']}: " . $e->getMessage());
                    $member_count = 0;
                }
                
                // Add size validation and warnings
                $size_validation = validate_group_size($member_count);
                
                $groups[] = [
                    'id' => $group['id'],
                    'displayName' => $group['displayName'],
                    'description' => $group['description'] ?? '',
                    'mail' => $group['mail'] ?? '',
                    'mailEnabled' => $group['mailEnabled'] ?? false,
                    'securityEnabled' => $group['securityEnabled'] ?? false,
                    'groupTypes' => $group['groupTypes'] ?? [],
                    'member_count' => $member_count,
                    'size_validation' => $size_validation,
                    'is_security_group' => $group['securityEnabled'] ?? false,
                    'is_distribution_group' => $group['mailEnabled'] ?? false
                ];
            }
        }
        
        $result = [
            'status' => 'success',
            'data' => [
                'groups' => $groups,
                'total' => count($groups),
                'page' => 1,
                'page_size' => $page_size,
                'group_type' => $group_type,
                'note' => 'Microsoft Graph API does not support $skip parameter. Showing first ' . $page_size . ' results for ' . $group_type . ' groups.'
            ]
        ];
        
        echo json_encode($result);
        exit;
        
    } catch (Exception $e) {
        echo json_encode(['status' => 'failed', 'message' => $e->getMessage()]);
        exit;
    }
}

/**
 * AJAX - Get Azure AD groups by multiple types
 * 
 * @param string 'client_id' -- OAuth client ID
 * @param array 'group_types' -- Array of group types to filter
 * @param int 'page' -- page number for pagination
 * @param int 'page_size' -- number of items per page
 * @param string 'search' -- search term for filtering groups
 * 
 * @echo json array -- [status, data with groups array]
 */
function get_groups_AAD_by_types(){
    check_nagios_session_protector();
    
    $client_id = grab_request_var('client_id', '');
    $group_types_raw = grab_request_var('group_types', []);
    $page = grab_request_var('page', 1);
    $page_size = grab_request_var('page_size', 100);
    $search = grab_request_var('search', '');
    
    // Decode JSON string if it's a string
    if (is_string($group_types_raw)) {
        $group_types = json_decode($group_types_raw, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            echo json_encode(['status' => 'failed', 'message' => _('Invalid group types format')]);
            exit;
        }
    } else {
        $group_types = $group_types_raw;
    }
    
    if(empty($client_id) || empty($group_types)){
        echo json_encode(['status' => 'failed', 'message' => _('Client ID and group types are required')]);
        exit;
    }
    
    try {
        // Get OAuth credentials
        $credentials = get_oauth_credentials($client_id);
        if(!$credentials){
            echo json_encode(['status' => 'failed', 'message' => _('OAuth credentials not found')]);
            exit;
        }
        
        // Create Microsoft Graph API client
        $graph_client = create_graph_client($credentials);
        
        $all_groups = [];
        $group_map = []; // To deduplicate groups
        
        // Get groups for each type
        foreach($group_types as $group_type) {
            $query_params = [
                '$top' => $page_size,
                '$orderby' => 'displayName'
            ];
            
            // Add type-specific filters
            $type_filters = [];
            switch($group_type) {
                case 'Microsoft365':
                    $type_filters[] = "groupTypes/any(c:c eq 'Unified')";
                    break;
                case 'Security':
                    $type_filters[] = "securityEnabled eq true";
                    break;
                case 'Distribution':
                    $type_filters[] = "mailEnabled eq true";
                    break;
                case 'Dynamic':
                    $type_filters[] = "groupTypes/any(c:c eq 'DynamicMembership')";
                    break;
                case 'Cloud':
                    $type_filters[] = "onPremisesSyncEnabled eq null";
                    break;
                case 'OnPremises':
                    $type_filters[] = "onPremisesSyncEnabled eq true";
                    break;
                default:
                    log_sso_error("Invalid group type: $group_type");
                    break; // Skip invalid types
            }
            
            // Combine type filter with search filter
            $all_filters = [];
            if(!empty($type_filters)) {
                $all_filters[] = '(' . implode(' and ', $type_filters) . ')';
            }
            if(!empty($search)){
                $all_filters[] = "(startswith(displayName,'$search') or startswith(description,'$search'))";
            }
            
            if(!empty($all_filters)) {
                $query_params['$filter'] = implode(' and ', $all_filters);
            }
            
            try {
                // Get groups from Microsoft Graph
                $response = $graph_client->get('https://graph.microsoft.com/v1.0/groups?' . http_build_query($query_params));
                $data = $response->getBody();
                $groups_data = json_decode($data, true);
                
                if(isset($groups_data['value'])){
                    foreach($groups_data['value'] as $group){
                        // Skip if we've already seen this group
                        if(isset($group_map[$group['id']])) {
                            continue;
                        }
                        
                        // Get member count for each group
                        $member_count = 0;
                        try {
                            $headers = [
                                'ConsistencyLevel' => 'eventual'
                            ];
                            $members_response = $graph_client->get("https://graph.microsoft.com/v1.0/groups/{$group['id']}/members?\$count=true", [
                                'headers' => $headers
                            ]);
                            $members_data = json_decode($members_response->getBody(), true);
                            
                            if(isset($members_data['@odata.count'])){
                                $member_count = $members_data['@odata.count'];
                            } else {
                                // Fallback estimation
                                if(isset($members_data['value'])) {
                                    $member_count = count($members_data['value']);
                                }
                            }
                        } catch (Exception $e) {
                            $member_count = 0;
                        }
                        
                        // Add size validation and warnings
                        $size_validation = validate_group_size($member_count);
                        
                        $group_data = [
                            'id' => $group['id'],
                            'displayName' => $group['displayName'],
                            'description' => $group['description'] ?? '',
                            'mail' => $group['mail'] ?? '',
                            'mailEnabled' => $group['mailEnabled'] ?? false,
                            'securityEnabled' => $group['securityEnabled'] ?? false,
                            'groupTypes' => $group['groupTypes'] ?? [],
                            'member_count' => $member_count,
                            'size_validation' => $size_validation,
                            'is_security_group' => $group['securityEnabled'] ?? false,
                            'is_distribution_group' => $group['mailEnabled'] ?? false
                        ];
                        
                        $group_map[$group['id']] = $group_data;
                    }
                }
            } catch (Exception $e) {
                log_sso_error("Error getting groups for type $group_type: " . $e->getMessage());
                continue; // Continue with other types
            }
        }
        
        $all_groups = array_values($group_map);
        
        $result = [
            'status' => 'success',
            'data' => [
                'groups' => $all_groups,
                'total' => count($all_groups),
                'page' => 1,
                'page_size' => $page_size,
                'group_types' => $group_types,
                'note' => 'Microsoft Graph API does not support $skip parameter. Showing first ' . $page_size . ' results for selected group types.'
            ]
        ];
        
        echo json_encode($result);
        exit;
        
    } catch (Exception $e) {
        echo json_encode(['status' => 'failed', 'message' => $e->getMessage()]);
        exit;
    }
}

/**
 * AJAX - Get users by group IDs
 * 
 * @param string 'client_id' -- OAuth client ID
 * @param array 'group_ids' -- array of group IDs to get users for
 * 
 * @echo json array -- [status, data with users array]
 */
function get_users_by_groups(){
    check_nagios_session_protector();
    
    $client_id = grab_request_var('client_id', '');
    $group_ids_raw = grab_request_var('group_ids', []);
    $page = grab_request_var('page', 1);
    $page_size = min(grab_request_var('page_size', 50), 100); // Cap at 100 users per request
    
    // Decode JSON string if it's a string
    if (is_string($group_ids_raw)) {
        $group_ids = json_decode($group_ids_raw, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            log_sso_error("JSON decode error for group_ids: " . json_last_error_msg());
            echo json_encode(['status' => 'failed', 'message' => _('Invalid group IDs format')]);
            exit;
        }
    } else {
        $group_ids = $group_ids_raw;
    }
    
    if(empty($client_id) || empty($group_ids)){
        echo json_encode(['status' => 'failed', 'message' => _('Client ID and group IDs are required')]);
        exit;
    }
        
    try {
        // Get OAuth credentials
        $credentials = get_oauth_credentials($client_id);
        if(!$credentials){
            echo json_encode(['status' => 'failed', 'message' => _('OAuth credentials not found')]);
            exit;
        }
        
        // Create Microsoft Graph API client
        $graph_client = create_graph_client($credentials);
        
        $all_users = [];
        
        // Get users for each group with proper Graph API pagination
        $group_next_links = []; // Track nextLink for each group
        $group_page_counts = []; // Track current page for each group
        
        foreach($group_ids as $group_id){
            try {
                // For the first page, get the initial results
                if ($page == 1) {
                    $headers = [
                        'ConsistencyLevel' => 'eventual'
                    ];
                    $response = $graph_client->get("https://graph.microsoft.com/v1.0/groups/$group_id/members?\$select=id,displayName,mail,userPrincipalName,givenName,surname,jobTitle,department,officeLocation&\$top=$page_size", [
                        'headers' => $headers
                    ]);
                } else {
                    // For subsequent pages, we need the nextLink from the previous request
                    // TODO: Implement full pagination by storing nextLink tokens in session/database
                    // and using them for subsequent requests
                    // For now, pagination beyond page 1 is not fully supported
                    log_sso_error("Pagination beyond page 1 requested for group $group_id but not fully implemented. Returning empty results for page $page.");
                    $members_data = ['value' => []];
                    continue;
                }
                
                $data = $response->getBody();
                $members_data = json_decode($data, true);
                
                // Store nextLink for future pagination
                if (isset($members_data['@odata.nextLink'])) {
                    $group_next_links[$group_id] = $members_data['@odata.nextLink'];
                }
                
            } catch (Exception $e) {
                log_sso_error("Error getting members for group $group_id: " . $e->getMessage());
                continue;
            }
            
            if(isset($members_data['value'])){
                foreach($members_data['value'] as $member){
                    // Only include user objects (not groups)
                    if(isset($member['@odata.type']) && $member['@odata.type'] === '#microsoft.graph.user'){
                        $all_users[] = [
                            'user_id' => $member['id'],
                            'name' => $member['displayName'],
                            'email' => $member['mail'],
                            'user_id_aad' => $member['id'],
                            'displayName' => $member['displayName'],
                            'userPrincipalName' => $member['userPrincipalName'],
                            'givenName' => $member['givenName'] ?? '',
                            'surname' => $member['surname'] ?? '',
                            'jobTitle' => $member['jobTitle'] ?? '',
                            'department' => $member['department'] ?? '',
                            'officeLocation' => $member['officeLocation'] ?? '',
                            'groups' => [$group_id] // Track which group this user came from
                        ];
                    }
                }
            } else {
                log_sso_error("No 'value' key in response for group $group_id");
            }
        }
        
        // Remove duplicates based on user_id
        $unique_users = [];
        $seen_ids = [];
        foreach($all_users as $user){
            if(!in_array($user['user_id'], $seen_ids)){
                $unique_users[] = $user;
                $seen_ids[] = $user['user_id'];
            } else {
                // Merge groups for duplicate users
                $existing_index = array_search($user['user_id'], $seen_ids);
                if($existing_index !== false){
                    $unique_users[$existing_index]['groups'] = array_unique(array_merge($unique_users[$existing_index]['groups'], $user['groups']));
                }
            }
        }
        
        // Determine if there are more pages by checking for @odata.nextLink
        $has_more = !empty($group_next_links);
        
        $result = [
            'status' => 'success',
            'data' => [
                'users' => $unique_users,
                'pagination' => [
                    'page' => $page,
                    'page_size' => $page_size,
                    'has_more' => $has_more,
                    'loaded_count' => count($unique_users)
                ]
            ]
        ];
        echo json_encode($result);
        exit;
        
    } catch (Exception $e) {
        echo json_encode(['status' => 'failed', 'message' => $e->getMessage()]);
        exit;
    }
}

/**
 * AJAX - Get administrative units (future feature)
 * 
 * @param string 'client_id' -- OAuth client ID
 * 
 * @echo json array -- [status, data with administrative units array]
 */
function get_administrative_units(){
    check_nagios_session_protector();
    
    $client_id = grab_request_var('client_id', '');
    
    if(empty($client_id)){
        echo json_encode(['status' => 'failed', 'message' => _('Client ID is required')]);
        exit;
    }
    
    // This is a future feature - return empty for now
    $result = [
        'status' => 'success',
        'data' => [
            'administrative_units' => [],
            'total' => 0
        ]
    ];
    
    echo json_encode($result);
    exit;
}

/**
 * AJAX - Get users by administrative unit (future feature)
 * 
 * @param string 'client_id' -- OAuth client ID
 * @param string 'unit_id' -- administrative unit ID
 * 
 * @echo json array -- [status, data with users array]
 */
function get_users_by_administrative_unit(){
    check_nagios_session_protector();
    
    $client_id = grab_request_var('client_id', '');
    $unit_id = grab_request_var('unit_id', '');
    
    if(empty($client_id) || empty($unit_id)){
        echo json_encode(['status' => 'failed', 'message' => _('Client ID and unit ID are required')]);
        exit;
    }
    
    // This is a future feature - return empty for now
    $result = [
        'status' => 'success',
        'data' => [
            'users' => [],
            'total' => 0
        ]
    ];
    
    echo json_encode($result);
    exit;
}

/**
 * Helper function to create Microsoft Graph API client
 * 
 * @param array $credentials -- OAuth credentials
 * @return object -- Graph API client
 */
function create_graph_client($credentials){
    require_once(dirname(__FILE__) . '/../createprovider.php');
    
    $params = [
        'clientId' => $credentials['clientId'],
        'clientSecret' => $credentials['clientSecret'],
    ];
    
    if(isset($credentials['tenantId'])){
        $params['tenantId'] = $credentials['tenantId'];
    }
    
    $provider = getProvider('azure', $params);
    $token = $provider->getAccessToken('client_credentials', ['scope' => 'https://graph.microsoft.com/.default']);
    
    // Create HTTP client with Bearer token
    $http_client = new \GuzzleHttp\Client([
        'headers' => [
            'Authorization' => 'Bearer ' . $token->getToken(),
            'Content-Type' => 'application/json'
        ]
    ]);
    
    return $http_client;
}

/**
 * Get OAuth credentials for the given client_id from the stored tenants file.
 *
 * Returns an array in the form:
 * [
 *   'clientId' => ...,
 *   'clientSecret' => ...,
 *   'tenantId' => ...,
 * ]
 *
 * @param string $client_id
 * @return array|false Returns credentials array if found, false otherwise
 */
function get_oauth_credentials($client_id)
{
    // Read and decrypt tenants credentials file
    $tenants_path = '/usr/local/nagiosxi/etc/components/oauth2/providers/AAD/tenants';
    if (!file_exists($tenants_path)) {
        return false;
    }

    $encrypted = file_get_contents($tenants_path);
    $json = decrypt_data($encrypted);
    if (!$json) {
        return false;
    }

    $tenants = json_decode($json, true);
    if (!is_array($tenants) || !array_key_exists($client_id, $tenants)) {
        return false;
    }

    $tenant = $tenants[$client_id];

    // Migrate or ensure keys for compatibility
    $credentials = [
        'clientId' => !empty($tenant['client_id']) ? $tenant['client_id'] : (isset($tenant['clientId']) ? $tenant['clientId'] : ''),
        'clientSecret' => !empty($tenant['client_secret']) ? $tenant['client_secret'] : (isset($tenant['clientSecret']) ? $tenant['clientSecret'] : ''),
        'tenantId' => !empty($tenant['tenant_id']) ? $tenant['tenant_id'] : (isset($tenant['tenantId']) ? $tenant['tenantId'] : ''),
    ];

    if (empty($credentials['clientId']) || empty($credentials['clientSecret']) || empty($credentials['tenantId'])) {
        return false;
    }

    return $credentials;
}

// function get_user_info_AAD($client_id, $tenant_id, $client_secret)
// {
//     try {
//         $provider = getProvider('azure', $params = array(
//             'tenantId' => $tenant_id,
//             'clientId' => $client_id,
//             'clientSecret' => $client_secret
//         ));

//         $token = $provider->getAccessToken('client_credentials', [
//             'scope' => 'https://graph.microsoft.com/.default'
//         ]);

//         if (strlen($token->getToken()) === 0) {
//             throw new Exception('Failed to authenticate with Azure AD.');
//         }

//         $usersRequest = $provider->getAuthenticatedRequest(
//             'GET',
//             'https://graph.microsoft.com/v1.0/users',
//             $token
//         );

//         $usersResponse = $provider->getResponse($usersRequest);
//         $users = json_decode($usersResponse->getBody()->getContents(), true);

//         if ($users === null || !is_array($users)) {
//             throw new Exception('Failed to retrieve users from Azure AD.');
//         }

//         return $users;
//     } catch (Exception $e) {
//         // echo json_encode(array('status' => 'error', 'message' => 'Failed to get user info from Azure AD.'));
//         log_sso_error('Failed to get user info from Azure AD: ' . $e->getMessage());
//         echo json_encode(array('status' => 'error', 'message' => $e->getMessage()));
//         exit();
//     }
// }

function get_deltas_from_AAD()
{
    /*
     * This function will get the deltas from AAD for individuals, groups, and administrative units.
     * 
     * TODO: This function is not yet implemented. It requires:
     * 1. Storage of delta tokens (deltaLink) from previous syncs
     * 2. Implementation of Microsoft Graph API delta query endpoints
     * 3. Handling of pagination for delta results
     * 4. Proper error handling and token refresh
     * 
     * For now, this function returns an empty array to prevent fatal errors.
     * When implemented, it should use Microsoft Graph API delta queries:
     * - https://graph.microsoft.com/v1.0/users/delta
     * - https://graph.microsoft.com/v1.0/groups/delta
     * 
     * @return array Array of delta changes with structure:
     *   [
     *     ['type' => 'new', 'user' => [...]],
     *     ['type' => 'deleted', 'user' => [...]],
     *     ['type' => 'updated', 'user' => [...]]
     *   ]
     */
    // FIXED: Removed infinite recursion - was calling itself
    // Return empty array until proper implementation
    log_sso_error('get_deltas_from_AAD() called but not yet implemented. Delta synchronization is not available.');
    return [];
}
