<?php
//
// DNS Server Config Wizard
// Copyright (c) 2010-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

dnsquery_configwizard_init();

function dnsquery_configwizard_init()
{
    $name = "dnsquery";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.0.2",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor a host or domain lookup/query via DNS."),
        CONFIGWIZARD_DISPLAYTITLE => "DNS Query",
        CONFIGWIZARD_FUNCTION => "dnsquery_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "community2.png",
        CONFIGWIZARD_FILTER_GROUPS => array('website','network'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );
    register_configwizard($name, $args);
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function dnsquery_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    $wizard_name = "dnsquery";

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:
            $address = grab_array_var($inargs, "domain_name", "");

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts($wizard_name);

            // Remove any existing session variables if this is the first time on the page
            if (isset($_POST['backButton']) == false) {
                unset($_SESSION['dns_query_wizard_address']);
                unset($_SESSION['dns_query_wizard_dnsserver']);
                unset($_SESSION['dns_query_wizard_dnsauthority']);
                unset($_SESSION['dns_query_wizard_address']);
                unset($_SESSION['dns_query_wizard_hostname']);
                unset($_SESSION['dns_query_wizard_ip']);
                $_SESSION['dns_query_wizard_services'] = array('0'=> 'dns', '1'=> 'dnsip');
            }

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "domain_name", "");

            if (array_key_exists('dns_query_wizard_address', $_SESSION) && $address == "") {
                $address = $_SESSION['dns_query_wizard_address'];
            }

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (have_value($address) == false) {
                $errmsg[$errors++] = "No address specified.";
            }

            if ($address) {
                $_SESSION['dns_query_wizard_address'] = $address;
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "domain_name");
            $dnsserver = grab_array_var($inargs, "dnsserver");
            $dnsauthority = grab_array_var($inargs, "dnsauthority");

            $ha = @gethostbyaddr($address);
            if ($ha == "")
                $ha = $address;
            $hostname = grab_array_var($inargs, "hostname", $ha);

            $ip = gethostbyname($hostname);

            $services_serial = grab_array_var($inargs, "services_serial");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial");

            if ($services_serial != "") {
                $services = json_decode(base64_decode($services_serial), true);
            } else {
                $services = grab_array_var($inargs, "services", array(
                    "0" => "dns",
                    "1" => "dnsip",
                ));
            }

            if ($serviceargs_serial != "") {
                $serviceargs = json_decode(base64_decode($serviceargs_serial), true);
            } else {
                $serviceargs = grab_array_var($inargs, "serviceargs", array());
            }

            if (array_key_exists('dns_query_wizard_address', $_SESSION) && $address == "") {
                $address = $_SESSION['dns_query_wizard_address'];
            }

            if (array_key_exists('dns_query_wizard_dnsserver', $_SESSION) && $dnsserver == "") {
                $dnsserver = $_SESSION['dns_query_wizard_dnsserver'];
            }

            if (array_key_exists('dns_query_wizard_dnsauthority', $_SESSION) && $dnsauthority == "") {
                $dnsauthority = $_SESSION['dns_query_wizard_dnsauthority'];
            }

            if (array_key_exists('dns_query_wizard_hostname', $_SESSION) && $hostname == "") {
                $hostname = $_SESSION['dns_query_wizard_hostname'];
            }

            if (array_key_exists('dns_query_wizard_ip', $_SESSION) && $ip == "") {
                $ip = $_SESSION['dns_query_wizard_ip'];
            }

            if (array_key_exists('dns_query_wizard_services', $_SESSION) && $services == "") {
                $services = $_SESSION['dns_query_wizard_services'];
            }

            if ($address) {
                $_SESSION['dns_query_wizard_address'] = $address;
            }

            if ($dnsserver) {
                $_SESSION['dns_query_wizard_dnsserver'] = $dnsserver;
            }

            if ($dnsauthority) {
                $_SESSION['dns_query_wizard_dnsauthority'] = $dnsauthority;
            }

            if ($hostname) {
                $_SESSION['dns_query_wizard_hostname'] = $hostname;
            }

            if ($ip) {
                $_SESSION['dns_query_wizard_ip'] = $ip;
            }

            if ($services) {
                $_SESSION['dns_query_wizard_services'] = $services;
            }

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "domain_name");
            $ip = grab_array_var($inargs, "ip");
            $hostname = grab_array_var($inargs, "hostname");

            $dnsserver = grab_array_var($inargs, "dnsserver");
            $dnsauthority = grab_array_var($inargs, "dnsauthority");

            $services = grab_array_var($inargs, "services", array());
            $serviceargs = grab_array_var($inargs, "serviceargs", array());

            if (isset($_POST['submitButton2']) && $dnsserver == "") {
                unset($_SESSION['dns_query_wizard_dnsserver']);
            }

            if (isset($_POST['submitButton2']) && $dnsauthority == "") {
                unset($_SESSION['dns_query_wizard_dnsauthority']);
            }

            if (isset($_POST['submitButton2']) && $dnsauthority == "") {
                unset($_SESSION['dns_query_wizard_dnsauthority']);
            }

            if (isset($_POST['submitButton2']) && $hostname == "") {
                unset($_SESSION['dns_query_wizard_hostname']);
            }

            if (isset($_POST['submitButton2']) && empty($services)) {
                unset($_SESSION['dns_query_wizard_services']);
            }

            if (array_key_exists('dns_query_wizard_address', $_SESSION) && $address == "") {
                $address = $_SESSION['dns_query_wizard_address'];
            }

            if (array_key_exists('dns_query_wizard_dnsserver', $_SESSION) && $dnsserver == "") {
                $dnsserver = $_SESSION['dns_query_wizard_dnsserver'];
            }

            if (array_key_exists('dns_query_wizard_dnsauthority', $_SESSION) && $dnsauthority == "") {
                $dnsauthority = $_SESSION['dns_query_wizard_dnsauthority'];
            }

            if (array_key_exists('dns_query_wizard_hostname', $_SESSION) && $hostname == "") {
                $hostname = $_SESSION['dns_query_wizard_hostname'];
            }

            if (array_key_exists('dns_query_wizard_ip', $_SESSION) && $ip == "") {
                $ip = $_SESSION['dns_query_wizard_ip'];
            }

            if (array_key_exists('dns_query_wizard_services', $_SESSION) && !$services) {
                $services = $_SESSION['dns_query_wizard_services'];
            }

            // check for errors
            $errors = 0;
            $errmsg = array();

            if (is_valid_host_name($hostname) == false) {
                $errmsg[$errors++] = _("Invalid host name.");
            }

            if ($dnsauthority && $dnsserver == "") {
                $errmsg[$errors++] = _("Enter a DNS Server to use for the query.");
            }

            /*
            if (array_key_exists("transfer",$services)){
                if ($username=="" || $password=="")
                    $errmsg[$errors++]="Username or password is blank.";
                if ($port=="")
                    $errmsg[$errors++]="Invalid port number.";
                }
            */

            if ($address) {
                $_SESSION['dns_query_wizard_address'] = $address;
            }

            if ($dnsserver) {
                $_SESSION['dns_query_wizard_dnsserver'] = $dnsserver;
            }

            if ($dnsauthority) {
                $_SESSION['dns_query_wizard_dnsauthority'] = $dnsauthority;
            }

            if ($hostname) {
                $_SESSION['dns_query_wizard_hostname'] = $hostname;
            }

            if ($ip) {
                $_SESSION['dns_query_wizard_ip'] = $ip;
            }

            if ($services) {
                $_SESSION['dns_query_wizard_services'] = $services;
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "domain_name");
            $ip = grab_array_var($inargs, "ip");
            $hostname = grab_array_var($inargs, "hostname");

            $dnsserver = grab_array_var($inargs, "dnsserver");
            $dnsauthority = grab_array_var($inargs, "dnsauthority");

            $services = grab_array_var($inargs, "services");
            $serviceargs = grab_array_var($inargs, "serviceargs");

            $output = '

        <input type="hidden" name="domain_name" value="' . encode_form_val($address) . '">
        <input type="hidden" name="hostname" value="' . encode_form_val($hostname) . '">
        <input type="hidden" name="ip" value="' . encode_form_val($ip) . '">
        <input type="hidden" name="dnsserver" value="' . encode_form_val($dnsserver) . '">
        <input type="hidden" name="dnsauthority" value="' . encode_form_val($dnsauthority) . '">
        <input type="hidden" name="services_serial" value="' . base64_encode(json_encode($services)) . '">
        <input type="hidden" name="serviceargs_serial" value="' . base64_encode(json_encode($serviceargs)) . '">
';
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:


            $output = '
            ';
            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            $hostname = grab_array_var($inargs, "hostname", "");
            $ip = grab_array_var($inargs, "ip", "");
            $address = grab_array_var($inargs, "domain_name", "");

            // this is different than normal
            $hostaddress = $ip;

            $dnsserver = grab_array_var($inargs, "dnsserver");
            $dnsauthority = grab_array_var($inargs, "dnsauthority");

            $services_serial = grab_array_var($inargs, "services_serial", "");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            $services = json_decode(base64_decode($services_serial), true);
            $serviceargs = json_decode(base64_decode($serviceargs_serial), true);

            if (array_key_exists('dns_query_wizard_address', $_SESSION) && $address == "") {
                $address = $_SESSION['dns_query_wizard_address'];
            }

            if (array_key_exists('dns_query_wizard_dnsserver', $_SESSION) && $dnsserver == "") {
                $dnsserver = $_SESSION['dns_query_wizard_dnsserver'];
            }

            if (array_key_exists('dns_query_wizard_dnsauthority', $_SESSION) && $dnsauthority == "") {
                $dnsauthority = $_SESSION['dns_query_wizard_dnsauthority'];
            }

            if (array_key_exists('dns_query_wizard_hostname', $_SESSION) && $hostname == "") {
                $hostname = $_SESSION['dns_query_wizard_hostname'];
            }

            if (array_key_exists('dns_query_wizard_ip', $_SESSION) && $ip == "") {
                $ip = $_SESSION['dns_query_wizard_ip'];
            }

            if (array_key_exists('dns_query_wizard_services', $_SESSION) && $services == "") {
                $services = $_SESSION['dns_query_wizard_services'];
            }

            /*
            echo "SERVICES<BR>";
            print_r($services);
            echo "<BR>";
            echo "SERVICEARGS<BR>";
            print_r($serviceargs);
            echo "<BR>";
            */

            // save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["hostname"] = $hostname;
            $meta_arr["domain_name"] = $address;
            $meta_arr["ip"] = $ip;
            $meta_arr["dnsserver"] = $dnsserver;
            $meta_arr["dnsauthority"] = $dnsauthority;
            $meta_arr["services"] = $services;
            $meta_arr["serivceargs"] = $serviceargs;
            save_configwizard_object_meta($wizard_name, $hostname, "", $meta_arr);

            $objs = array();

            if (!host_exists($hostname)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_dnsquery_host",
                    "host_name" => $hostname,
                    "address" => $hostaddress,
                    "icon_image" => "server.png",
                    "statusmap_image" => "server.png",
                    "_xiwizard" => $wizard_name,
                );
            }

            $pluginopts = "";
            if ($dnsserver != "") {
                $pluginopts .= " -s " . escapeshellarg($dnsserver);
                if ($dnsauthority != "")
                    $pluginopts .= " -A";
            }

            // see which services we should monitor
            foreach ($services as $service) {

                //echo "PROCESSING: $svc -> $svcstate<BR>\n";

                switch ($service) {

                    case "dns":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "DNS Resolution - " . $address,
                            "use" => "xiwizard_dnsquery_service",
                            "check_command" => "check_xi_service_dnsquery!-H " . escapeshellarg($address) . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "dnsip":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "DNS IP Match - " . $address,
                            "use" => "xiwizard_dnsquery_service",
                            "check_command" => "check_xi_service_dnsquery!-H " . escapeshellarg($address) . " -a " . escapeshellarg($ip) . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    default:
                        break;
                }
            }

            //echo "OBJECTS:<BR>";
            //print_r($objs);
            //exit();

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:
            break;
    }

    return $output;
}


?>
