<?php
//
// Metrics Component
// Copyright (c) 2010-2024 Nagios Enterprises, LLC. All rights reserved.
//

require_once(dirname(__FILE__) . '/../componenthelper.inc.php');
include_once(dirname(__FILE__) . '/dashlet.inc.php');

$metrics_component_name = "metrics";
metrics_component_init();

////////////////////////////////////////////////////////////////////////
// COMPONENT INIT FUNCTIONS
////////////////////////////////////////////////////////////////////////

function metrics_component_init()
{
    global $metrics_component_name;
    $versionok = metrics_component_checkversion();

    $desc = "";
    if (!$versionok) {
        $desc = "<b>" . _("Error: This component requires Nagios XI 2011R1 or later.") . "</b>";
    }

    $args = array(
        COMPONENT_NAME => $metrics_component_name,
        COMPONENT_AUTHOR => "Nagios Enterprises, LLC",
        COMPONENT_DESCRIPTION => _("Displays metrics such as disk, CPU, and memory usage. ") . $desc,
        COMPONENT_TITLE => _("Metrics"),
        COMPONENT_VERSION => '2.0.1'
    );

    register_component($metrics_component_name, $args);

    if ($versionok) {
        register_callback(CALLBACK_MENUS_INITIALIZED, 'metrics_component_addmenu');
    }

    $args = array();
    $args[DASHLET_NAME] = "metrics";
    $args[DASHLET_TITLE] = "Metrics Overview";
    $args[DASHLET_FUNCTION] = "metrics_dashlet_func";
    $args[DASHLET_DESCRIPTION] = _("Displays an overview metrics such as disk, CPU, and memory usage.");
    $args[DASHLET_FILTER_GROUPS] = array("metrics");
    $args[DASHLET_WIDTH] = "1000";
    $args[DASHLET_HEIGHT] = "200";
    $args[DASHLET_INBOARD_CLASS] = "metrics_map_inboard";
    $args[DASHLET_OUTBOARD_CLASS] = "metrics_map_outboard";
    $args[DASHLET_AUTHOR] = "Nagios Enterprises, LLC";
    $args[DASHLET_COPYRIGHT] = "Dashlet Copyright &copy; 2011 Nagios Enterprises. All rights reserved.";
    $args[DASHLET_HOMEPAGE] = "http://www.nagios.com";
    $args[DASHLET_SHOWASAVAILABLE] = true;
    $args[DASHLET_ISCUSTOM] = false;
    register_dashlet($args[DASHLET_NAME], $args);

    $args = array();
    $args[DASHLET_NAME] = "metricsguage";
    $args[DASHLET_TITLE] = "Metrics Guage";
    $args[DASHLET_FUNCTION] = "metricsguage_dashlet_func";
    $args[DASHLET_DESCRIPTION] = _("Displays service metric such as disk, CPU, and memory usage.");
    $args[DASHLET_WIDTH] = "200";
    $args[DASHLET_HEIGHT] = "160";
    $args[DASHLET_INBOARD_CLASS] = "metrics_map_inboard";
    $args[DASHLET_OUTBOARD_CLASS] = "metrics_map_outboard";
    $args[DASHLET_AUTHOR] = "Nagios Enterprises, LLC";
    $args[DASHLET_COPYRIGHT] = "Dashlet Copyright &copy; 2011 Nagios Enterprises. All rights reserved.";
    $args[DASHLET_HOMEPAGE] = "http://www.nagios.com";
    $args[DASHLET_SHOWASAVAILABLE] = false;
    $args[DASHLET_ISCUSTOM] = false;
    register_dashlet($args[DASHLET_NAME], $args);
}


///////////////////////////////////////////////////////////////////////////////////////////
// MISC FUNCTIONS
///////////////////////////////////////////////////////////////////////////////////////////

function metrics_component_checkversion()
{
    if (!function_exists('get_product_release'))
        return false;
    if (get_product_release() < 530)
        return false;
    return true;
}

function metrics_component_addmenu($arg = null)
{
    global $metrics_component_name;

    $mi = find_menu_item(MENU_HOME, "menu-home-sectionend-details", "id");
    if ($mi == null)
        return;

    $order = grab_array_var($mi, "order", "");
    if ($order == "")
        return;

    $neworder = $order - 0.1;

    add_menu_item(MENU_HOME, array(
        "type" => "linkspacer",
        "order" => $neworder,
    ));

    add_menu_item(MENU_HOME, array(
        "type" => "link",
        "title" => _("Metrics"),
        "id" => "menu-home-metrics",
        "order" => $neworder + 0.001,
        "opts" => array(
            "href" => get_base_url() . 'includes/components/metrics/',
            "icon" => "fa-tachometer"
        )
    ));
}


///////////////////////////////////////////////////////////////////////////////////////////
// AJAX FUNCTIONS
///////////////////////////////////////////////////////////////////////////////////////////

function xicore_ajax_get_metricsguage_dashlet_html($args = null)
{
    $output = '';

    $host = grab_array_var($args, "host", "");
    $service = grab_array_var($args, "service", "");
    $metric = grab_array_var($args, "metric", "");

    $percent = grab_array_var($args, "percent", "");
    $current = grab_array_var($args, "current", "");
    $uom = grab_array_var($args, "uom", "");
    $warn = grab_array_var($args, "warn", "");
    $crit = grab_array_var($args, "crit", "");
    $min = grab_array_var($args, "min", "");
    $max = grab_array_var($args, "max", "");
    $command = grab_array_var($args, "command");

    $plugin_output = grab_array_var($args, "plugin_output", "");

    $sortorder = grab_array_var($args, "sortorder", "desc");

    $mode = grab_array_var($args, "mode", DASHLET_MODE_INBOARD);

    if ($mode == DASHLET_MODE_INBOARD) {
        $args = array(
            "host" => $host,
            "service" => $service,
            "metric" => $metric,
        );
        $metricdata = get_service_metrics($args);

        foreach ($metricdata as $id => $arr) {
            $current = grab_array_var($arr, "current");
            $warn = grab_array_var($arr, "warn");
            $crit = grab_array_var($arr, "crit");
            $min = grab_array_var($arr, "min");
            $max = grab_array_var($arr, "max");
            $percent = grab_array_var($arr, "sortval");
            $command = grab_array_var($arr, "command");
        }
    }

    $base_url = get_base_url();

    $display_val = $percent . "%";

    $imgurlb = "";

    // adjustments for load
    if ($metric == "load") {
        $max = $crit * 1.2;
        $percent = ($current / ($max - $min)) * 100;
        $display_val = $current;
    }

    $warn_width = 0;
    $crit_width = 0;
    $min = $min ? $min : 0;
    if ($crit > 0 && $max > 0) {
        $crit_width = ($crit / ($max - $min)) * 100;
    }
    if ($warn > 0 && $max > 0) {
        $warn_width = ($warn / ($max - $min)) * 100;
    }

    $width = intval($percent);
    $color = get_graph_color($command, $warn_width, $crit_width, $width, $max, $plugin_output);

    $status_url = $base_url . "includes/components/xicore/status.php?show=servicedetail&host=" . urlencode($host) . "&service=" . urlencode($service);
    $title = $plugin_output;


    //metrics_component_get_display_params($metric,intval($percent),$width,$color);
    $output .= '<div class="metric_gauge_box">';

    $output .= '<div class="ptext h-full w-full" style="background-color: ' . $color . '; width: ' . $width . 'px;">';
    $output .= '<b style="position: relative; z-index: 2;">' . $display_val . '</b>';
    $output .= '</div>';

    if ($crit_width > 0)
        $output .= '<div class="vertbar h-full" style="border-right: 1px solid red; width: ' . $crit_width . 'px;"></div>';
    if ($warn_width > 0)
        $output .= '<div class="h-full vertbar" style="border-right: 1px solid #e5e500; width: ' . $warn_width . 'px;"></div>';

    $output .= '</div>';
    
    $output .= '<a href="' . $status_url . '" title="' . $title . '"><b>' . $host . '</b><br><b>' . $service . '</b></a><br>';

    return $output;
}

function xicore_ajax_get_metrics_dashlet_html($args = null)
{
    $host = grab_array_var($args, "host", "");
    $hostgroup = grab_array_var($args, "hostgroup", "");
    $servicegroup = grab_array_var($args, "servicegroup", "");
    $maxitems = grab_array_var($args, "maxitems", 20);
    $metric = grab_array_var($args, "metric", "disk");
    $showperfgraphs = grab_array_var($args, "showperfgraphs", true);
    $details = grab_array_var($args, "details");

    // special "all" stuff
    if ($hostgroup == "all")
        $hostgroup = "";
    if ($servicegroup == "all")
        $servicegroup = "";
    if ($host == "all")
        $host = "";

    // can do hostgroup OR servicegroup OR host
    if ($hostgroup != "") {
        $servicegroup = "";
        $host = "";
    } else if ($servicegroup != "") {
        $host = "";
    }

    // get service metrics
    $metricdata = get_service_metrics($args);

    $output = "";

    $pretitle = "";
    if ($host != "")
        $pretitle = "For Host '$host' ";
    else if ($hostgroup != "")
        $pretitle = "For Hostgroup '$hostgroup' ";
    else if ($servicegroup != "")
        $pretitle = "For Servicegroup '$servicegroup' ";

    $output .= '
    <h5>' . get_metric_description($metric) . ' ' . $pretitle . '</h5>';

    // $tableclass = "table table-condensed table-auto-width table-striped table-bordered";
    $tableclass = "table table-striped table-bordered";
    $tablestyles = "table-layout: fixed; width: 1000px;";
    $tdstyles = "
        width: 20%; 
        overflow: hidden; 
        text-overflow: ellipsis; 
        white-space: normal;
        
        -webkit-box-orient: vertical; 
        -webkit-line-clamp: 3;
    ";
    $finaltdstyles = "
        width: 60%;
        overflow: hidden;
        text-overflow: ellipsis; 
        white-space: normal;
        
        -webkit-box-orient: vertical; 
        -webkit-line-clamp: 3;
    ";

    $output .= '<table class="' . $tableclass . '" style="' . $tablestyles . '">
        <thead>
            <tr>
                <th style="' . $tdstyles . '">' . _('Host') . '</th>
                <th style="' . $tdstyles . '">' . _('Service') . '</th>
                <th style="' . $tdstyles . '">' . get_metric_value_description($metric) . '</th>';

    // hide details column if requested
    if (!$details) {
        $output .= '
                <th style="' . $finaltdstyles . '">
                    ' . _('Details') . '
                </th>
            </tr>';
    } else {
        $output .= '
            </tr>';
    }

    $output .= '
        </thead>

        <tbody>';

    $base_url = get_base_url();
    $status_url = $base_url . "includes/components/xicore/status.php";

    $current_item = 0;
    $data_count = count($metricdata);
    if ($data_count > 0) {

        foreach ($metricdata as $id => $arr) {
            $current_item++;

            if ($current_item > $maxitems)
                break;

            $current = grab_array_var($arr, "current");
            $warn = grab_array_var($arr, "warn");
            $crit = grab_array_var($arr, "crit");
            $max = grab_array_var($arr, "max");
            $command = grab_array_var($arr, "command");
            $plugin_output = grab_array_var($arr, "output");

            $hostname = grab_array_var($arr, "host_name");
            $servicename = grab_array_var($arr, "service_name");

            $output .= '<tr>';
            $output .= '
                    <td style="'. $tdstyles . '">
                        <a href="' . $status_url . '?show=hostdetail&host=' . urlencode($hostname) . '">' . $hostname . '</a>
                    </td>';
            $output .= '
                    <td style="'. $tdstyles . '">
                        <a href="' . $status_url . '?show=servicedetail&host=' . urlencode($hostname) . '&service=' . urlencode($servicename) . '">' . $servicename . '</a>
                    </td>';
            $output .= '<td style="'. $tdstyles . '">';
            $output .= '<span style="float: left; margin-right: 10px;">' . $arr["displayval"] . '</span>';

            metrics_component_get_display_params($metric, intval($arr["sortval"]), $current, $warn, $crit, $width, $color, $command, $max, $plugin_output);

            $output .= '<div style="width: 157px;">';

            $output .= '<div class="statbar-neptune" style="background-color: ' . $color . '; width: ' . $width . 'px; height: 15px; margin-left: 58px;"></div>';
            $output .= '</div></td>';

            if (!$details) {
                // $output .= '<td style="' . $tdstyles . ' white-space: normal; display: -webkit-box; -webkit-box-orient: vertical; -webkit-line-clamp: 3; overflow: hidden;">';
                $output .= '<td style="' . $finaltdstyles . '">';
                $output .= $arr["output"];
                $output .= '</td>';
            }

            $output .= '</tr>';
        }
    } else {
        $output .= '<tr>
                        <td style="' . $tdstyles . '" colspan="4">
                            ' . _('No matching data to display.') . '
                        </td>
                    </tr>';
    }

    $output .= '
        </tbody>
    </table>';

    $output .= '
        <div class="ajax_date mt-1" style="padding-bottom: 42px;">
            ' . _('Last Updated') . ': ' . get_datetime_string(time()) . '
        </div>';

    return $output;
}


function metrics_component_get_display_params($metric, $sortval, $current, $warn, $crit, &$width, &$color, $command, $max, $plugin_output)
{
    $width = $sortval;
    $color = "green";

    $use_global_thresholds = false;
    if ($use_global_thresholds == true) {
        switch ($metric) {

            case "disk":
                // some defaults (should be configurable later)
                if ($sortval > 90)
                    $color = "red";
                else if ($sortval > 75)
                    $color = "orange";
                else if ($sortval > 60)
                    $color = "yellow";
                break;

            case "cpu":

                // some defaults (should be configurable later)
                if ($sortval > 95)
                    $color = "red";
                else if ($sortval > 85)
                    $color = "orange";
                else if ($sortval > 70)
                    $color = "yellow";
                break;

            case "memory":

                // some defaults (should be configurable later)
                if ($sortval > 95)
                    $color = "red";
                else if ($sortval > 90)
                    $color = "orange";
                else if ($sortval > 85)
                    $color = "yellow";
                break;

            // special case for load
            case "load":
                if ($width > 100)
                    $width = 100;
                break;

            case "swap":
                // some defaults (should be configurable later)
                if ($sortval > 95)
                    $color = "red";
                else if ($sortval > 85)
                    $color = "orange";
                else if ($sortval > 70)
                    $color = "yellow";
                break;

            default:
                break;
        }
    } else {

        if ($width <= 0) {
            $width = 1;
        }
        if ($width > 100) {
            $width = 100;
        }

        $color = get_graph_color($command, $warn, $crit, $current, $max, $plugin_output);

    }
}

function get_graph_color($command, $warn, $crit, $current, $max=100, $plugin_output=0) {

    $theme = get_theme();
    $ok_color = ($theme == 'colorblind') ? '#56B4E9' : '#00EE5F';
    $warn_color = ($theme == 'colorblind') ? '#D55E00' : '#ECFD20';
    $crit_color = ($theme == 'colorblind') ? '#CC79A7' : '#E24646';
    $colors = array("ok" => "{$ok_color}", "warn" => "{$warn_color}", "crit" => "{$crit_color}");
    $colorstate = "ok";

    // If we have plugin output, try to get color state from that
    // Potentially some commands' output won't work. Add those exceptions to this array to force actual calculations
     $cmd_requires_calc = array("add commands with ambiguous output here");

    if ($plugin_output && ! in_array($command,$cmd_requires_calc)) {
        $pats = array('crit' =>'/crit/i', 'warn' => '/warn/i', 'ok' => '/ok/i');

        foreach ($pats as $k => $pat) {
            $is_match = preg_match($pat, $plugin_output);

            if (preg_match($pat, $plugin_output)) {
                $colorstate = $k;
                return $colors[$colorstate];
            }
        }
    }

    // We couldn't determine the color state from the output, so do normal calculations

    // Come on, do it backwards if we know high values are the good values...
    // Only if the critical value is lower than warning (meaning good values are high)
    if ($crit < $warn && $command != 'check_xi_ncpa') {

        // Free space metric shows "% utilization", e.g. 90%, but color is based on free space, e.g. 10%.
        // So, set current to free space for comparison with thresholds
        $current = $command == 'check_local_disk' ? $max-$current : $current;

        if (!empty($crit) && floatval($current) < floatval($crit)) {
            $colorstate = "crit";
        } else if (!empty($warn) && floatval($current) < floatval($warn)) {
            $colorstate = "warn";
        }
    // Standard check here, if value higher than warn/crit
    } else {
        if (!empty($crit) && floatval($current) > floatval($crit)) {
            $colorstate = "crit";
        } else if (!empty($warn) && floatval($current) > floatval($warn)) {
            $colorstate = "warn";
        }
    }

    $color = $colors[$colorstate];

    return $color;
}
