<?php
//
// Domain Expiration Config Wizard
// Copyright (c) 2013-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

/**
 * @param $domain_name
 *
 * @return bool
 */
function is_valid_domain_name($domain_name)
{
    return (preg_match("/^([a-z\d](-*[a-z\d])*)(\.([a-z\d](-*[a-z\d])*))*$/i", $domain_name) //valid chars check
        && preg_match("/^.{1,253}$/", $domain_name) //overall length check
        && preg_match("/^[^\.]{1,63}(\.[^\.]{1,63})*$/", $domain_name)); //length of each label
}

domain_expiration_configwizard_init();

function domain_expiration_configwizard_init()
{
    $name = "domain_expiration";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.0.1",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor a domain expiration"),
        CONFIGWIZARD_DISPLAYTITLE => _("Domain Expiration"),
        CONFIGWIZARD_FUNCTION => "domain_expiration_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "domain_expiration.png",
        CONFIGWIZARD_FILTER_GROUPS => array('website','network'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );
    register_configwizard($name, $args);
}

/**
 * @return array
 */
function domain_expiration_configwizard_check_prereqs()
{
    $errors = array();

    if (!file_exists("/usr/local/nagios/libexec/check_domain.php")) {
        $errors[] = _('It looks like you are missing check_domain.php on your Nagios XI server. To use this wizard you must install domain expiration on your server.');
    }

    exec("which whois 2>&1", $output, $return_var);

    if ($return_var != 0) {
        $errors[] = _('It looks like you are missing the whois command on your Nagios XI server.') . '<br><br> Run: &nbsp; <b>yum install jwhois -y</b> or <b>yum install whois</b> &nbsp; as root user on your Nagios XI server.';
    }

    return $errors;
}


/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function domain_expiration_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    $wizard_name = "domain_expiration";

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            $errors = domain_expiration_configwizard_check_prereqs();

            if ($errors) {
                $output .= '<div class="message"><ul class="errorMessage">';

                foreach ($errors as $error) {
                    $output .= "<li><p>$error</p></li>";
                }

                $output .= '</ul></div>';

            } else {
                $address = grab_array_var($inargs, "domain_name", "");

                // Save data from clicking "back" in stage 2
                $services = grab_array_var($inargs, "services", array());
                $serviceargs = grab_array_var($inargs, "serviceargs", array());

                # Get the existing host/node configurations.
                # TODO: Include passwords/secrets?
                $nodes = get_configwizard_hosts($wizard_name);

                ########################################################################################
                # Load the html
                # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
                #   to load the PHP from the Step1 file into the $output string.
                ########################################################################################
                ob_start();
                include __DIR__.'/steps/step1.php';
                $output = ob_get_clean();
            }

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "domain_name", "");
            $address_replaced = nagiosccm_replace_user_macros($address);

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (have_value($address_replaced) == false) {
                $errmsg[$errors++] = _("No address specified.");
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "domain_name");
            $warning = grab_array_var($inargs, "warning");
            $critical = grab_array_var($inargs, "critical");

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "domain_name");
            $address_replaced = nagiosccm_replace_user_macros($address);
            $warning = grab_array_var($inargs, "warning");
            $critical = grab_array_var($inargs, "critical");

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (is_valid_domain_name($address_replaced) == false)
                $errmsg[$errors++] = "Invalid domain name.";

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE3OPTS:
            $output .= '<div style="margin-bottom: 20px;">' . _('The selected domain will be checked once per day. Click next to continue.') . '</div>';
            $result = CONFIGWIZARD_HIDE_OPTIONS;
            break;

        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "domain_name");
            $warning = grab_array_var($inargs, "warning");
            $critical = grab_array_var($inargs, "critical");

            $output = '

                <input type="hidden" name="domain_name" value="' . encode_form_val($address) . '" />
                <input type="hidden" name="warning" value="' . encode_form_val($warning) . '" />
                <input type="hidden" name="critical" value="' . encode_form_val($critical) . '" />

            ';
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            $address = grab_array_var($inargs, "domain_name");
            $warning = grab_array_var($inargs, "warning");
            $critical = grab_array_var($inargs, "critical");

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:

            $address = grab_array_var($inargs, "domain_name");
            $warning = grab_array_var($inargs, "warning");
            $critical = grab_array_var($inargs, "critical");

            $output = '

                <input type="hidden" name="domain_name" value="' . encode_form_val($address) . '" />
                <input type="hidden" name="warning" value="' . encode_form_val($warning) . '" />
                <input type="hidden" name="critical" value="' . encode_form_val($critical) . '" />

            ';
            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            $address = grab_array_var($inargs, "domain_name", "");
            $warning = grab_array_var($inargs, "warning");
            $critical = grab_array_var($inargs, "critical");

            // Save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["domain_name"] = $address;
            $meta_arr["warning"] = $warning;
            $meta_arr["critical"] = $critical;
            save_configwizard_object_meta($wizard_name, $address, "", $meta_arr);

            $objs = array();

            if (!host_exists($address)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_domain_expiration_host_v2",
                    "host_name" => $address,
                    "address" => $address,
                    "icon_image" => "domain_expiration.png",
                    "statusmap_image" => "domain_expiration.png",
                    "_xiwizard" => $wizard_name,
                );
            }

            $objs[] = array(
                "type" => OBJECTTYPE_SERVICE,
                "host_name" => $address,
                "service_description" => "Domain Expiration",
                "use" => "xiwizard_domain_expiration_service_v2",
                "check_command" => "check_xi_domain_v2!" . $address . "!-w " . $warning . "!-c " . $critical,
                "check_interval" => 1440,
                "_xiwizard" => $wizard_name,
            );

            // Return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:
            break;
    }

    return $output;
}

?>
