<?php
//
// Nagios XI Server Config Wizard
// Copyright (c) 2010-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

nagiosxiserver_configwizard_init();

function nagiosxiserver_configwizard_init()
{
    $name = "nagiosxiserver";
    $args = array(
        CONFIGWIZARD_NAME =>            $name,
        CONFIGWIZARD_TYPE =>            CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION =>     _("Monitor a remote Nagios XI server."),
        CONFIGWIZARD_DISPLAYTITLE =>    _("Nagios XI Server"),
        CONFIGWIZARD_FUNCTION =>        "nagiosxiserver_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE =>    "nagiosxiserver.png",
        CONFIGWIZARD_VERSION =>         "2.0.1",
        CONFIGWIZARD_FILTER_GROUPS =>   array('nagios', 'server'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );
    register_configwizard($name, $args);
}

/**
 * @param $url
 * @param $username
 * @param $password
 *
 * @return string
 */
function nagiosxiserver_configwizard_get_ticket($url, $username, $password)
{
    $ticket = "";

    // Form the url to get the user's ticket
    $xiurl = $url;
    $xiurl .= "/backend/";
    $xiurl .= "?cmd=getticket&username=" . $username . "&password=" . md5($password);

    // Args to control how we get data
    $args = array(
        'method' => 'get',
        'return_info' => false,
        'disable_verifypeer' => true,
    );

    $data = load_url($xiurl, $args);

    // Turn the data into XML
    if ($data) {

        if (strpos($data, "Authentication Failure") !== false) {
            return $ticket;
        }

        $xml = @simplexml_load_string($data);

        if ($xml) {
            $ticket = strval($xml);
        }
    }

    return $ticket;
}


/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function nagiosxiserver_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    $wizard_name = "nagiosxiserver";

    // Initialize return code and output
    $result = 0;
    $output = "";

    $ticket = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            $address = grab_array_var($inargs, "ip_address", "");
            $url = grab_array_var($inargs, "url", "");
            $apikey = grab_array_var($inargs, "apikey", "");
            $old_auth = grab_array_var($inargs, "old_auth", 0);
            $username = grab_array_var($inargs, "username", "nagiosadmin");
            $password = grab_array_var($inargs, "password", "");

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address", "");
            $url = grab_array_var($inargs, "url", "");
            $apikey = grab_array_var($inargs, "apikey", "");
            $old_auth = grab_array_var($inargs, "old_auth", 0);
            $username = grab_array_var($inargs, "username", "nagiosadmin");
            $password = grab_array_var($inargs, "password", "");

            $address = nagiosccm_replace_user_macros($address);
            $url = nagiosccm_replace_user_macros($url);
            $apikey = nagiosccm_replace_user_macros($apikey);
            $username = nagiosccm_replace_user_macros($username);
            $password = nagiosccm_replace_user_macros($password);

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (have_value($address) == false)
                $errmsg[$errors++] = _("No address specified.");
            if (have_value($url) == false)
                $errmsg[$errors++] = _("No URL specified.");


            // Check for valid username/password
            if (empty($apikey)) {
                if (have_value($username) == false)
                    $errmsg[$errors++] = _("No username specified.");
                if (have_value($password) == false)
                    $errmsg[$errors++] = _("No password specified.");
            }

            // Check authentication, version
            if ($errors == 0) {
                // Get ticket...
                if ($old_auth) {
                    $ticket = nagiosxiserver_configwizard_get_ticket($url, $username, $password);

                    // See if we have a ticket
                    if (empty($ticket)) {
                        $errmsg[$errors++] = _("Unable to authenticate to remote Nagios XI server - Check your credentials and the remote XI server version.");
                    }
                }
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $url = grab_array_var($inargs, "url", "");
            $apikey = grab_array_var($inargs, "apikey", "");
            $old_auth = grab_array_var($inargs, "old_auth", 0);
            $username = grab_array_var($inargs, "username", "nagiosadmin");
            $password = grab_array_var($inargs, "password", "");

            $address_replaced = nagiosccm_replace_user_macros($address);
            $url_replaced = nagiosccm_replace_user_macros($url);
            $apikey_replaced = nagiosccm_replace_user_macros($apikey);
            $username_replaced = nagiosccm_replace_user_macros($username);
            $password_replaced = nagiosccm_replace_user_macros($password);

            $ha = @gethostbyaddr($address);

            if ($ha == "") {
                $ha = $address;
            }

            $hostname = grab_array_var($inargs, "hostname", $ha);
            $hostname_replaced = nagiosccm_replace_user_macros($hostname);

            $services = grab_array_var($inargs, "services", array(
                "ping" => "on",
                "webui" => "on",
                "daemons" => "on",
                "jobs" => "on",
                "datatransfer" => "on",
                "load" => "on",
                "iowait" => "on",
            ));

            $serviceargs = grab_array_var($inargs, "serviceargs", array(
                "load_warning" => "5,4,4",
                "load_critical" => "10,10,7",
                "iowait_warning" => "5",
                "iowait_critical" => "15",
            ));

            $services_serial = grab_array_var($inargs, "services_serial");

            if ($services_serial != "") {
                $services = json_decode(base64_decode($services_serial), true);
            }

            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial");

            if ($serviceargs_serial != "") {
                $serviceargs = json_decode(base64_decode($serviceargs_serial), true);
            }

            // Get ticket used to access Nagios XI backend
            if ($old_auth) {
                $ticket = nagiosxiserver_configwizard_get_ticket($url_replaced, $username_replaced, $password_replaced);
            }

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $hostname = nagiosccm_replace_user_macros($hostname);
            $url = grab_array_var($inargs, "url", "");
            $apikey = grab_array_var($inargs, "apikey", "");
            $old_auth = grab_array_var($inargs, "old_auth", 0);
            $username = grab_array_var($inargs, "username", "");
            $ticket = grab_array_var($inargs, "ticket", "");

            $services = grab_array_var($inargs, "services", array());
            $serviceargs = grab_array_var($inargs, "serviceargs", array());

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (is_valid_host_name($hostname) == false) {
                $errmsg[$errors++] = _("Invalid host name.");
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;


        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $url = grab_array_var($inargs, "url", "");
            $apikey = grab_array_var($inargs, "apikey", "");
            $old_auth = grab_array_var($inargs, "old_auth", 0);
            $username = grab_array_var($inargs, "username", "");
            $ticket = grab_array_var($inargs, "ticket", "");

            $services = grab_array_var($inargs, "services");
            $serviceargs = grab_array_var($inargs, "serviceargs");

            $services_serial = grab_array_var($inargs, "services_serial", base64_encode(json_encode($services)));
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", base64_encode(json_encode($serviceargs)));

            $output = '
        <input type="hidden" name="ip_address" value="' . encode_form_val($address) . '">
        <input type="hidden" name="hostname" value="' . encode_form_val($hostname) . '">
        <input type="hidden" name="url" value="' . encode_form_val($url) . '">
        <input type="hidden" name="apikey" value="' . encode_form_val($apikey) . '">
        <input type="hidden" name="old_auth" value="' . intval($old_auth) . '">
        <input type="hidden" name="username" value="' . encode_form_val($username) . '">
        <input type="hidden" name="ticket" value="' . encode_form_val($ticket) . '">
        <input type="hidden" name="services_serial" value="' . $services_serial . '">
        <input type="hidden" name="serviceargs_serial" value="' . $serviceargs_serial . '">
';
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:
            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:
            $output = '';
            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            $hostname = grab_array_var($inargs, "hostname", "");
            $address = grab_array_var($inargs, "ip_address", "");
            $url = grab_array_var($inargs, "url", "");
            $apikey = grab_array_var($inargs, "apikey", "");
            $old_auth = grab_array_var($inargs, "old_auth", 0);
            $username = grab_array_var($inargs, "username", "");
            $ticket = grab_array_var($inargs, "ticket", "");

            $services_serial = grab_array_var($inargs, "services_serial", "");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            $services = json_decode(base64_decode($services_serial), true);
            $serviceargs = json_decode(base64_decode($serviceargs_serial), true);

            // Save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["hostname"] = $hostname;
            $meta_arr["ip_address"] = $address;
            $meta_arr["url"] = $url;
            $meta_arr["apikey"] = $apikey;
            $meta_arr["old_auth"] = $old_auth;
            $meta_arr["username"] = $username;
            $meta_arr["ticket"] = $ticket;
            $meta_arr["services"] = $services;
            $meta_arr["serivceargs"] = $serviceargs;
            save_configwizard_object_meta($wizard_name, $hostname, "", $meta_arr);

            $objs = array();

            if (!host_exists($hostname)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_nagiosxiserver_host",
                    "host_name" => $hostname,
                    "address" => $address,
                    "icon_image" => "nagiosxiserver.png",
                    "statusmap_image" => "nagiosxiserver.png",
                    "_xiwizard" => $wizard_name,
                );
            }

            // common plugin opts
            $commonopts = "--address=\$HOSTADDRESS\$ --url='" . $url . "'";

            if ($old_auth) {
                $commonopts .= " --username=" . $username . " --ticket=\"" . $ticket . "\" ";
            } else {
                $commonopts .= " --apikey='" . $apikey . "'";
            }

            foreach ($services as $svcvar => $svcval) {

                $pluginopts = "";
                $pluginopts .= $commonopts;

                switch ($svcvar) {

                    case "ping":

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Ping",
                            "use" => "xiwizard_nagiosxiserver_ping_service",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "webui":

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "HTTP",
                            "use" => "xiwizard_nagiosxiserver_http_service",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "daemons":

                        $pluginopts = $commonopts . " --mode=daemons";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Nagios XI Daemons",
                            "use" => "xiwizard_nagiosxiserver_service",
                            "check_command" => "check_xi_nagiosxiserver!" . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "jobs":

                        $pluginopts = $commonopts . " --mode=jobs";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Nagios XI Jobs",
                            "use" => "xiwizard_nagiosxiserver_service",
                            "check_command" => "check_xi_nagiosxiserver!" . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "iowait":

                        $pluginopts = $commonopts . " --mode=iowait --warn=\"" . grab_array_var($serviceargs, "iowait_warning") . "\" --crit=\"" . grab_array_var($serviceargs, "iowait_critical") . "\"";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "I/O Wait",
                            "use" => "xiwizard_nagiosxiserver_service",
                            "check_command" => "check_xi_nagiosxiserver!" . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "load":

                        $pluginopts = $commonopts . " --mode=load --warn=\"" . grab_array_var($serviceargs, "load_warning") . "\" --crit=\"" . grab_array_var($serviceargs, "load_critical") . "\"";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Load",
                            "use" => "xiwizard_nagiosxiserver_service",
                            "check_command" => "check_xi_nagiosxiserver!" . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;


                    default:
                        break;
                }
            }

            // Return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:
            break;
    }

    return $output;
}
