<?php
//
// Network Report
// Copyright (c) 2014-2020 Nagios Enterprises, LLC. All rights reserved.
//

require_once(dirname(__FILE__) . '/../componenthelper.inc.php');

// Initialization stuff
pre_init();
init_session();

// Grab GET or POST variables and check authentication
grab_request_vars();
check_prereqs();
check_authentication(false);

route_request();

function route_request()
{
    global $request;

    $mode = grab_request_var("mode", "");
    switch ($mode) {

        case "pdf":
            get_nna_reports_pdf();
            break;

        default:
            display_nna_reports();
            break;
    }
}

// Display the Reports
// - Grabs saved reports from the current NNA Server(s) selected in config settings and displays them
function display_nna_reports()
{
    do_page_start(array("page_title" => _("Network Report")), true);

    // Check to make sure the user has NNA integration or not, display some info on
    // how to get integration if they don't already have Network Analyzer
    if (!has_nna_servers()) {
        display_nna_no_servers();
        do_page_end(true);
        exit();
    }

    $temp = nna_get_first_data();

    // Grab the passed values
    $hideoptions = grab_request_var("hideoptions", 0);
    $rid = grab_request_var("report", 1);
    $object_type = grab_request_var("object_type", "source");
    $object = grab_request_var("object", $temp['object']);
    $server = grab_request_var("nna_server", $temp['server']);
    $object_name = grab_request_var("object_name", $temp['object_name']);
    $view_id = grab_request_var("view_id", "");
    $view_name = grab_request_var("view_name", "");
    $excluders = grab_request_var("excluders", array());
    $manual_run = grab_request_var("manual_run", 0);

    $server_data = json_decode($server, true);
    $address = $server_data['address'] ?? '';
    $secure = $server_data['secure'] ?? 0;
    $api_key = $server_data['api_key'] ?? '';
    $look_back_hours = $server_data['look_back_hours'] ?? 4;
    $allow_invalid_ssl = $server_data['allow_invalid_ssl'] ?? 0;

    if (empty($allow_invalid_ssl)) {
        $allow_invalid_ssl = false;
    }

    /* // this isn't used anywhere other than in the tab
    if (empty($look_back_hours)) {
        $look_back_hours = DEFAULT_LOOK_BACK_HOURS;
    }
    $look_back_hours = intval($look_back_hours);
    $look_back_hours = "-" . $look_back_hours;
    */

    // If we have an object and rid and not disabled loading
    $disable_report_auto_run = get_option("disable_report_auto_run", 0);
    if ($object > 0 && $rid > 0 && ($manual_run == 1 || $disable_report_auto_run == 0)) {

        // Create the api url
        if ($secure) {
            $sec = "s";
        } else {
            $sec = "";
        }
        $base_url = "http" . $sec . "://" . $address . "/nagiosna/";
        $nna_url = $base_url . "index.php";
        $api_url = $base_url . "api/";
        $report_viz_url = $api_url . "reports/reportviz?rid=" . $rid . "&token=" . $api_key;

        $url = $api_url . "reports/execute?q[rid]=" . $rid;
        if ($object_type == "source") {
            $url .= "&q[sid]=" . $object;
            $report_viz_url .= "&sid=" . $object;
        } else if ($object_type == "sourcegroup") {
            $url .= "&q[gid]=" . $object;
            $report_viz_url .= "&gid=" . $object;
        }

        // Check if view and add on
        if (!empty($view_name)) {
            $url .= "&q[vid]=" . $view_id;
            $report_viz_url .= "&vid=" . $view_id;
        }

        // Get the report data
        $url .= "&token=" . $api_key;
        $json = nna_file_get_contents($url, $allow_invalid_ssl);
        $report_data = json_decode($json);

        // Grab the actual report info
        $url = $api_url . "reports/read?q[rid]=" . $rid;
        $url .= "&token=" . $api_key;
        $json = nna_file_get_contents($url, $allow_invalid_ssl);
        $report = json_decode($json);
        $report = $report[0];

    }

    // Hide all options if the users is generating a PDF
    if (!$hideoptions) {
        ?>

        <script type="text/javascript">
            var nna_server = "<?php json_encode($server); ?>";
            var report_id = <?php echo $rid; ?>;
            var object_type = "<?php echo $object_type; ?>";
            var object = <?php if (empty($object)) { echo '""'; } else { echo $object; } ?>;
            var object_name = "<?php echo $object_name; ?>";
            var view_id = "<?php echo $view_id; ?>";
            var show_excluders = <?php if (empty($excluders)) { echo 0; } else { echo 1; } ?>;

            $(document).ready(function() {

                if (nna_server == "") {
                    nna_server = $('#nna_server').val();
                }

                // Load all reports
                load_reports_from_server(nna_server);

                // Load default sources or sourcegroups
                if (object_type == "source") {
                    load_sources_from_server(nna_server);
                } else if (object_type == "sourcegroup") {
                    load_sourcegroups_from_server(nna_server);
                }
                $('select[name="object_type"]').val(object_type);

                // Load reports on server change
                $('#nna_server').change(function () {
                    $('#use-view-container').hide();
                    load_reports_from_server($(this).val());
                    load_nna_objects();
                });

                // Save the view_id when changing views
                $('#views').change(function () {
                    view_id = $('#views option:selected').val();
                    $('#view_name').val($('#views option:selected').text());
                });

                // Show excluders
                $('#show-excluders').click(function () {
                    if (show_excluders) {
                        $('#excluders').hide();
                        show_excluders = 0;
                        $('#show-excluders i').html('add');
                    } else {
                        $('#excluders').show();
                        show_excluders = 1;
                        $('#show-excluders i').html('remove');
                    }
                });

                // Load sources or sourcegroups on change
                $('select[name="object_type"]').change(function() {
                    load_nna_objects();
                });

                $('#object').change(function () {
                    var obj = $(this).find("option:selected");
                    var server = $('#nna_server').val();
                    server = JSON.parse(server);
                    var api_url = get_nna_api_url(server['address'], server['secure']);
                    load_sources_views(api_url, server['api_key'], obj.val());
                    $('#object_name').val(obj.text());
                });

            });

            function load_reports_from_server(server) {
                var server = JSON.parse(server);
                var api_url = get_nna_api_url(server['address'], server['secure']);
                $('#error').hide();

                $.post('<?php get_base_url();?> /nagiosxi/includes/components/nagiosna/nna-ajaxhelper.php', {
                    cmd: 'get_query_data',
                    nsp: '<?php echo get_nagios_session_protector_id(); ?>',
                    server: JSON.stringify(server),
                    endpoint: 'reports/read',
                    ssl: '<?php echo $allow_invalid_ssl?>'
                }, function() {}, "json")
                .done(function(data){

                    $('#report').html("");
                    $.each(data, function (k, v) {
                        var selected = "";
                        if (v.rid == report_id) {
                            selected = "selected";
                        }
                        $('#report').append('<option value="' + v.rid + '" ' + selected + '>' + v.name + '</option>');
                    });

                }, 'json')
                    .fail(function (data) {
                        var error = '<?php echo _("Failed connect to API. Check your connection to the host (using SSL?) and make sure your Nagios Network Analyzer is version 2014R1.5 or higher."); ?>';
                        if (data.status == 404) {
                            error = '<?php echo _("404 - API not found. The address may be wrong."); ?>';
                        }
                        $("#error").html(error).show();
                    });

            }

            function load_sources_from_server(server) {
                var server = JSON.parse(server);
                var api_url = get_nna_api_url(server['address'], server['secure']);

                $.post('<?php get_base_url();?> /nagiosxi/includes/components/nagiosna/nna-ajaxhelper.php', {
                    cmd: 'get_query_data',
                    nsp: '<?php echo get_nagios_session_protector_id(); ?>',
                    server: JSON.stringify(server),
                    endpoint: 'sources/read',
                    ssl: '<?php echo $allow_invalid_ssl?>'
                }, function() {}, "json")
                .done(function(data){

                    $('#object').html("");
                    var selected_id = 0;
                    $.each(data, function (k, v) {
                        var selected = "";
                        if (object_type == "source" && object == v.sid) {
                            selected = "selected";
                            selected_id = v.sid;
                        }
                        $('#object').append('<option value="' + v.sid + '" ' + selected + '>' + v.name + '</option>');
                    });

                    // Make sure the actual name is saved and sent
                    $('#object_name').val($('#object option:selected').text());

                    // Grab the source and create a "Use View checkbox if it has any associated views"
                    load_sources_views(api_url, server["api_key"], selected_id);

                }, 'json');
            }

            function load_sourcegroups_from_server(server) {
                var server = JSON.parse(server);
                var api_url = get_nna_api_url(server['address'], server['secure']);

                $.post('<?php get_base_url();?> /nagiosxi/includes/components/nagiosna/nna-ajaxhelper.php', {
                    cmd: 'get_query_data',
                    nsp: '<?php echo get_nagios_session_protector_id(); ?>',
                    server: JSON.stringify(server),
                    endpoint: 'groups/read',
                    ssl: '<?php echo $allow_invalid_ssl?>'
                }, function() {}, "json")
                .done(function(data){

                    $('#object').html("");
                    $.each(data, function (k, v) {
                        var selected = "";
                        if (object_type == "sourcegroup" && object == v.gid) {
                            selected = "selected";
                        }
                        $('#object').append('<option value="' + v.gid + '" ' + selected + '>' + v.name + '</option>');
                    });

                    // Make sure the actual name is saved and sent
                    $('#object_name').val($('#object option:selected').text());

                }, 'json');
            }

            function load_sources_views(api_url, token, sid) {
                $.post('<?php get_base_url();?> /nagiosxi/includes/components/nagiosna/nna-ajaxhelper.php', {
                    cmd: 'get_source_views',
                    nsp: '<?php echo get_nagios_session_protector_id(); ?>',
                    url: api_url,
                    token: token,
                    sid: sid,
                    endpoint: 'views/get_views',
                    ssl: '<?php echo $allow_invalid_ssl?>'
                }, function() {}, "json")
                .done(function(data){

                    if (!data.error) {
                        var views = data;
                        if (views.length > 0) {
                            $('#use-view-container').show();
                            $('#views').html('<option value=""></option>');
                            $.each(views, function (k, v) {

                                var selected = "";
                                if (v.vid == view_id) {
                                    selected = " selected";
                                }

                                $('#views').append('<option value="' + v.vid + '"' + selected + '>' + v.name + '</option>');
                            });
                        } else {
                            $('#use-view-container').hide();
                        }

                        // Make sure the actual name is saved and sent
                        $('#view_name').val($('#views option:selected').text());
                    }

                }, 'json');
            }

            function get_nna_api_url(address, https) {
                var secure = "";
                if (https == "1") {
                    secure = "s";
                }

                return "http" + secure + "://" + address + "/nagiosna/api/";
            }

            function load_nna_objects() {
                var checked_type = $('select[name="object_type"] option:selected').val();
                if (checked_type == "source") {
                    load_sources_from_server($('#nna_server').val());
                } else if (checked_type == "sourcegroup") {
                    $('#use-view-container').hide();
                    $('#view_name').val('');
                    load_sourcegroups_from_server($('#nna_server').val());
                }
            }
        </script>

<script type='text/javascript' src='<?php echo get_base_url(); ?>includes/js/reports.js?<?php echo get_build_id(); ?>'></script>

<form method="get">
<div class="well report-options form-inline">

        <div class="reportexportlinks">
            <?php
            $pdf_url = $_SERVER['REQUEST_URI'];
            echo get_add_myreport_html("Nagios Network Analyzer Report", $pdf_url, array());
            if (strpos($pdf_url, "?") === false) {
                $pdf_fixed_url = $pdf_url . "?mode=pdf";
            } else {
                $pdf_fixed_url = $pdf_url . "&mode=pdf";
            }
            ?>
            <a title='<?php echo _("Download as PDF"); ?>' class='btn btn-sm btn-default tt-bind icon-in-btn btn-export' data-placement='bottom' href="<?php echo $pdf_fixed_url; ?>" ><i class='material-symbols-outlined md-middle md-400'>picture_as_pdf</i></a>
        </div>

        <div class="neptune-drawer-options">
        
        <div class="reportoptionpicker">
            <div class="input-group">
                <label class="input-group-addon"><?php echo _('Server'); ?></label>
                <?php display_nna_servers(false, $server); ?>
            </div>

            <div class="input-group">
                <label class="input-group-addon"><?php echo _('Data'); ?></label>
                    <select name="object_type" class="form-control">
                        <option value="source"><?php echo _("Source"); ?></option>
                        <option value="sourcegroup"><?php echo _("Source Group"); ?></option>
                    </select>
                    <select id="object" name="object" style="border-left: 0;" class="form-control"></select>
                    <span style="<?php if (empty($view_name)) { echo "display: none;"; } ?>" id="use-view-container"><select id="views" name="view_id" style="border-left: 0;" class="form-control rounded-r"></select></span>
                    <input type="hidden" id="view_name" name="view_name" value="<?php echo $view_name; ?>">
                </span>
            </div>

            <div class="input-group input-group-margin-left">
                <label class="input-group-addon"><?php echo _('Report'); ?></label>
                <select id="report" name="report" class="form-control"></select>
            </div>

            <span style="margin: 0 10px;" id="network-report-toggle">
                <?php if (empty($excluders)) { ?>
                    <button type="button" class="tt-bind btn btn-sm btn-default neptune-btn " data-placement="bottom" title="<?php echo _("Toggle column excluders"); ?>" id="show-excluders">
                        <i class="material-symbols-outlined md-20 md-400">add</i>
                    </button>
                <?php } else { ?>
                    <button type="button" class="tt-bind btn btn-sm btn-default neptune-btn" data-placement="bottom" title="<?php echo _("Toggle column excluders"); ?>" id="show-excluders">
                        <i class="material-symbols-outlined md-20 md-400">remove</i>
                    </button>
                <?php } ?>
            </span>

            <button class="btn btn-sm btn-primary neptune-btn " style="vertical-align: top;" id="run" type="submit"><?php echo _("Run"); ?></button>
        </div>

        <div style="margin-bottom: 10px; <?php if (empty($excluders)) { echo "display: none;"; } ?>" id="excluders" class="excluders">
            <table>
                <tr>
                    <td>
                        <span><?php echo _("Exclude the checked columns from the report table"); ?>: &nbsp;</span>
                    </td>
                    <td class="nna-select-table">
                        <span class="checkbox">
                            <label>
                                <input type="checkbox" name="excluders[]" value="start" <?php if (in_array("start", $excluders)) { echo "checked"; } ?>>
                                <?php echo _("Start Date"); ?>
                            </label>
                        </span>
                        <span class="checkbox">
                            <label>
                                <input type="checkbox" name="excluders[]" value="end" <?php if (in_array("end", $excluders)) { echo "checked"; } ?>>
                                <?php echo _("End Date"); ?>
                            </label>
                        </span>
                        <span class="checkbox">
                            <label>
                                <input type="checkbox" name="excluders[]" value="duration" <?php if (in_array("duration", $excluders)) { echo "checked"; } ?>>
                                <?php echo _("Duration"); ?>
                            </label>
                        </span>
                        <span class="checkbox">
                            <label>
                                <input type="checkbox" name="excluders[]" value="protocol" <?php if (in_array("protocol", $excluders)) { echo "checked"; } ?>>
                                <?php echo _("Protocol"); ?>
                            </label>
                        </span>
                        <span class="checkbox">
                            <label>
                                <input type="checkbox" name="excluders[]" value="srcip" <?php if (in_array("srcip", $excluders)) { echo "checked"; } ?>>
                                <?php echo _("Source IP"); ?>
                            </label>
                        </span>
                        <span class="checkbox">
                            <label>
                                <input type="checkbox" name="excluders[]" value="dstip" <?php if (in_array("dstip", $excluders)) { echo "checked"; } ?>>
                                <?php echo _("Destination IP"); ?>
                            </label>
                        </span>
                        <span class="checkbox">
                            <label>
                                <input type="checkbox" name="excluders[]" value="srcport" <?php if (in_array("srcport", $excluders)) { echo "checked"; } ?>>
                                <?php echo _("Source Port"); ?>
                            </label>
                        </span>
                        <span class="checkbox">
                            <label>
                                <input type="checkbox" name="excluders[]" value="dstport" <?php if (in_array("dstport", $excluders)) { echo "checked"; } ?>>
                                <?php echo _("Destination Port"); ?>
                            </label>
                        </span>
                    </td>
                </tr>
                <tr>
                    <?php if (!is_neptune()) { ?><td><?php echo _("(Useful if you can't fit the data on a pdf)"); ?></td> <?php } ?>
                    <td class="nna-select-table">
                        <span class="checkbox">
                            <label>
                                <input type="checkbox" name="excluders[]" value="flows" <?php if (in_array("flows", $excluders)) { echo "checked"; } ?>>
                                <?php echo _("Flows"); ?>
                            </label>
                        </span>
                        <span class="checkbox">
                            <label>
                                <input type="checkbox" name="excluders[]" value="flowspercent" <?php if (in_array("flowspercent", $excluders)) { echo "checked"; } ?>>
                                <?php echo _("Flows %"); ?>
                            </label>
                        </span>
                        <span class="checkbox">
                            <label>
                                <input type="checkbox" name="excluders[]" value="packets" <?php if (in_array("packets", $excluders)) { echo "checked"; } ?>>
                                <?php echo _("Packets"); ?>
                            </label>
                        </span>
                        <span class="checkbox">
                            <label>
                                <input type="checkbox" name="excluders[]" value="packetspercent" <?php if (in_array("packetspercent", $excluders)) { echo "checked"; } ?>>
                                <?php echo _("Packets %"); ?>
                            </label>
                        </span>
                        <span class="checkbox">
                            <label>
                                <input type="checkbox" name="excluders[]" value="bytes" <?php if (in_array("bytes", $excluders)) { echo "checked"; } ?>>
                                <?php echo _("Bytes"); ?>
                            </label>
                        </span>
                        <span class="checkbox">
                            <label>
                                <input type="checkbox" name="excluders[]" value="bytespercent" <?php if (in_array("bytespercent", $excluders)) { echo "checked"; } ?>>
                                <?php echo _("Bytes %"); ?>
                            </label>
                        </span>
                        <span class="checkbox">
                            <label>
                                <input type="checkbox" name="excluders[]" value="pps" <?php if (in_array("pps", $excluders)) { echo "checked"; } ?>>
                                <?php echo _("Packets/Sec"); ?>
                            </label>
                        </span>
                        <span class="checkbox">
                            <label>
                                <input type="checkbox" name="excluders[]" value="bps" <?php if (in_array("bps", $excluders)) { echo "checked"; } ?>>
                                <?php echo _("Bits/Sec"); ?>
                            </label>
                        </span>
                        <span class="checkbox">
                            <label>
                                <input type="checkbox" name="excluders[]" value="bpp" <?php if (in_array("bpp", $excluders)) { echo "checked"; } ?>>
                                <?php echo _("Bytes/Packet"); ?>
                            </label>
                        </span>
                    </td>
                </tr>
            </table>
        </div>

        <!-- Set a variable to let us know it's okay to run this -->
        <input type="hidden" name="manual_run" value="1">
        <input type="hidden" value="" name="object_name" id="object_name">

        </div>


    </div>
</form>

    <div class="error" id="error" style="display:none;"></div>

    <?php

    } // End: Hideoptions

    if ($report && $report_data) {

        $dn = false;

        // Strip record header (the top th part) from records
        $records = json_decode(json_encode($report_data->records), true);
        $head = array_slice($records, 0, 1);
        $head = $head[0];
        $records = array_slice($records, 1);
        $summary = $report_data->summary;

        if ($records > 0) {
            $record = $records[0];
            if (is_array($record) && ( array_key_exists('srcdn', $record) || array_key_exists('dstdn', $record))) {
                $dn = true;
            }
        }

        $order = $report->toporder;

        // Let's display the graphs?
        $vizdata = json_decode(nna_file_get_contents($report_viz_url, $allow_invalid_ssl));

        // Get aggs and aggt
        foreach ($vizdata->datasets as $k => $v) {
            $aggs[] = $k;
            $aggt[] = $v;
        }

        $pcolor = "#000000";
        if (is_dark_theme()) {
            $pcolor = '#EEEEEE';
        }
        if (is_neptune()) {
            $pcolor = 'var(--foreground)';
        }
        ?>

        <script>
        var aggs = <?php echo json_encode($aggs); ?>;
        var aggt = <?php echo json_encode($aggt); ?>;
        var g_title = '<?php echo $report->name; ?>';

        $(document).ready(function () {
            make_pie();
            $.each(aggs, function (i, d) {
                make_chord(d, 600, 900, aggt[i]);
            });

            $(window).resize(function() {
                make_pie();
            });

            $('.fullscreen-button').click(function () {
                resize_to_fullscreen(this);
            });

        });

        function resize_to_fullscreen(node) {
            $('#whiteout').css({ opacity: "0.95" });
            whiteout();

            var div_id = $(node).attr('data');
            $(div_id).addClass('front-and-center');
            $(div_id).css({'z-index': 10000});

            $(div_id).css('position', 'fixed')
                     .css('top', 10)
                     .css('left', 10)
                     .css('right', 10)
                     .css('bottom', 10)
                     .css('height', 'auto');

            switch (div_id) {
                case '#pie':
                    pie.setSize($(div_id).width(), $(div_id).height() - 140);
                    var opt = pie.series[0].options;
                    opt.dataLabels.enabled = true;
                    pie.series[0].update(opt);
                    break;
                <?php foreach($vizdata->datasets as $dataset => $garbage): ?>
                case '#<?php echo $dataset; ?>':
                    $(div_id).find('svg').height($(div_id).height() - 140);
                    break;
                <?php endforeach ?>
                default:
                    break;
            }

            var title = $(node).parent().find('span').html();
            $(div_id).prepend('<span class="graph-title">' + title + '</span><div class="close-perfdata-popup btn-close" onclick="resize_to_normal(\'' + div_id + '\')" style="z-index: 100001;""></div>');
        }

        function resize_to_normal(div_id) {
            clear_whiteout();
            $('.close-perfdata-popup').remove();

            $(div_id).css({ top: 0, left: 0, 'z-index': 1, position: 'inherit' })
                .removeClass('front-and-center')
                .width('').height('')
                .children('span.graph-title').each(function (i, d) {
                    $(d).remove();
                });

            switch (div_id) {
                case '#pie':
                    pie.setSize($('#pie').width(), $('#pie').height());
                    pie.setTitle('');
                    var opt = pie.series[0].options;
                    opt.dataLabels.enabled = false;
                    pie.series[0].update(opt);
                    break;
                <?php foreach($vizdata->datasets as $dataset => $garbage): ?>
                case '#<?php echo $dataset; ?>':
                    $(div_id).find('svg').height('100%');
                    break;
                <?php endforeach ?>
                default:
                    break;
            }
        }

        function make_pie() {
            var json = <?php echo $vizdata->json; ?>;

            $(function () {

                pie = new Highcharts.Chart({
                    chart: {
                        renderTo: 'pie',
                        backgroundColor: null,
                        plotBorderWidth: null,
                        plotShadow: false,
                        animation: false
                    },
                    exporting: {
                        enabled: false
                    },
                    credits: {
                        enabled: false
                    },
                    title: {
                        text: ' '
                    },
                    tooltip: {
                        pointFormat: '{series.name}: <b>{point.percentage:.1f}%</b>'
                    },
                    plotOptions: {
                        pie: {
                            allowPointSelect: true,
                            animation: false,
                            cursor: 'pointer',
                            showInLegend: true,
                            dataLabels: {
                                enabled: false,
                                color: '<?php echo $pcolor; ?>',
                                connectorColor: '<?php echo $pcolor; ?>',
                                format: '<b>{point.name}</b>: {point.percentage:.1f} %'
                            }
                        }
                    },
                    series: json
                });
                pie.setTitle('');

                var n_width = $('#pie').width();
                if (n_width == 0) {
                    n_width = 200;
                }
                pie.setSize(n_width, $('#pie').height());
            });

            if (json[0].data[0].name == null) {
                $('#highcharts-0').html('<div class="text-center"><?php echo _("No Data"); ?></div>');
            }
        }

        function make_chord(agg2, h, w, title) {

            var r1 = h / 2 - 50,
                r0 = r1 - 80;

            var fill = d3.scale.category20c();

            var chord = d3.layout.chord()
                .padding(.04)
                .sortSubgroups(d3.descending)
                .sortChords(d3.descending);

            var arc = d3.svg.arc()
                .innerRadius(r0)
                .outerRadius(r0 + 20);

            var svg = d3.select("#" + agg2).append("svg:svg")
                .attr("viewBox", "0 0 " + w + " " + h)
                .attr("width", "100%")
                .attr("height", "100%")
                .append("svg:g")
                .attr("transform", "translate(" + w / 2 + "," + h / 2 + ")");
            window[agg2] = svg;
            // hit our ajaxhelper here instead of hitting the nna api directly.
            $.post('<?php get_base_url();?> /nagiosxi/includes/components/nagiosna/nna-ajaxhelper.php', {
                cmd: 'make_report_chord',
                nsp: '<?php echo get_nagios_session_protector_id(); ?>',
                apiurl: '<?php echo $vizdata->apiurl;?>',
                chordquery: '<?php echo $vizdata->chordquery;?>',
                agg_suffix: agg2,
                ip: '<?php echo $address?>',
                security: '<?php echo $sec?>',
                ssl: '<?php echo $allow_invalid_ssl?>'
            }, function() {}, "json")
            .done(function(imports){

                var indexByName = {},
                    nameByIndex = {},
                    matrix = imports.matrix,
                    labels = imports.names,
                    diff = imports.diff,
                    warning = imports.warning,
                    n = 0;

                self.names = [];

                // Compute a unique index for each package name.
                labels.forEach(function (d) {
                    if (!(d in indexByName)) {
                        nameByIndex[n] = d;
                        indexByName[d] = n++;
                        names.push(d);
                    }
                });

                chord.matrix(matrix);

                var g = svg.selectAll("g.group")
                    .data(chord.groups)
                    .enter().append("svg:g")
                    .attr("class", "group")
                    .on("mouseover", fade(.02))
                    .on("mouseout", fade(.80));

                g.append("svg:path")
                    .style("stroke", function (d) {
                        return fill(d.index);
                    })
                    .style("fill", function (d) {
                        return fill(d.index);
                    })
                    .attr("d", arc);

                g.append("svg:text")
                    .each(function (d) {
                        d.angle = (d.startAngle + d.endAngle) / 2;
                    })
                    .attr("dy", ".35em")
                    .attr("text-anchor", function (d) {
                        return d.angle > Math.PI ? "end" : null;
                    })
                    .attr("transform", function (d) {
                        return "rotate(" + (d.angle * 180 / Math.PI - 90) + ")"
                        + "translate(" + (r0 + 26) + ")"
                        + (d.angle > Math.PI ? "rotate(180)" : "");
                    })
                    .attr("class", function (d) {
                        return (diff[d.index]) ? 'black' : 'gray';
                    })
                    .text(function (d) {
                        return nameByIndex[d.index];
                    });

                svg.selectAll("path.chord")
                    .data(chord.chords)
                    .enter().append("svg:path")
                    .attr("class", "chord")
                    .style("stroke", function (d) {
                        return d3.rgb(fill(d.source.index)).darker();
                    })
                    .style("fill", function (d) {
                        return fill(d.source.index);
                    })
                    .attr("d", d3.svg.chord().radius(r0));

                $('#' + agg2).children('.reportviz-throbber').hide();

                svg.append('svg:title').text(title);

                if (warning) {
                    svg.append('text')
                        .attr('x', width / 2)
                        .attr('y', 0 - (margin.top / 2))
                        .attr('text-anchor', 'middle')
                        .style('font-size', '10px')
                        .text('Dataset was truncated due to size.');
                }

            });

            // Returns an event handler for fading a given chord group.
            function fade(opacity) {
                return function (d, i) {
                    svg.selectAll("path.chord")
                        .filter(function (d) {
                            return d.source.index != i && d.target.index != i;
                        })
                        .transition()
                        .style("stroke-opacity", opacity)
                        .style("fill-opacity", opacity);
                };
            }
        }
        </script>

        <div>

            <h1 style="padding: 0;"><?php echo $report->name; ?></h1>

            <p>
                <div class="neptune-form-subtext"><?php if ($object_type == 'sourcegroup') { echo _('Source Group'); } else { echo ucfirst($object_type); } ?>:
                    <b><?php echo $object_name; ?></b> <?php if ($view_name) {
                        echo _('using view ') . "<b>" . $view_name . "</b>";
                    } ?> <?php echo _('on'); ?> <b><?php echo get_nna_server_name_from_address($address); ?></b>
                    (<?php echo _("Showing top") . " " . $report->top . " " . _("based on") . " <i>" . nna_human_readable_header('val', $report) . "</i> " . _("and ordered by") . " <i>" . nna_human_readable_header($report->toporder); ?></i>)
                </div>
                <div class="neptune-form-subtext" style="margin-top: 5px;">
                    <?php echo _("Timeframe") . ": <b>" . nna_human_readable_timeframe($report); ?></b>
                </div>
                <?php
                if (!empty($report->rawquery)) {
                    echo '<div style="margin-top: 5px;">' . _("Limiters") . ': ' . $report->rawquery . '</div>';
                }
                ?>
            </p>

            <div class="container-fluid" style="margin: 15px auto; padding-right: 5px; padding-left: 5px;">
                <div class="row thumbnails">
                    <div class="col-xs-6 col-sm-6 col-lg-6">
                        <div class="thumbnail" style="height: 250px;">
                            <div id="pie" class="chartrender">
                            </div>
                        </div>
                        <h5>
                            <?php if (!$hideoptions) { ?>
                            <a class="btn btn-sm btn-default fullscreen-button" style="margin-right: 5px;" data="#pie">
                                <i class="material-symbols-outlined md-400 md-18">open_in_full</i>
                            </a>
                            <?php } ?>
                            <span><?php echo _("Percent Usage Chart"); ?></span>
                        </h5>
                    </div>
                    <?php
                    foreach ($vizdata->datasets as $dataset => $description) {
                        $description = str_replace("<=>", '<i class="fa fa-arrows-h"></i>', $description);
                    ?>
                        <div class="col-xs-6 col-sm-6 col-lg-6">
                            <div class="thumbnail" style="height: 250px;">
                                <div id="<?php echo $dataset; ?>" class="chartrender">
                                    <div class="reportviz-throbber"><div class="valign-throbber centerme"><i class="fa fa-spinner fa-spin"></i></div></div>
                                </div>
                            </div>
                            <h5>
                                <?php if (!$hideoptions) { ?>
                                <a class="btn btn-sm btn-default fullscreen-button" style="margin-right: 5px;" data="#<?php echo $dataset; ?>">
                                    <i class="material-symbols-outlined md-400 md-18">open_in_full</i>
                                </a>
                                <?php } ?>
                                <span><?php echo _("Chord Diagram") . " (" . $description . ")"; ?></span>
                            </h5>
                        </div>
                    <?php } ?>
                </div>
            </div>

            <table class="table table-condensed table-striped table-bordered">
                <thead>
                <tr>
                    <?php
                    foreach ($head as $k => $h) {
                        if (!in_array($k, $excluders)) {
                            $highlighted = "";
                            if ($k == $report->toporder) {
                                $highlighted = 'class="highlight"';
                            }                
                            echo '<th class="neptune-td-nowrap"' . $highlighted . '>' . nna_human_readable_header($h, $report, $dn) . '</th>';
                        }
                    }
                    ?>
                </tr>
                </thead>
                <tbody>
                <?php
                // Display a message if there is no data
                if (count($records) == 0) {
                    echo '<tr><td colspan="99">'._("No data returned for this report.").'</td></tr>';
                } else {

                    // Display data
                    foreach ($records as $record) {
                        echo '<tr>';
                        foreach ($record as $k => $v) {
                            if ($k == "srcdn" || $k == "dstdn") { continue; }
                            if (!in_array($k, $excluders)) {

                                // Show a little bar with the amount of data sent
                                $display = '';
                                $make = false;
                                if ($k == "bytes") {
                                    $percent = round(($v / $summary->bytes) * 100, 2);
                                    $make = true;
                                } else if ($k == "flows") {
                                    $percent = round(($v / $summary->flows) * 100, 2);
                                    $make = true;
                                } else if ($k == "packets") {
                                    $percent = round(($v / $summary->packets) * 100, 2);
                                    $make = true;
                                }

                                // Make the little bar's html
                                if ($make) {

                                    // Theme colors
                                    $prog_color = "#666";
                                    $bar_color = "#FFF";
                                    if (get_theme() == "xi5dark") {
                                        $prog_color = "#CCC";
                                        $bar_color = "#000";
                                    }

                                    $display = '<div style="background-color: ' . $bar_color . '; height: 2px; width: 100%;"><div style="background-color: ' . $prog_color . '; height: 2px; width: ' . $percent . '%;"></div></div>';
                                }

                                $highlighted = "";
                                if ($k == $report->toporder) {
                                    $highlighted = 'class="highlight"';
                                }

                                if ($dn) {
                                    if ($k == "srcip") {
                                        if (array_key_exists('srcdn', $record)) {
                                            $v = $record->srcdn;
                                        }
                                    } else if ($k == "dstip") {
                                        if (array_key_exists('dstdn', $record)) {
                                            $v = $record->dstdn;
                                        }
                                    }
                                }

                                echo '<td ' . $highlighted . '>' . nna_human_readable_value($v, $k) . $display .'</td>';
                            }
                        }
                        echo '</tr>';
                    }
                }
                ?>
                </tbody>
            </table>

        </div>

    <?php
    } // End Report

    do_page_end(true);
}

// End: Display reports (display_nna_reports)

// Function to do the pdf generating for this report
function get_nna_reports_pdf()
{
    global $cfg;

    // Grab the backend ticket and username
    $username = $_SESSION["username"];
    $backend_ticket = get_user_attr(0, "backend_ticket");

    // Assemble actual URL that will be gotten
    $uri = str_replace("mode=pdf", "hideoptions=1", $_SERVER["REQUEST_URI"]);
    $fullurl = "http://127.0.0.1{$uri}";
    $urlparts = parse_url($fullurl);
    $uri = str_replace("/nagiosxi", "", $urlparts['path']);
    $newurl = get_internal_url() . $uri;
    if (!empty($urlparts['query'])) {
        $newurl .= "?";
        $newurl .= $urlparts['query'];
    }

    // Add single auth token
    $newurl .= "&token=".user_generate_auth_token(get_user_id($username));

    // Add language to url
    $language = $_SESSION['language'];
    $newurl .= "&locale=" . $language;

    // Do page rendering
    $aurl = $newurl;
    $reportid = time();

    $afile = "page.pdf";
    $filename = "scheduledreport-{$username}-{$reportid}-{$afile}";
    $fname = get_tmp_dir() . "/" . $filename;
    
    $args = [
        "filename" => $filename,
        "url" => $aurl,
        "type" => "pdf",
    ];

    $command_id = submit_command(COMMAND_DOWNLOAD_REPORT, serialize($args));

    if ($command_id > 0) {
        // Wait for up to 5 minutes
        for ($x = 0; $x < 300; $x++) {
            $status_code = -1;
            $result_code = -1;
            $args = [
                "command_id" => $command_id
            ];
            $xml = get_command_status_xml($args);
            if ($xml) {
                if ($xml->command[0]) {
                    $status_code = intval($xml->command[0]->status_code);
                    $result_code = intval($xml->command[0]->result_code);
                }
            }
            if ($status_code == 2) {
                if ($result_code == 0) {
                    break;
                }
            }
            sleep(1);
        }
    } else {
        echo "\n\n************\nERROR: Failed to render URL '" . $aurl . "' as '" . $fname . "'\n************\n\n";
        die();
    }

    if (!file_exists($fname)) {
        echo "\n\n************\nERROR: Failed to render URL '" . $aurl . "' as '" . $fname . "'\n************\n\n";
        die();
    } else {
        // We'll be outputting a PDF
        header('Content-type: application/pdf');

        // It will be called network_report.pdf
        header("Content-Disposition: attachment; filename=" . time() ."-network_report.pdf");

        // The PDF source is in original.pdf
        readfile($fname);
        unlink($fname);
    }
}
