<?php
//
// Mail Server Config Wizard
// Copyright (c) 2011-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

mailserver_configwizard_init();



function mailserver_configwizard_init()
{
    $name = "mailserver";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.1.0",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor an email server."),
        CONFIGWIZARD_DISPLAYTITLE => _("Mail Server"),
        CONFIGWIZARD_FUNCTION => "mailserver_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "mailserver.png",
        CONFIGWIZARD_FILTER_GROUPS => array('email', 'server'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );
    register_configwizard($name, $args);
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function mailserver_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    $wizard_name = "mailserver";

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            $nextstep = encode_form_val(grab_array_var($_POST,'nextstep',false),ENT_QUOTES);
            // Clear the session if a user is entering the wizard for the first time
            if ($nextstep == '') {
                unset($_SESSION[$wizard_name]); 
            }

            $address_session = grab_array_var($_SESSION[$wizard_name], 'ip_address', "");
            $address = grab_array_var($inargs, "ip_address", $address_session);

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            $back = array_key_exists("backButton", $_POST); 

            // Don't revalidate if user is just pressing back
            if ($back) {
                break;
            }

            // Get variables that were passed to us
            $address_session = grab_array_var($_SESSION[$wizard_name], 'ip_address', "");
            $address = grab_array_var($inargs, "ip_address", $address_session);
            $hostname_session = grab_array_var($_SESSION[$wizard_name], 'hostname', "");
            $hostname = grab_array_var($inargs, "hostname", $hostname_session);
            $address = nagiosccm_replace_user_macros($address);

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (have_value($address) == false) {
                $errmsg[$errors++] = _("No address specified.");
            } else if (!filter_var($address, FILTER_VALIDATE_IP)) {
                $errmsg[$errors++] = _("Invalid IP address.");
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            } else {
                $_SESSION[$wizard_name]['ip_address'] = $address;
                $_SESSION[$wizard_name]['hostname'] = $hostname;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            $back = encode_form_val(grab_array_var($_POST,'backButton',false),ENT_QUOTES);

            // Get variables that were passed to us
            $address_session = grab_array_var($_SESSION[$wizard_name], 'ip_address', "");
            $address = grab_array_var($inargs, "ip_address", $address_session);
            $hostname_session = grab_array_var($_SESSION[$wizard_name], 'hostname', "");
            $hostname = grab_array_var($inargs, "hostname", $hostname_session);
            $hostname = @gethostbyaddr($address);

            if ($hostname == "") {
                $hostname = $address;
            }

            $services_default = array(
                "ping" => 1,
                "smtp" => 1,
                "imap" => 1,
                "pop" => 1,
                "rbl" => 1,
            );

            $services = grab_array_var($_SESSION[$wizard_name], 'services', $services_default);
            $services = grab_array_var($inargs, 'services', $services);

            $serviceargs = grab_array_var($_SESSION[$wizard_name], "serviceargs", "");

            if (!is_array($serviceargs)) {
                $serviceargs_default = array(
                    "rbl_servers" => "zen.spamhaus.org bl.spamcop.net dnsbl.ahbl.org dnsbl.njabl.org dnsbl.sorbs.net virbl.dnsbl.bit.nl rbl.efnet.org phishing.rbl.msrbl.net 0spam.fusionzero.com list.dsbl.org multihop.dsbl.org unconfirmed.dsbl.org will-spam-for-food.eu.org blacklist.spambag.org blackholes.brainerd.net blackholes.uceb.org spamsources.dnsbl.info map.spam-rbl.com ns1.unsubscore.com psbl.surriel.com l2.spews.dnsbl.sorbs.net bl.csma.biz sbl.csma.biz dynablock.njabl.org no-more-funn.moensted.dk  ubl.unsubscore.com dnsbl-1.uceprotect.net dnsbl-2.uceprotect.net dnsbl-3.uceprotect.net spamguard.leadmon.net opm.blitzed.org bl.spamcannibal.org rbl.schulte.org dnsbl.ahbl.org virbl.dnsbl.bit.nl combined.rbl.msrbl.net",
                );

                $serviceargs = grab_array_var($inargs, "serviceargs", $serviceargs_default);
            }

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;


        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // Don't revalidate if user is pressing back button
            $back = array_key_exists("backButton", $_POST);         
            if ($back) {
                break;
            }

            // Get variables that were passed to us
            $address_session = grab_array_var($_SESSION[$wizard_name], 'ip_address', "");
            $address = grab_array_var($inargs, "ip_address", $address_session);

            $hostname_session = grab_array_var($_SESSION[$wizard_name], 'hostname', "");
            $hostname = grab_array_var($inargs, "hostname", $hostname_session);
            $hostname = nagiosccm_replace_user_macros($hostname);

            $services = grab_array_var($_SESSION[$wizard_name], 'services', $services_default);
            $services = grab_array_var($inargs, 'services', $services);

            $serviceargs = grab_array_var($inargs, 'serviceargs');
            
            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (is_valid_host_name($hostname) == false) {
                $errmsg[$errors++] = "Invalid host name.";
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            $_SESSION[$wizard_name]["hostname"] = $hostname;
            $_SESSION[$wizard_name]["services"] = $services;
            $_SESSION[$wizard_name]["serviceargs"] = $serviceargs;

            break;


        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:

            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            // We will always have these in $_SESSION if we made it this far
            $address = $_SESSION[$wizard_name]["ip_address"];
            $hostname = $_SESSION[$wizard_name]["hostname"];
            $services = $_SESSION[$wizard_name]["services"];
            $serviceargs = $_SESSION[$wizard_name]["serviceargs"];

            // save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["hostname"] = $hostname;
            $meta_arr["ip_address"] = $address;
            $meta_arr["services"] = $services;
            $meta_arr["serviceargs"] = $serviceargs;
            save_configwizard_object_meta($wizard_name, $hostname, "", $meta_arr);

            $objs = array();

            if (!host_exists($hostname)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_mailserver_host",
                    "host_name" => $hostname,
                    "address" => $address,
                    "icon_image" => "mailserver.png",
                    "statusmap_image" => "mailserver.png",
                    "_xiwizard" => $wizard_name,
                );
            }

            // see which services we should monitor
            foreach ($services as $svc => $svcstate) {

                //echo "PROCESSING: $svc -> $svcstate<BR>\n";

                switch ($svc) {

                    case "ping":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Ping",
                            "use" => "xiwizard_mailserver_ping_service",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "imap":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "IMAP",
                            "use" => "xiwizard_imap_service",
                            "check_command" => "check_xi_service_imap!-j",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "pop":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "POP",
                            "use" => "xiwizard_pop_service",
                            "check_command" => "check_xi_service_pop!-j",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "smtp":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "SMTP",
                            "use" => "xiwizard_smtp_service",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "rbl":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Blacklist Status",
                            "use" => "xiwizard_mailserver_service",
                            "check_command" => "check_mailserver_rbl!-B " . $serviceargs["rbl_servers"],
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    default:
                        break;
                }
            }

            // echo "OBJECTS:<BR><pre>";
            // print_r($objs);
            // echo "</pre>";
            // exit();

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:
            break;
    }

    return $output;
}
