<?php
//
// Copyright (c) 2008-2020 Nagios Enterprises, LLC. All rights reserved.
//

require_once(dirname(__FILE__) . '/../../common.inc.php');


// Start session
init_session(false, false);

// Grab GET or POST variables and check pre-reqs
grab_request_vars();
check_prereqs(false);
check_authentication(false);


route_request();


function route_request()
{
    check_nagios_session_protector();

    $cmd = grab_request_var("cmd", "");

    switch ($cmd) {
        case "get_status":
            get_status();
            break;
        case "get_query_data":
            get_query_data();
            break;
        case "get_graph_data":
            get_graph_data();
            break;
        case "get_source_views":
            get_source_views();
            break;
        default:
            break;
    }
    exit;
}

function fetch_api_data($server, $token = '', $version = "1", $ssl = 0, $allow_invalid_ssl = 0, $endpoint = null) {
    $protocol = $ssl ? "https" : "http";
    $path = $version === "1" ? "/nagiosna/index.php/api/" : "/api/v1/";
    $url = $protocol . "://" . $server . $path . $endpoint;

    $options = [];

    if ($version === "1") {
        if (!empty($token)) {
            $url .= (strpos($url, '?') === false ? '?' : '&') . "token=" . urlencode($token);
        }
    } else {
        if (!empty($token)) {
            $options['http'] = [
                'header' => "Authorization: Bearer " . $token
            ];
        }
    }

    if ($allow_invalid_ssl == 1) {
        $options['ssl'] = [
            "verify_peer"      => false,
            "verify_peer_name" => false
        ];
    }

    $context = !empty($options) ? stream_context_create($options) : null;
    return file_get_contents($url, false, $context);
}

function get_status() {
    $server = grab_request_var("server", "");
    $token = grab_request_var("token", "");
    $version = grab_request_var("version", "1");
    $ssl = grab_request_var("ssl", 0);
    $allow_invalid_ssl = grab_request_var("allow_invalid_ssl", 0);

    // The 2026 endpoint uses mptstat for cpu which takes at least 1 second
    $endpoint = $version === "1" ? "system/cpu_status" : "system/status/memory";
    $response = fetch_api_data($server, $token, $version, $ssl, $allow_invalid_ssl, $endpoint);

    echo $response;
}

function get_query_data() {
    $server = grab_request_var("server", "");
    $token = grab_request_var("token", "");
    $version = grab_request_var("version", "1");
    $ssl = grab_request_var("ssl", 0);
    $allow_invalid_ssl = grab_request_var("allow_invalid_ssl", 0);
    $object = grab_request_var("object", "");

    if ($version === "1") {
        if ($object === "sources") {
            $endpoint = "sources/read";
        } else {
            $endpoint = "groups/read";
        }
    } else {
        if ($object === "sources") {
            $endpoint = "sources";
        } else {
            $endpoint = "source-groups";
        }
    }

    $response = fetch_api_data($server, $token, $version, $ssl, $allow_invalid_ssl, $endpoint);
    echo $response;
}

function get_source_views() {
    $server = grab_request_var("server", "");
    $token = grab_request_var("token", "");
    $version = grab_request_var("version", "1");
    $ssl = grab_request_var("ssl", 0);
    $allow_invalid_ssl = grab_request_var("allow_invalid_ssl", 0);
    $sid = grab_request_var("sid", '');

    $endpoint = "views/get_views?q[sid]=" . $sid;

    $response = fetch_api_data($server, $token, $version, $ssl, $allow_invalid_ssl, $endpoint);
    echo $response;
}

function get_graph_data() {
    $server = grab_request_var("server", "");
    $token = grab_request_var("token", "");
    $version = grab_request_var("version", "1");
    $ssl = grab_request_var("ssl", 0);
    $allow_invalid_ssl = grab_request_var("allow_invalid_ssl", 0);
    $object_type = grab_request_var("object_type", '');
    $object_id = grab_request_var("object_id", '');
    $use_views = grab_request_var("use_views", '');
    $view_id = grab_request_var("view_id", '');

    $vars = [
        'q[Bytes]' => 'bytes',
        'q[Flows]' => 'flows',
        'q[Packets]' => 'packets',
        'begindate' => '-1 week'
    ];

    $data_type = $object_type === 'sources' ? 'sid' : 'gid';
    $endpoint = "graphs/execute" . '?' . $data_type . '=' . $object_id;
    if ($use_views === 'on') {
        $endpoint.= '&vid=' . $view_id;
    }

    foreach ($vars as $k => $v) {
        $endpoint .= "&$k=" . urlencode($v);
    }

    $response = fetch_api_data($server, $token, $version, $ssl, $allow_invalid_ssl, $endpoint);
    echo $response;
}