#!/usr/bin/env php
<?php
//
// JSON Config Wizard
// Copyright (c) 2020 Nagios Enterprises, LLC. All rights reserved.
//

$longopts = array ("help");
$options = getopt("w:c:k:u:h:s:r:n:a:", $longopts);

$usage = "usage: php check_json.php -u <url> -k <key> [-s <string> -r <status_code> -n <status_code>] | [-c <critical_theshold> -w <warning_threshold>] [-h <header>] [--help]

    -u url
    -k json key
    -s treat key as a string
    -r status code if <string> matches <key> value (defaults to 0 when -s used)
    -n status code if <string> does not match <key> value (defaults to 2 when -s is used)
    -c critical threshold
    -w warning threshold
    -h additional header add to request. Can be specified multiple times.
    -a <USER>:<PASS> The username/password combination to use in HTTP authentication

    --help this help screen\n";

if (isset($options["help"])) {
    echo "$usage";
    exit(0);
}

// Check options
if (!isset($options['u'])) {
    echo "-u <URL> is required\n";
    exit(3);
}

// Set up headers
$headers = array('Accept: application/json');
if (isset($options["h"])) {
    settype($options['h'], 'array');
    $headers = array_merge($headers, $options['h']);
}

$ch = curl_init();
curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
curl_setopt($ch, CURLOPT_URL, $options["u"]);
curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);

if (isset($options['a'])) {
    curl_setopt_array($ch, [
        CURLOPT_HTTPAUTH => CURLAUTH_ANY,
        CURLOPT_USERPWD => $options['a'],
    ]);
}

$chData = curl_exec($ch);
$chDataArray = json_decode($chData, TRUE);

curl_close($ch);

// Error parsing JSON data
if (!$chDataArray) {
    echo "UNKNOWN - Could not parse JSON data\n";
    exit(3);
}

// Find the key
if (isset($options["k"])) {

    // Break up variable name so it displays properly
    $k = str_replace(array('[', ']'), array('.', ''), $options["k"]);
    $associative_keys = explode(".", $k);
    $current_arr = $chDataArray;

    // Check each key and continue
    foreach ($associative_keys as $key) {
        if (isset($current_arr[$key])) {
            $current_arr = $current_arr[$key];
        } else {
            break;
        }
    }

}

if (isset($options["w"])) {
    $warning = new Nagios_Plugin_Range($options["w"]);
}

if (isset($options["c"])) {
    $critical = new Nagios_Plugin_Range($options["c"]);
}

if (!isset($options["s"])) {

    // If we cannot find the array key... or value isn't numeric
    if (!is_numeric($current_arr)) {
        echo "UNKNOWN - Could not find the key or the value is not numeric\n";
        exit(3);
    }

    if ($critical->check_range($current_arr)) {
        echo "CRITICAL - " . $options["k"] . " = " . $current_arr . "|" . $options["k"] . "=" . $current_arr . "\n";
        exit(2);
    }

    if ($warning->check_range($current_arr)) {
        echo "WARNING - " . $options["k"] . " = " . $current_arr . "|" . $options["k"] . "=" . $current_arr . "\n";
        exit(1);
    }

    echo "OK - " . $options["k"] . " = " . $current_arr . "|" . $options["k"] . "=" . $current_arr . "\n";
    exit(0);

} else {

    $match_status_code = grab_array_var($options, "r", 0);
    $no_match_status_code = grab_array_var($options, "n", 2);

    $string = $options["s"];

    if ($string == $current_arr) {
        $status_code = intval($match_status_code);
        echo get_status_msg($status_code) . " - Value '" . $current_arr . "' matched '" . $string . "'\n";
    } else {
        $status_code = intval($no_match_status_code);
        echo get_status_msg($status_code) . " - Value '" . $current_arr . "' did not match '" . $string . "'\n";
    }

    exit($status_code);
}


function get_status_msg($status_code)
{
    $msg = "UNKNOWN";

    switch ($status_code)
    {
        case 0:
            $msg = "OK";
            break;
        case 1:
            $msg = "WARNING";
            break;
        case 2:
            $msg = "CRITICAL";
            break;
    }

    return $msg;
}


// Gets a value from an array, returning a default value if not found
function grab_array_var($a, $k, $d=null)
{
    return (is_array($a) && array_key_exists($k, $a)) ? $a[$k] : $d;
}


/*
 * Created on Sep 14, (c) 2006 by Marcel K�hn
 * GPL licensed, marcel@kuehns.org
 *
 * Ported nagios range checking from Nagios::Plugin::Range
 * by Doug Warner <doug@warner.fm>
 *
 * Cleaned up and improved Oct 12, (c) 2018 by Joe Momma
 */

define('STATE_OK',0);
define('STATE_WARNING',1);
define('STATE_CRITICAL',2);
define('STATE_UNKNOWN',3);

class Nagios_Plugin_Range {

    const INSIDE = 1;
    const OUTSIDE = 0;
    protected $range;
    protected $start = 0;
    protected $end = 0;
    protected $start_infinity = false;
    protected $end_infinity = false;
    protected $alert_on;

    public function __construct($range) {
        $this->range = $range;
        $this->parse_range_string($range);
    }

    public function valid() {
        return isset($this->alert_on);
    }

    public function check_range($value) {
        $false = false;
        $true = true;
        if ($this->alert_on == self::INSIDE) {
            $false = true;
            $true = false;
        }

        if (!$this->end_infinity && !$this->start_infinity ) {
            if ($this->start <= $value && $value <= $this->end) {
                return $false;
            } else {
                return $true;
            }
        }
        elseif (!$this->start_infinity && $this->end_infinity) {
            if ( $value >= $this->start ) {
                return $false;
            } else {
                return $true;
            }
        }
        elseif ($this->start_infinity && !$this->end_infinity) {
            if ($value <= $this->end) {
                return $false;
            } else {
                return $true;
            }
        }
        else {
            return $false;
        }
    }

    # Returns a N::P::Range object if the string is a conforms to a Nagios Plugin range string, otherwise null
    protected function parse_range_string($string) {
        $valid = 0;
        $alert_on = self::OUTSIDE;

        $string = preg_replace('/\s/', '', $string);

        $value = "[-+]?[\d\.]+";
        $value_re = "{$value}(?:e{$value})?";

        # check for valid range definition
        if ( !preg_match('/[\d~]/', $string)
            || !preg_match("/^\@?({$value_re}|~)?(:({$value_re})?)?$/", $string)) {
            echo "invalid range definition '$string'";
            exit(STATE_UNKNOWN);
        }

        if (preg_match('/^\@/', $string)) {
            $string = preg_replace('/^\@/', '', $string);
            $alert_on = self::INSIDE;
        }

        # '~:x'
        if (preg_match('/^~/', $string)) {
            $string = preg_replace('/^~/', '', $string);
            $this->start_infinity = true;
        }

        # '10:'
        if (preg_match("/^({$value_re})?:/", $string, $matches)) {
            if (!empty($matches[1])) {
                $this->set_range_start($matches[1]);
            }
            $this->end_infinity = true;  # overridden below if there's an end specified
            $string = preg_replace("/^({$value_re})?:/", '', $string);
            $valid++;
        }

        # 'x:10' or '10'
        if (preg_match("/^({$value_re})$/", $string)) {
            $this->set_range_end($string);
            $valid++;
        }

        if ($valid
            && ($this->start_infinity || $this->end_infinity || $this->start <= $this->end)) {
            $this->alert_on = $alert_on;
        }
    }

    protected function set_range_start($value) {
        $this->start = (float) $value;
        if (empty($this->start)) {
            $this->start = 0;
        }
        $this->start_infinity = false;
    }

    protected function set_range_end($value) {
        $this->end = (float) $value;
        if (empty($this->end)) {
            $this->end = 0;
        }
        $this->end_infinity = false;
    }

}
