#!/usr/bin/env python
"""
Nagios Network Analyzer Plugin

Check Network Analyzer sources, views, and sourcegroups for abnormal behavior.

"""

import sys
import optparse
import json
import urllib.request
import urllib.parse
import ssl
import time

# Keep this in line with the config wizard version
__VERSION__ = "2.1.0"

# Display a list of options to be sent to the plugin
def parse_args():
    parser = optparse.OptionParser()
    parser.add_option("-H", "--hostname", help="Nagios Network Analyzer server hostname")
    parser.add_option("-K", "--key", help="API key to use for authentication on the connecting Nagios Network Analyzer server")
    parser.add_option("--version", default="1", help="API version to use (1 or 2)")

    parser.add_option("-m", "--metric", help="Metric: bytes, flows, packets, or behavior (behavior only works on sources in legacy NNA)")
    parser.add_option("-S", "--source", help="Source ID or name (must use only one of source or sourcegroup)")
    parser.add_option("-G", "--sourcegroup", help="Sourcegroup ID or name (must use only one of source or sourcegroup)")
    parser.add_option("-v", "--view", help="View ID or name (legacy NNA only, used with source)")

    parser.add_option("-w", "--warning", default=None, type="int", help="Warning threshold")
    parser.add_option("-c", "--critical", default=None, type="int", help="Critical threshold")
    parser.add_option("--secure", action="store_true", default=False, help="Use HTTPS")
    parser.add_option("--ignorecert", action="store_true", default=False, help="Ignore invalid SSL certs")
    parser.add_option("--exists", default=None, action="store_true", help="Check if source/sourcegroup exists")
    parser.add_option("--verbose", action="store_true", default=False, help="Verbose output")
    parser.add_option("--noperfdata", action="store_true", default=False, help="Disable perfdata output")

    options, _ = parser.parse_args()

    # Validation
    if not options.hostname:
        parser.error("Hostname is required. Use --help for more info.")
    if not options.key:
        parser.error("API key is required. Use --help for more info.")
    if options.metric == "behavior":
        if not options.source or options.version == '2':
            parser.error("You must only use the Abnormal Behavior check on sources in legacy NNA. Use --help for more info.")
    if options.view:
        if not options.source or options.version == '2':
            parser.error("You must use a view only if you have a source selected in legacy NNA. Use --help for more info.")
    elif not options.source and not options.sourcegroup:
        parser.error("You must specify either --source or --sourcegroup.")
    if options.metric != "behavior":
        if not options.warning and not options.critical and not options.exists:
            parser.error("You must set warning and critical values. Use --help for more info.")
    if options.source and options.sourcegroup:
        parser.error("You may only specify one of --source or --sourcegroup.")
    if not options.metric and not options.exists:
        parser.error("You must set a metric to use. Use --help for more info.")

    return options

def fetch_data(endpoint, options, params=None):
    protocol = "https" if options.secure else "http"
    base_path = "/api/v1/" if options.version == "2" else "/nagiosna/index.php/api/"
    url = f"{protocol}://{options.hostname}{base_path}{endpoint}"

    if params:
        qs = urllib.parse.urlencode(params)
        url = f"{url}?{qs}"

    headers = {}
    if options.version == "2":
        headers["Authorization"] = f"Bearer {options.key}"
    else:
        if "token" not in url:
            token_sep = "&" if "?" in url else "?"
            url = f"{url}{token_sep}token={options.key}"

    req = urllib.request.Request(url, headers=headers)
    context = ssl._create_unverified_context() if options.ignorecert else None

    if options.verbose:
        print(f"[DEBUG] Requesting: {url}")

    try:
        with urllib.request.urlopen(req, context=context) as response:
            return json.load(response)
    except Exception as e:
        if options.verbose:
            print(f"[ERROR] Fetch failed: {e}")
        raise

def main(options):
    if options.source:
        id_type, id_val, text_type = "sid", options.source, "source"
    else:
        id_type, id_val, text_type = "gid", options.sourcegroup, "sourcegroup"

    if options.exists:
        endpoint = "sources/read" if (options.version == "1" and text_type == "source") else \
                   "groups/read" if (options.version == "1") else \
                   f"sources/{id_val}" if (text_type == "source") else f"source-groups/{id_val}"

        params = {"q[" + id_type + "]": id_val} if options.version == "1" else None
        data = fetch_data(endpoint, options, params)

        if not data:
            print(f"DOWN - The {text_type} does not exist")
            sys.exit(2)
        print(f"UP - The {text_type} exists")
        sys.exit(0)

    if options.metric == "behavior":
        endpoint = "graphs/failures"
        params = {"sid": id_val, "begindate": "-10 minutes", "enddate": "-1 second"}
        data = fetch_data(endpoint, options, params)

        if "error" in data:
            print(f"CRITICAL - {data['error']}")
            sys.exit(2)

        if data['data'][0]:
            print("CRITICAL - Abnormal behavior detected")
            sys.exit(2)
        print("OK - No abnormal behavior detected")
        sys.exit(0)

    endpoint = "graphs/execute"
    metric_map = {"bytes": "q[Bytes]", "flows": "q[Flows]", "packets": "q[Packets]"}
    metric_key = metric_map.get(options.metric.lower(), "q[Bytes]")

    params = {
        id_type: id_val,
        metric_key: options.metric,
        "begindate": "-10 minutes",
        "enddate": "-1 second",
    }

    data = fetch_data(endpoint, options, params)

    if isinstance(data, list) and data and data[0].get("total", 0) == 0:
        time.sleep(3)
        data = fetch_data(endpoint, options, params)

    if "error" in data:
        print(f"CRITICAL - {data['error']}")
        sys.exit(2)

    total = data[0].get("total", 0)
    check_thresholds(total, options.metric, options.warning, options.critical, options.noperfdata)

def check_thresholds(value, metric, warning, critical, noperfdata):
    perf = "" if noperfdata else f"|{metric}={value}"
    if critical and value >= critical:
        print(f"CRITICAL - {value} {metric} sent/received{perf}")
        sys.exit(2)
    elif warning and value >= warning:
        print(f"WARNING - {value} {metric} sent/received{perf}")
        sys.exit(1)
    else:
        print(f"OK - {value} {metric} sent/received{perf}")
        sys.exit(0)

if __name__ == "__main__":
    options = parse_args()
    try:
        main(options)
    except Exception as e:
        if options.verbose:
            print(f"ERROR: {e}")
        else:
            print("UNKNOWN - Error occurred while running the plugin.")
        sys.exit(3)
