<?php
//
// JSON Config Wizard
// Copyright (c) 2020-2024 Nagios Enterprises, LLC. All rights reserved.
//
// "Party... Party hard" -CD
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

json_configwizard_init();

function json_configwizard_init()
{
    $name = "json";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.0.2",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor JSON data output from a URL or API.") . '<div class="hide">REST API</div>',
        CONFIGWIZARD_DISPLAYTITLE => _("JSON"),
        CONFIGWIZARD_FUNCTION => "json_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "json.png",
        CONFIGWIZARD_COPYRIGHT => "Copyright &copy; 2020-2024 Nagios Enterprises, LLC.",
        CONFIGWIZARD_AUTHOR => "Nagios Enterprises, LLC",
        CONFIGWIZARD_FILTER_GROUPS => array('website', 'xml', 'api'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );
    register_configwizard($name, $args);
}

/**
 * @param string $mode
 * @param        $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function json_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    $wizard_name = "json";

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            $address = grab_array_var($inargs, "address", "");
            $port = grab_array_var($inargs, "port", "");

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "address", "");
            $address = nagiosccm_replace_user_macros($address);

            $port = grab_array_var($inargs, "port", "");

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (empty($address)) {
                $errmsg[$errors++] = _("No address specified.");
            }

            if (!empty($port) && !is_numeric($port)) {
                $errmsg[$errors++] = _("Invalid port number.");
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "address");
            $port = grab_array_var($inargs, "port", "");
            $ha = @gethostbyaddr($address);

            if ($ha == "") {
                $ha = $address;
            }

            $hostname = grab_array_var($inargs, "hostname", $ha);

            $services = grab_array_var($inargs, "services", array());
            $services_serial = grab_array_var($inargs, "services_serial");

            if ($services_serial != "") {
                $services = json_decode(base64_decode($services_serial), true);
            }

            // Integer checks
            if (!array_key_exists("integer_check", $services)) {
                $services["integer_check"] = array();
            }

            for ($x = 0; $x < 1; $x++) {
                if (!array_key_exists($x, $services["integer_check"]))
                    $services["integer_check"][$x] = array();
                if (!array_key_exists("url", $services["integer_check"][$x]))
                    $services["integer_check"][$x]["url"] = "";
                if (!array_key_exists("key", $services["integer_check"][$x]))
                    $services["integer_check"][$x]["key"] = "";
                if (!array_key_exists("warning", $services["integer_check"][$x]))
                    $services["integer_check"][$x]["warning"] = "";
                if (!array_key_exists("critical", $services["integer_check"][$x]))
                    $services["integer_check"][$x]["critical"] = "";
                if (!array_key_exists("header", $services["integer_check"][$x]))
                    $services["integer_check"][$x]["header"] = "";
            }

            // String checks
            if (!array_key_exists("string_check", $services)) {
                $services["string_check"] = array();
            }

            for ($x = 0; $x < 1; $x++) {
                if (!array_key_exists($x, $services["string_check"]))
                    $services["string_check"][$x] = array();
                if (!array_key_exists("url", $services["string_check"][$x]))
                    $services["string_check"][$x]["url"] = "";
                if (!array_key_exists("key", $services["string_check"][$x]))
                    $services["string_check"][$x]["key"] = "";
                if (!array_key_exists("string", $services["string_check"][$x]))
                    $services["string_check"][$x]["string"] = "";
                if (!array_key_exists("result_code_if_match", $services["string_check"][$x]))
                    $services["string_check"][$x]["result_code_if_match"] = 0;
                if (!array_key_exists("result_code_if_no_match", $services["string_check"][$x]))
                    $services["string_check"][$x]["result_code_if_no_match"] = 2;
            }

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "address");
            $hostname = grab_array_var($inargs, "hostname");
            $hostname = nagiosccm_replace_user_macros($hostname);

            $services = grab_array_var($inargs, "services");
            $port = grab_array_var($inargs, "port", "");

            // check for errors
            $errors = 0;
            $integer_errors = 0;
            $string_errors = 0;

            $errmsg = array();
            $integer_errmsg = array();
            $string_errmsg = array();

            if (is_valid_host_name($hostname) == false) {
                $errmsg[$errors++] = _("Invalid host name.");
            }

            // integer_check_

            foreach ($services["integer_check"] as $id => $portarr) {

                $url = grab_array_var($portarr, "url", "");
                $key = grab_array_var($portarr, "key", "");
                $warning = grab_array_var($portarr, "warning", "");
                $critical = grab_array_var($portarr, "critical", "");

                if ( isset($url) || isset($key) || isset($warning) || isset($critical) || isset($header) ) {
                    if ( !isset($url) || !isset($key) || !isset($warning) || !isset($critical) ){
                        $integer_errmsg[$integer_errors++] = _("Invalid integer service check configured.");
                    } else {
                        $integer_config = 1;
                    }
                }
            }


            // string_check_

            foreach ($services["string_check"] as $id => $portarr) {

                $url = grab_array_var($portarr, "url", "");
                $key = grab_array_var($portarr, "key", "");
                $string = grab_array_var($portarr, "string", "");
                $result_code_if_match = grab_array_var($portarr, "result_code_if_match", "");
                $result_code_if_no_match = grab_array_var($portarr, "result_code_if_no_match", "");

                if ( isset($url) || isset($key) || isset($string) || isset($result_code_if_match) || isset($result_code_if_no_match) || isset($header) ) {
                    if ( !isset($url) || !isset($key) || !isset($string) || !isset($result_code_if_match) || !isset($result_code_if_no_match) ){
                        $string_errmsg[$string_errors++] = _("Invalid string service check configured");
                    } else {
                        $string_config = 1;
                    }
                }
            }

            if ($integer_config == NULL && $string_config == NULL ) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = _("Must configure at least one service check");
                $result = 1;
            }

            if ($errors > 0 || $integer_errors > 0|| $string_errors > 0 ) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = array_merge($errmsg, $integer_errmsg, $string_errmsg);
                $result = 1;
            }

            break;


        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "address");
            $hostname = grab_array_var($inargs, "hostname");
            $services = grab_array_var($inargs, "services");
            $serviceargs = grab_array_var($inargs, "serviceargs");
            $port = grab_array_var($inargs, "port", "");

            $services_serial = grab_array_var($inargs, "services_serial", base64_encode(json_encode($services)));
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", base64_encode(json_encode($serviceargs)));

            $output = '

        <input type="hidden" name="address" value="' . encode_form_val($address) . '">
        <input type="hidden" name="hostname" value="' . encode_form_val($hostname) . '">
        <input type="hidden" name="services_serial" value="' . $services_serial . '">
        <input type="hidden" name="serviceargs_serial" value="' . $serviceargs_serial . '">
        <input type="hidden" name="port" value="' . encode_form_val($port) . '">
';
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:

            $output = '

            ';
            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            $address = grab_array_var($inargs, "address", "");
            $hostname = grab_array_var($inargs, "hostname", "");
            $port = grab_array_var($inargs, "port", "");

            $services_serial = grab_array_var($inargs, "services_serial", "");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            $services = json_decode(base64_decode($services_serial), true);
            $serviceargs = json_decode(base64_decode($serviceargs_serial), true);

            $hostaddress = $address;

            // save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["hostname"] = $hostname;
            $meta_arr["address"] = $address;
            $meta_arr["services"] = $services;
            $meta_arr["serivceargs"] = $serviceargs;
            $meta_arr["port"] = $port;

            save_configwizard_object_meta($wizard_name, $hostname, "", $meta_arr);

            $objs = array();

            if (!host_exists($hostname)) {

                // Choose the port for check_tcp
                $tmp_port = $port;
                if ($port == "") {
                    $tmp_port = "80";
                }

                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_json_host",
                    "host_name" => $hostname,
                    "address" => $hostaddress,
                    "icon_image" => "json.png",
                    "statusmap_image" => "json.png",
                    "check_command" => "check_tcp!$tmp_port!",
                    "_xiwizard" => $wizard_name,
                );
            }

            // configure integer checks
            if (array_key_exists("integer_check", $services)) {
                foreach ($services["integer_check"] as $id => $svc_arr) {

                    $url = grab_array_var($svc_arr, "url", "");
                    $key = grab_array_var($svc_arr, "key", "");
                    $desc = grab_array_var($svc_arr, "desc", $key);
                    $warning = grab_array_var($svc_arr, "warning", "");
                    $critical = grab_array_var($svc_arr, "critical", "");
                    $header = grab_array_var($svc_arr, "header", "");

                    if (empty($desc) || empty($url)) {
                        continue;
                    }

                    $args = "-u " . escapeshellarg($url) . " -k " . escapeshellarg($key);
                    if (!empty($header)) {
                        $args .= " -h " . escapeshellarg($header);
                    }

                    $args .= " -w " . intval($warning) . " -c " . intval($critical);

                    $use = "xiwizard_json_service";
                    $check_command = "check_json!" . $args;

                    $objs[] = array(
                        "type" => OBJECTTYPE_SERVICE,
                        "host_name" => $hostname,
                        "service_description" => $desc,
                        "use" => $use,
                        "check_command" => $check_command,
                        "_xiwizard" => $wizard_name,
                    );
                }
            }

            // configure string checks
            if (array_key_exists("string_check", $services)) {
                foreach ($services["string_check"] as $id => $svc_arr) {

                    $url = grab_array_var($svc_arr, "url", "");
                    $key = grab_array_var($svc_arr, "key", "");
                    $desc = grab_array_var($svc_arr, "desc", $key);
                    $string = grab_array_var($svc_arr, "string", "");
                    $result_code_if_match = grab_array_var($svc_arr, "result_code_if_match", 0);
                    $result_code_if_no_match = grab_array_var($svc_arr, "result_code_if_no_match", 0);
                    $header = grab_array_var($svc_arr, "header", "");

                    if (empty($desc) || empty($url)) {
                        continue;
                    }

                    $args = "-u " . escapeshellarg($url) . " -k " . escapeshellarg($key);
                    if (!empty($header)) {
                        $args .= " -h " . escapeshellarg($header);
                    }

                    $args .= " -s " . escapeshellarg($string) . " -r " . intval($result_code_if_match) . " -n " . intval($result_code_if_no_match);

                    $use = "xiwizard_json_service";
                    $check_command = "check_json!" . $args;

                    $objs[] = array(
                        "type" => OBJECTTYPE_SERVICE,
                        "host_name" => $hostname,
                        "service_description" => $desc,
                        "use" => $use,
                        "check_command" => $check_command,
                        "_xiwizard" => $wizard_name,
                    );
                }
            }

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:
            break;
    }

    return $output;
}
