<?php
//
// SLA Config Wizard
// Copyright (c) 2016-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__).'/../configwizardhelper.inc.php');

// Run the initialization function
sla_configwizard_init();

function sla_configwizard_init()
{
    $name = "sla";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.0.2",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor host and service Service Level Agreements (SLA) to ensure they are met."),
        CONFIGWIZARD_DISPLAYTITLE => _("SLA"),
        CONFIGWIZARD_FUNCTION => "sla_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "sla.png",
        CONFIGWIZARD_FILTER_GROUPS => array('nagios'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );
    register_configwizard($name,$args);
}

/**
 * Check pre-reqs for configuration wizard
 *
 * @return  array   Array of errors
 */
function sla_configwizard_check_prereqs()
{
    $errors = array();

    if (!file_exists("/usr/local/nagios/libexec/check_xi_sla.php")) {
        $errors[] = _('It looks like you are missing check_xi_sla.php on your Nagios XI server. To use this wizard you must install the check_xi_sla.php plugin on your server located in the this wizards plugin directory here: /usr/local/nagios/libexec/');
    }

    return $errors;
}

/**
 * @param   string  $mode
 * @param   null    $inargs
 * @param           $outargs
 * @param           $result
 * @return  string
 */
function sla_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    global $db_tables;
    $wizard_name = "sla";

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {

        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            $reportperiod = grab_array_var($inargs, "reportperiod", "24h");
            $custom_reportperiod = grab_array_var($inargs, "custom_reportperiod", "");
            $custom_reportperiod_unit = grab_array_var($inargs, "custom_reportperiod_unit", "h");
            $advanced = grab_array_var($inargs, "advanced", array());
            $errors = sla_configwizard_check_prereqs();
            $selectedhostconfig = grab_array_var($inargs, "selectedhostconfig", "");
            $operation = grab_array_var($inargs, "operation", "new");

            if (!is_array($advanced)) {
                $advanced = json_decode(base64_decode($advanced), true);
            }

            if ($errors) {
                $services = grab_array_var($inargs, "services", array());
                $serviceargs = grab_array_var($inargs, "serviceargs", array());

                $output .= '<div class="message"><ul class="errorMessage">';

                foreach($errors as $error) {
                    $output .= "<li><p>$error</p></li>";
                }

                $output .= '</ul></div>';

            } else {
                # Get the existing host/node configurations.
                # TODO: Include passwords/secrets?
                $nodes = get_configwizard_hosts($wizard_name);

                ########################################################################################
                # Load the html
                # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
                #   to load the PHP from the Step1 file into the $output string.
                ########################################################################################
                ob_start();
                include __DIR__.'/steps/step1.php';
                $output = ob_get_clean();
            }

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            // Get variables that were passed to us
            $reportperiod = grab_array_var($inargs, "reportperiod", "24h");
            $custom_reportperiod = grab_array_var($inargs, "custom_reportperiod", "");
            $custom_reportperiod_unit = grab_array_var($inargs, "custom_reportperiod_unit", "");
            $advanced = grab_array_var($inargs, "advanced", array());
            $selectedhostconfig = grab_array_var($inargs, "selectedhostconfig", "");
            $operation = grab_array_var($inargs, "operation", "new");

            $services = grab_array_var($inargs, "services", "");
            $serviceargs = grab_array_var($inargs, "serviceargs", "");
            $services_serial = grab_array_var($inargs, 'services_serial');
            $serviceargs_serial = grab_array_var($inargs, 'serviceargs_serial');

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            $services = grab_array_var($inargs, "services", "");
            $serviceargs = grab_array_var($inargs, "serviceargs", "");
            $services_serial = grab_array_var($inargs, 'services_serial');
            $serviceargs_serial = grab_array_var($inargs, 'serviceargs_serial');

            // get variables that were passed to us
            $username = grab_array_var($inargs, "username", "");
            $hostname = grab_array_var($inargs, "hostname", _("SLA"));
            $reportperiod = grab_array_var($inargs, "reportperiod", "24h");
            $custom_reportperiod = grab_array_var($inargs, "custom_reportperiod", "");
            $custom_reportperiod_unit = grab_array_var($inargs, "custom_reportperiod_unit", "");
            $advanced = grab_array_var($inargs, "advanced", array());
            $warning = grab_array_var($inargs, "warning", "");
            $critical = grab_array_var($inargs, "critical", "");
            $selectedhostconfig = grab_array_var($inargs, "selectedhostconfig", "");
            $operation = grab_array_var($inargs, "operation", "new");

            if (!is_array($advanced)) {
                $advanced = json_decode(base64_decode($advanced), true);
            }

            // Get display report period
            $displayreportperiod = _("Last 24 Hours");

            if ($reportperiod == '7d') {
                $displayreportperiod = _("Last 7 Days");
            } else if ($reportperiod == '1M') {
                $displayreportperiod = _("Last Month");
            } else if ($reportperiod == 'custom') {
                $displayreportperiod = _("Last").' '.$custom_reportperiod;
                if ($custom_reportperiod_unit == "m") {
                    $displayreportperiod .= " "._("Minutes");
                } else if ($custom_reportperiod_unit == "h") {
                    $displayreportperiod .= " "._("Hours");
                } else if ($custom_reportperiod_unit == "d") {
                    $displayreportperiod .= " "._("Days");
                } else {
                    $displayreportperiod .= " "._("Months");
                }
            }

            // Initialize container vars for form
            $host = "";
            $service = "";
            $hostgroup = "";
            $servicegroup = "";

            // If no serialized data, use current request data
            $services = "";
            $services_serial = grab_array_var($inargs, "services_serial", "");

            if (!empty($services_serial)) {
                $services = json_decode(base64_decode($services_serial), true);
            }

            $serviceargs = "";
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            if (!empty($serviceargs_serial)) {
                $serviceargs = json_decode(base64_decode($serviceargs_serial), true);
            }

            // Check if retention data available
            // NOTE: If nothing is checked, $services will come back empty, 
            //       while $serviceargs has data.
            if (!is_array($services)) {
                // Initialize services array variables
                $services_default = grab_array_var($inargs, "services", array(
                    "host" => array(),
                    "service" => array(),
                    "hostgroup" => array(),
                    "servicegroup" => array()
                ));

                $services = grab_array_var($inargs, "services", $services_default);
                $services_serial = base64_encode(json_encode($services));

                // Initialize serviceargs array variables
                $serviceargs_default = array(
                    "host" => array(),
                    "service" => array(),
                    "hostgroup" => array(),
                    "servicegroup" => array(),
                    "advanced" => array(
                        "assumeinitialstates" => "yes",
                        "assumestateretention" => "yes",
                        "assumestatesduringdowntime" => "yes",
                        "includesoftstates" => "no",
                        "asssumehs" => 3,
                        "asssumess" => 6
                    )
                );

                if (!is_array($serviceargs)) {
                    $serviceargs = grab_array_var($inargs, "serviceargs", $serviceargs_default);

                    foreach ($services as $svc => $val) {
                        for ($x = 0; $x < 2; $x++) {
                            if (!array_key_exists($x, $services))
                                $services[$svc][$x] = 'off';

                            if (!array_key_exists($x, $services)) {
                                if (strcmp($x, 'service') !== 0) {
                                    $serviceargs[$svc][$x] = array(
                                        $svc => $svc,
                                        "username" => $username,
                                        "host" => "",
                                        "warning" => "",
                                        "critical" => ""
                                    );
                                } else {
                                    $serviceargs[$svc][$x] = array(
                                        $svc => $svc,
                                        "username" => $username,
                                        "warning" => "",
                                        "critical" => ""
                                    );

                                }
                            }
                        }
                    }
                }
            }

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // get variables that were passed to us
            $hostname = grab_array_var($inargs, "hostname", "");
            $username = grab_array_var($inargs, "username", "");
            $warning = grab_array_var($inargs, "warning", "");
            $critical = grab_array_var($inargs, "critical", "");
            $advanced = grab_array_var($inargs, "advanced", "");
            $services = grab_array_var($inargs, "services", "");
            $serviceargs = grab_array_var($inargs, "serviceargs", "");

            // Use encoded data if user came back from future screen.
            $services_serial = grab_array_var($inargs, 'services_serial');

            // Preferentialy use form data, because there could be data changes.
            // If no form data, use serialized data, if available.
            if (!$services && $services_serial) {
                $services = json_decode(base64_decode($services_serial), true);
            }

            $serviceargs_serial = grab_array_var($inargs, 'serviceargs_serial');

            // Preferentialy use form data, because there could be data changes.
            // If no form data, use serialized data, if available.
            if (!$serviceargs && $serviceargs_serial) {
                $serviceargs = json_decode(base64_decode($serviceargs_serial), true);
            }

            // If no serialized data, use current request data if available.
            if (!$services) {
                $services = grab_array_var($inargs, 'services');
            }

            if (!$serviceargs) {
                $serviceargs = grab_array_var($inargs, 'serviceargs');
            }

            // check for errors
            $errors = 0;
            $errmsg = array();

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE3OPTS:

            $hostname = grab_array_var($inargs, "hostname", "SLA");
            $username = grab_array_var($inargs, "username", "");
            $reportperiod = grab_array_var($inargs, "reportperiod", "24h");
            $custom_reportperiod = grab_array_var($inargs, "custom_reportperiod", "");
            $custom_reportperiod_unit = grab_array_var($inargs, "custom_reportperiod_unit", "");
            $warning = grab_array_var($inargs, "warning");
            $critical = grab_array_var($inargs, "critical");
            $advanced = grab_array_var($inargs, "advanced", array());

            $check_interval = 24 * 60;

            if ($reportperiod == 'custom') {
                $check_interval = get_minutes_from_reportperiod($custom_reportperiod, $custom_reportperiod_unit);
            }

            $outargs[CONFIGWIZARD_OVERRIDE_OPTIONS]["check_interval"] = $check_interval;
            $outargs[CONFIGWIZARD_OVERRIDE_OPTIONS]["retry_interval"] = floor($check_interval / 5);

            break;

        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            $hostname = grab_array_var($inargs, "hostname", "SLA");
            $username = grab_array_var($inargs, "username", "");
            $reportperiod = grab_array_var($inargs, "reportperiod", "24h");
            $custom_reportperiod = grab_array_var($inargs, "custom_reportperiod", "");
            $custom_reportperiod_unit = grab_array_var($inargs, "custom_reportperiod_unit", "");
            $warning = grab_array_var($inargs, "warning");
            $critical = grab_array_var($inargs, "critical");
            $advanced = grab_array_var($inargs, "advanced", array());
            $services = grab_array_var($inargs, "services");
            $serviceargs = grab_array_var($inargs, "serviceargs");
            $services_serial = grab_array_var($inargs, "services_serial", base64_encode(json_encode($services)));
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", base64_encode(json_encode($serviceargs)));

            #
            # Data passed back from "next" steps.
            #
            $passbackdata = grab_array_var($inargs, "passbackdata", "");

            if ($passbackdata) {
                $decodedpassbackdata = json_decode(base64_decode($passbackdata), true);

                $hostname = $decodedpassbackdata['hostname'];
                $username = $decodedpassbackdata['username'];
                # Not currently passed back
                #$reportperiod = grab_array_var($inargs, "reportperiod", "24h");
                #$custom_reportperiod = grab_array_var($inargs, "custom_reportperiod", "");
                #$custom_reportperiod_unit = grab_array_var($inargs, "custom_reportperiod_unit", "");
                $warning = $decodedpassbackdata['warning'];
                $critical = $decodedpassbackdata['critical'];
                $advanced = $decodedpassbackdata['advanced'];
                $services_serial = $decodedpassbackdata['services_serial'];
                $serviceargs_serial = $decodedpassbackdata['serviceargs_serial'];
            }

            $output = '
                <input type="hidden" name="hostname" value="' . encode_form_val($hostname) . '" />
                <input type="hidden" name="username" value="' . encode_form_val($username) . '" />
                <input type="hidden" name="warning" value="' . intval($warning) . '" />
                <input type="hidden" name="critical" value="' . intval($critical) . '" />
                <input type="hidden" name="advanced" value="' . encode_form_val($advanced) . '" />
                <input type="hidden" name="services_serial" value="' . $services_serial . '">
                <input type="hidden" name="serviceargs_serial" value="' . $serviceargs_serial . '">
            ';
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:

            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:
            $hostname = grab_array_var($inargs, "hostname", "SLA");
            $reportperiod = grab_array_var($inargs, "reportperiod", "24h");
            $custom_reportperiod = grab_array_var($inargs, "custom_reportperiod", "");
            $custom_reportperiod_unit = grab_array_var($inargs, "custom_reportperiod_unit", "");
            $warning = grab_array_var($inargs, "warning");
            $critical = grab_array_var($inargs, "critical");
            $advanced = grab_array_var($inargs, "advanced", array());
            $services = grab_array_var($inargs, "services", array());
            $serviceargs = grab_array_var($inargs, "serviceargs", array());
            $services_serial = grab_array_var($inargs, "services_serial", "");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            $advanced = json_decode(base64_decode($advanced), true);
            $services = json_decode(base64_decode($services_serial), true);
            $serviceargs = json_decode(base64_decode($serviceargs_serial), true);

            foreach ($advanced as $k => $v) {
                $advancedoptions[] = $k."=".$v;
            }
            $advancedoptions = implode(',', $advancedoptions);

            // save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["hostname"] = $hostname;
            $meta_arr["reportperiod"] = $reportperiod;
            $meta_arr["custom_reportperiod"] = $custom_reportperiod;
            $meta_arr["custom_reportperiod_unit"] = $custom_reportperiod_unit;
            $meta_arr["warning"] = $warning;
            $meta_arr["critical"] = $critical;
            $meta_arr["advanced"] = $advanced;
            $meta_arr["services"] = $services;
            $meta_arr["serviceargs"] = $serviceargs;
            save_configwizard_object_meta($wizard_name, "127.0.0.1", "", $meta_arr);

            // Construct service check command variable list
            $argvars = "";

            $objs = array();

            if (!host_exists($hostname)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_generic_host",
                    "host_name" => $hostname,
                    "address" => "127.0.0.1",
                    "icon_image" => "sla.png",
                    "statusmap_image" => "sla.png",
                    "_xiwizard" => $wizard_name,
                );
            }

            // see which services we should monitor
            foreach ($services as $svc => $svcstate) {

                switch ($svc) {

                    case "host":
                        foreach ($svcstate as $i => $v) {
                            // create service for each on checkbox
                            if ($v != "on")
                                continue;

                            $argvars = " -u '".$_SESSION['username']."'";
                            $argvars .= " -h " . escapeshellarg($serviceargs['host'][$i]['host']);
                            $argvars .= " -w @" . $serviceargs['host'][$i]['warning'];
                            $argvars .= " -c @" . $serviceargs['host'][$i]['critical'];
                            $argvars .= " -a '" . $advancedoptions . "'";

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $hostname,
                                "use" => "xiwizard_check_sla2",
                                "service_description" => "SLA for " . $serviceargs['host'][$i]['host'],
                                "check_command" => "check_xi_sla2!" . $argvars,
                                "_xiwizard" => $wizard_name,
                            );
                        }
                        break;

                    case "service":
                        foreach ($svcstate as $i => $v) {
                            // create service for each on checkbox
                            if ($v != "on")
                                continue;

                            $service = $serviceargs['service'][$i]['service'];
                            $service_description = $service;

                            $argvars = " -u '".$_SESSION['username']."'";
                            $argvars .= " -h " . escapeshellarg($serviceargs['service'][$i]['host']);

                            if (!empty($service)) {
                                $argvars .= " -s " . escapeshellarg($service);
                            } else {
                                $argvars .= " -A ";
                                $service_description = _("all service average");
                            }

                            $argvars .= " -w @" . $serviceargs['service'][$i]['warning'];
                            $argvars .= " -c @" . $serviceargs['service'][$i]['critical'];
                            $argvars .= " -a '" . $advancedoptions . "'";

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $hostname,
                                "use" => "xiwizard_check_sla2",
                                "service_description" => "SLA for " . $serviceargs['service'][$i]['host'] . ' - ' . $service_description,
                                "check_command" => "check_xi_sla2!" . $argvars,
                                "_xiwizard" => $wizard_name,
                            );
                        }
                        break;

                    case "hostgroup":
                        foreach ($svcstate as $i => $v) {
                            // create service for each on checkbox
                            if ($v != "on")
                                continue;

                            $argvars = " -u '".$_SESSION['username']."'";
                            $argvars .= " -g " . escapeshellarg($serviceargs['hostgroup'][$i]['hostgroup']);
                            $argvars .= " -w @" . $serviceargs['hostgroup'][$i]['warning'];
                            $argvars .= " -c @" . $serviceargs['hostgroup'][$i]['critical'];
                            $argvars .= " -a '" . $advancedoptions . "'";

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $hostname,
                                "use" => "xiwizard_check_sla2",
                                "service_description" => "SLA for hostgroup " . $serviceargs['hostgroup'][$i]['hostgroup'],
                                "check_command" => "check_xi_sla2!" . $argvars,
                                "_xiwizard" => $wizard_name,
                            );
                        }
                        break;

                    case "servicegroup":
                        foreach ($svcstate as $i => $v) {
                            // create service for each on checkbox
                            if ($v != "on")
                                continue;

                            $argvars = " -u '".$_SESSION['username']."'";
                            $argvars .= " -e " . escapeshellarg($serviceargs['servicegroup'][$i]['servicegroup']);
                            $argvars .= " -w @" . $serviceargs['servicegroup'][$i]['warning'];
                            $argvars .= " -c @" . $serviceargs['servicegroup'][$i]['critical'];
                            $argvars .= " -a '" . $advancedoptions . "'";

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $hostname,
                                "use" => "xiwizard_check_sla",
                                "service_description" => "SLA for servicegroup " . $serviceargs['servicegroup'][$i]['servicegroup'],
                                "check_command" => "check_xi_sla2!" . $argvars,
                                "_xiwizard" => $wizard_name,
                            );
                        }
                        break;
                }
            }

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:
            break;
        }

    return $output;
}

function get_minutes_from_reportperiod($amount, $unit)
{
    switch ($unit)
    {
        case 'm':
            $mins = $amount;
            break;

        case 'h':
            $mins = $amount * 60;
            break;

        case 'd':
            $mins = $amount * (24 * 60);
            break;

        case 'M':
            $mins = $amount * (30 * 24 * 60);
            break;
    }

    return $mins;
}
