<?php
//
// Solaris Config Wizard
// Copyright (c) 2011-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

solaris_configwizard_init();

function solaris_configwizard_init()
{
    $name = "solaris";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.0.1",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor a Solaris server."),
        CONFIGWIZARD_DISPLAYTITLE => "Solaris",
        CONFIGWIZARD_FUNCTION => "solaris_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "solaris.png",
        CONFIGWIZARD_FILTER_GROUPS => array('otheros', 'server'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );
    register_configwizard($name, $args);
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function solaris_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    $wizard_name = "solaris";
    $agent_url = "https://assets.nagios.com/downloads/nagiosxi/agents/solaris-nrpe-agent.tar.gz";
    $agent_doc_url = "https://assets.nagios.com/downloads/nagiosxi/docs/Installing_The_XI_Solaris_Agent.pdf ";

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            $ip_address = grab_array_var($inargs, "ip_address", "");
            $agent = grab_array_var($inargs, "agent", "");

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            // Get variables that were passed to us
            $ip_address = grab_array_var($inargs, "ip_address", "");
            $ip_address = nagiosccm_replace_user_macros($ip_address);
            $agent = grab_array_var($inargs, "agent", "");

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (have_value($ip_address) == false) {
                $errmsg[$errors++] = _("No address specified.");
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            // Redirect to the NCPA wizard
            if ($agent == 'ncpa') {
                $nsp = get_nagios_session_protector_id();
                header("Location: monitoringwizard.php?update=1&nextstep=2&wizard=ncpa&os=sol&ip_address=$ip_address&nsp=$nsp");
                die();
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us
            $ip_address = grab_array_var($inargs, "ip_address");
            $agent = grab_array_var($inargs, "agent", "");
            $ha = @gethostbyaddr($ip_address);

            if ($ha == "") {
                $ha = $ip_address;
            }

            $hostname = grab_array_var($inargs, "hostname", $ha);
            $password = "";
            $services = "";
            $services_serial = grab_array_var($inargs, "services_serial", "");

            if ($services_serial != "") {
                $services = json_decode(base64_decode($services_serial), true);
            }

            if (!is_array($services)) {
                $services_default = array(
                    "ping" => 1,
                    "load" => 1,
                    "cpustats" => 1,
                    "memory" => 1,
                    "swap" => 1,
                    "openfiles" => 1,
                    "users" => 1,
                    "procs" => 1,
                    "disk" => 1,
                    "servicestate" => array(),
                    "processstate" => array(),
                );

                $services_default["servicestate"][0] = "on";
                $services_default["servicestate"][1] = "on";
                $services = grab_array_var($inargs, "services", $services_default);
            }

            $serviceargs = "";
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            if ($serviceargs_serial != "") {
                $serviceargs = json_decode(base64_decode($serviceargs_serial),  true);
            }

            if (!is_array($serviceargs)) {
                $serviceargs_default = array(

                    "memory_warning" => 15,
                    "memory_critical" => 5,

                    "load_warning" => "15,10,5",
                    "load_critical" => "30,20,10",

                    "cpustats_warning" => "70,40,30",
                    "cpustats_critical" => "90,60,40",

                    "openfiles_warning" => 3000,
                    "openfiles_critical" => 5000,

                    "swap_warning" => 5,
                    "swap_critical" => 10,

                    "users_warning" => 5,
                    "users_critical" => 10,

                    "procs_warning" => 150,
                    "procs_critical" => 250,

                    "processstate" => array(),
                    "servicestate" => array(),
                    "counter" => array(),
                );

                for ($x = 0; $x < 5; $x++) {
                    $serviceargs_default["disk_warning"][$x] = 20;
                    $serviceargs_default["disk_critical"][$x] = 10;
                    $serviceargs_default["disk"][$x] = ($x == 0) ? "/" : "";
                }

                for ($x = 0; $x < 4; $x++) {
                    if ($x == 0) {
                        $serviceargs_default['processstate'][$x]['process'] = 'sendmail';
                        $serviceargs_default['processstate'][$x]['name'] = 'Sendmail';
                        $services['processstate'][$x] = 'on';
                    } else {
                        $serviceargs_default['processstate'][$x]['process'] = '';
                        $serviceargs_default['processstate'][$x]['name'] = '';
                        $services['processstate'][$x] = 'off';
                    }
                }

                for ($x = 0; $x < 5; $x++) {
                    if ($x == 0) {
                        $serviceargs_default['servicestate'][$x]['service'] = "ssh";
                        $serviceargs_default['servicestate'][$x]['name'] = "SSH Server";
                    } else if ($x == 1) {
                        $serviceargs_default['servicestate'][$x]['service'] = "cron";
                        $serviceargs_default['servicestate'][$x]['name'] = "Cron Scheduling Daemon";
                    } else if ($x == 2) {
                        $serviceargs_default['servicestate'][$x]['service'] = "system-log";
                        $serviceargs_default['servicestate'][$x]['name'] = "System Logging Daemon";
                        $services["servicestate"][$x] = "off";
                    } else if ($x == 3) {
                        $serviceargs_default['servicestate'][$x]['service'] = "httpd";
                        $serviceargs_default['servicestate'][$x]['name'] = "Apache Web Server";
                        $services["servicestate"][$x] = "off";
                    } else if ($x == 4) {
                        $serviceargs_default['servicestate'][$x]['service'] = "smtp";
                        $serviceargs_default['servicestate'][$x]['name'] = "Sendmail Mail Transfer Agent";
                        $services["servicestate"][$x] = "off";
                    } else {
                        $serviceargs_default['servicestate'][$x]['service'] = "";
                        $serviceargs_default['servicestate'][$x]['name'] = "";
                        $services["servicestate"][$x] = "off";
                    }
                }

                $serviceargs = grab_array_var($inargs, "serviceargs", $serviceargs_default);
            }

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // Get variables that were passed to us
            $ip_address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $hostname = nagiosccm_replace_user_macros($hostname);
            $agent = grab_array_var($inargs, "agent", "");

            // Check for errors
            $errors = 0;
            $errmsg = array();
            if (is_valid_host_name($hostname) == false)
                $errmsg[$errors++] = _("Invalid host name.");

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;


        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // get variables that were passed to us
            $ip_address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $agent = grab_array_var($inargs, "agent", "");

            $services = "";
            $services_serial = grab_array_var($inargs, "services_serial");
            if ($services_serial != "")
                $services = json_decode(base64_decode($services_serial), true);
            else
                $services = grab_array_var($inargs, "services");

            $serviceargs = "";
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial");
            if ($serviceargs_serial != "")
                $serviceargs = json_decode(base64_decode($serviceargs_serial), true);
            else
                $serviceargs = grab_array_var($inargs, "serviceargs");

            $output = '

        <input type="hidden" name="ip_address" value="' . $ip_address . '">
        <input type="hidden" name="hostname" value="' . $hostname . '">
        <input type="hidden" name="agent" value="' . $agent . '">
        <input type="hidden" name="services_serial" value="' . base64_encode(json_encode($services)) . '">
        <input type="hidden" name="serviceargs_serial" value="' . base64_encode(json_encode($serviceargs)) . '">
';
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:


            $output = '
            <!--
            <p>' . _('Dont forget to download and install the Solaris Agent on the target server') . '!</p>
            <p><a href="' . $agent_url . '"><img src="' . theme_image("download.png") . '"></a> <a href="' . $agent_url . '"><b>' . _('Download Agent') . '<b></a></p>
            //-->
            ';
            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            $hostname = grab_array_var($inargs, "hostname", "");
            $ip_address = grab_array_var($inargs, "ip_address", "");
            $agent = grab_array_var($inargs, "agent", "");
            $hostaddress = $ip_address;

            $services_serial = grab_array_var($inargs, "services_serial", "");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            $services = json_decode(base64_decode($services_serial), true);
            $serviceargs = json_decode(base64_decode($serviceargs_serial), true);

            // save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["hostname"] = $hostname;
            $meta_arr["ip_address"] = $ip_address;
            $meta_arr["agent"] = $agent;
            $meta_arr["services"] = $services;
            $meta_arr["serivceargs"] = $serviceargs;
            save_configwizard_object_meta($wizard_name, $hostname, "", $meta_arr);

            $objs = array();

            if (!host_exists($hostname)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_linuxserver_host",
                    "host_name" => $hostname,
                    "address" => $hostaddress,
                    "icon_image" => "solaris.png",
                    "statusmap_image" => "solaris.png",
                    "_xiwizard" => $wizard_name,
                );
            }

            // see which services we should monitor
            foreach ($services as $svc => $svcstate) {

                //echo "PROCESSING: $svc -> $svcstate<BR>\n";

                switch ($svc) {

                    case "ping":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Ping",
                            "use" => "xiwizard_linuxserver_ping_service",
                            "_xiwizard" => $wizard_name,
                        );
                        break;


                    case "load":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Load",
                            "use" => "xiwizard_nrpe_service",
                            "check_command" => "check_nrpe!check_load!-a '-w " . $serviceargs["load_warning"] . " -c " . $serviceargs["load_critical"] . "'",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "cpustats":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "CPU Stats",
                            "use" => "xiwizard_nrpe_service",
                            "check_command" => "check_nrpe!check_cpu_stats!-a '-w " . $serviceargs["cpustats_warning"] . " -c " . $serviceargs["cpustats_critical"] . "'",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "memory":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Memory Usage",
                            "use" => "xiwizard_nrpe_service",
                            "check_command" => "check_nrpe!check_mem!-a '-w " . (100 - intval($serviceargs["cpustats_warning"])) . " -c " . (100 - intval($serviceargs["cpustats_critical"])) . "'",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "swap":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Swap Usage",
                            "use" => "xiwizard_nrpe_service",
                            "check_command" => "check_nrpe!check_swap!-a '-w " . (100 - intval($serviceargs["swap_warning"])) . " -c " . (100 - intval($serviceargs["swap_critical"])) . "'",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "openfiles":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Open Files",
                            "use" => "xiwizard_nrpe_service",
                            "check_command" => "check_nrpe!check_open_files!-a '-w " . $serviceargs["openfiles_warning"] . " -c " . $serviceargs["openfiles_critical"] . "'",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "users":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Users",
                            "use" => "xiwizard_nrpe_service",
                            "check_command" => "check_nrpe!check_users!-a '-w " . $serviceargs["users_warning"] . " -c " . $serviceargs["users_critical"] . "'",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "procs":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Total Processes",
                            "use" => "xiwizard_nrpe_service",
                            "check_command" => "check_nrpe!check_all_procs!-a '-w " . $serviceargs["procs_warning"] . " -c " . $serviceargs["procs_critical"] . "'",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "disk":
                        $donedisks = array();
                        $diskid = 0;
                        foreach ($serviceargs["disk"] as $diskname) {

                            if ($diskname == "")
                                continue;

                            //echo "HANDLING DISK: $diskname<BR>";

                            // we already configured this disk
                            if (in_array($diskname, $donedisks))
                                continue;
                            $donedisks[] = $diskname;

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $hostname,
                                "service_description" => $diskname . " Disk Usage",
                                "use" => "xiwizard_nrpe_service",
                                "check_command" => "check_nrpe!check_disk!-a '-w " . $serviceargs["disk_warning"][$diskid] . "% -c " . $serviceargs["disk_critical"][$diskid] . "% -p " . $diskname . "'",
                                "_xiwizard" => $wizard_name,
                            );

                            $diskid++;
                        }
                        break;

                    case "servicestate":
                        //if (!($RHEL || $DEB)) break;

                        $enabledservices = $svcstate;
                        foreach ($enabledservices as $sid => $sstate) {

                            $sname = $serviceargs["servicestate"][$sid]["service"];
                            $sdesc = $serviceargs["servicestate"][$sid]["name"];

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $hostname,
                                "service_description" => $sdesc,
                                "use" => "xiwizard_nrpe_service",
                                "check_command" => "check_nrpe!check_init_service!-a '" . $sname . "'",
                                "_xiwizard" => $wizard_name,
                            );
                        }
                        break;

                    case "processstate":

                        $enabledprocs = $svcstate;
                        foreach ($enabledprocs as $pid => $pstate) {

                            $pname = $serviceargs["processstate"][$pid]["process"];
                            $pdesc = $serviceargs["processstate"][$pid]["name"];

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $hostname,
                                "service_description" => $pdesc,
                                "use" => "xiwizard_nrpe_service",
                                "check_command" => "check_nrpe!check_services!-a '" . $pname . "'",
                                "_xiwizard" => $wizard_name,
                            );
                        }
                        break;

                    default:
                        break;
                }
            }

            //echo "OBJECTS:<BR>";
            //print_r($objs);
            //exit();

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:
            break;
    }

    return $output;
}
