import javax.management.ObjectName;
import javax.management.MalformedObjectNameException;
import java.lang.Math;

// For file read/write
import java.util.Arrays;
import java.util.List;
import java.io.IOException;
import java.io.File;
import java.net.URISyntaxException;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.StandardOpenOption;

public class CheckParameters {
	// Most of these arrays should be single-element, but in some cases we'll need multiple parameters for a check.
	private boolean doesCheckFile; // determines whether persistent data is held between checks
	private String fileName;
	private String svcName;
	private String perfName;
	private String url;
	private String type;
	private ObjectName[] objects;
	private String[] methodsAndAttributes;
	private boolean[] isMethod;
	private String[] compositeDataKeys;
	private String unitPerf;
	private String unitSvc;
	private Object valueSvc;

	// For checks that don't use the hidden files, url isn't needed.
	public CheckParameters(String svcName, String perfName, String type, String objectNames, String attributes, String methods, String keys, String unitPerf, String unitSvc) {
		this(false, null, null, svcName, perfName, type, objectNames, attributes, methods, keys, unitPerf, unitSvc);
	}

	public CheckParameters(boolean checksFile, String fileName, String url, String svcName, String perfName, String type, String objectNames, String attributes, String methods, String keys, String unitPerf, String unitSvc) {
		doesCheckFile = checksFile;
		this.fileName = null;
		if (checksFile && fileName != null) {
			this.fileName = fileName.replaceAll("\\s", ""); // remove all whitespace
		}
		this.url = url;
		this.svcName = svcName;
		this.perfName = perfName;
		this.unitPerf = unitPerf;
		this.unitSvc = unitSvc;
		valueSvc = null;

		String[] objectsRaw = objectNames.split(";"); // ObjectNames contain commas.
		String[] attributesRaw = {};
		String[] methodsRaw;
		String[] keysRaw;
		if (attributes != null && !attributes.isEmpty()) {
			attributesRaw = attributes.split(",");
			if (attributesRaw.length == 0) {
				attributesRaw = new String[objectsRaw.length];
				for (int i = 0; i < objectsRaw.length; i++) {
					attributesRaw[i] = "";
				}
			}
		}
		if (methods != null && !methods.isEmpty()) {
			methodsRaw = methods.split(",");
			if (methodsRaw.length == 0) {
				methodsRaw = new String[objectsRaw.length];
				for (int i = 0; i < objectsRaw.length; i++) {
					methodsRaw[i] = "";
				}
			}
		}
		else {
			methodsRaw = new String[attributesRaw.length];
			for (int i = 0; i < objectsRaw.length; i++) {
				methodsRaw[i] = "";
			}	
		}
		if (keys != null && !keys.isEmpty()) {
			keysRaw = keys.split(",");
			if (keysRaw.length == 0) {
				keysRaw = new String[objectsRaw.length];
				for (int i = 0; i < objectsRaw.length; i++) {
					keysRaw[i] = "";
				}
			}
		}
		else {
			keysRaw = new String[objectsRaw.length];
			for (int i = 0; i < objectsRaw.length; i++) {
				keysRaw[i] = "";
			}	
		}

		if ((objectsRaw.length != attributesRaw.length)
			|| ((methodsRaw.length != 0)
			    && (objectsRaw.length != methodsRaw.length))
			|| ((keysRaw.length != 0)
				&& (objectsRaw.length != keysRaw.length))) {

			throw new IllegalArgumentException("All of the lists (object, attribute, method, key) must be of equal length.");
		}

		objects = new ObjectName[objectsRaw.length];
		for (int i = 0; i < objectsRaw.length; i++) {
			try {
				if(objectsRaw[i].equals("time")) {
					objects[i] = null;
				}
				else {
					objects[i] = new ObjectName(objectsRaw[i]);
				}
			}
			catch (MalformedObjectNameException e) {
				throw new IllegalArgumentException(objects[i] + " could not be converted to ObjectName. Please contact the plugin maintainer.");
			}
		}

		if (attributesRaw.length != 0) {

			isMethod = new boolean[attributesRaw.length];
			methodsAndAttributes = new String[attributesRaw.length];
			for(int i = 0; i < attributesRaw.length; i++) {
				methodsAndAttributes[i] = methodsRaw[i];
				isMethod[i] = true;
				if (attributesRaw[i] != null && !(attributesRaw[i].isEmpty())) {
					methodsAndAttributes[i] = attributesRaw[i];
					isMethod[i] = false;
				}
			}
		}
    	compositeDataKeys = keysRaw;

    	if(type.equals("long") || type.equals("double") || type.equals("string") || type.equals("rate")) {
    		this.type = type;
    	}
    	else {
    		throw new IllegalArgumentException("CheckParameters defined incorrectly: type must be one of long, double, string, or rate.");
    	}
	}

	Object getCheckValue(Object[] values) {
		Object[] prev = new Object[values.length];
		if (doesCheckFile) {
			prev = readLastValueWriteNewValue(values);
			if (prev == null || prev.length == 0) {
				return 0;
			}
			for (int i = 0; i < prev.length; i++) {
				if (values[i] instanceof Double) {
					if (prev[i] instanceof String) {
						prev[i] = Double.parseDouble((String) prev[i]);
					}
					if (prev[i] instanceof Double && (double) prev[i] <= (double) values[i]) {
						values[i] = (double) values[i] - (double) prev[i];
					}
				}
				else if (values[i] instanceof Long || values[i] instanceof Integer) {
					if (values[i] instanceof Integer) {
						values[i] = ((Integer) values[i]).longValue();
					}
					if (prev[i] instanceof String) {
						long temp = Long.parseLong((String) prev[i]);
						prev[i] = temp;
					}
					if (prev[i] instanceof Long && (long) prev[i] <= (long) values[i]) {
						values[i] = (long) values[i] - (long) prev[i];
					}
				}
			}	
		}
		Object ret = null;
		if (type.equals("rate")) {
			if (values.length == 2 && (values[0] instanceof Double || values[0] instanceof Long) && (values[1] instanceof Double || values[1] instanceof Long)) {
				for (int i = 0; i < values.length; i++) {
					if (values[i] instanceof Long) {
						values[i] = new Long((long) values[i]).doubleValue();
					}
				}
				double val = ((double) values[0])/ ((double) values[1]);
				negotiateValue(val);
				ret = (Object) val;
			}
			else {
				throw new IllegalArgumentException("These CheckParameters are a rate but parameter list " + values[0] + " " + values[1] + " isn't 2 numbers");
			}
		}
		else if (type.equals("string")) {
			if (values.length == 1 && values[0] instanceof String) {
				ret = (Object) ((String) values[0]);
			}
			else {
				throw new IllegalArgumentException("These CheckParameters represent a String check but parameter list isn't a String");
			}
		}
		else if (type.equals("long")) {
			if (values.length == 1 && values[0] instanceof Long) {
				long longform = (long) values[0];
				ret = (Object) ((long) values[0]);
				negotiateValue(longform/1.0);
			}
			else if (values.length == 1 && values[0] instanceof Integer) {
				long longform = (long) (int) values[0];
				ret = (Object) ((long) (int) values[0]);
				negotiateValue(longform/1.0);
			}
			else {
				throw new IllegalArgumentException("These CheckParameters represent an integer check but parameter list isn't a long");
			}
		}
		else if (type.equals("double")) {
			if (values.length == 1 && values[0] instanceof Double) {
				values[0] = negotiateValue((double) values[0]);
				ret = (Object) ((double) values[0]);
			}
			else {
				throw new IllegalArgumentException("These CheckParameters represent a floating-point check but parameter list isn't a double");
			}
		}
		else {
			throw new IllegalArgumentException("Unsupported type for " + svcName + ". Please contact the plugin maintainer.");
		}
		return ret;
	}

	/* This takes an array of data, formatted like
	 * [<Service URL>, <Service Name>, <data>...]
	 * It checks in the data file For a line which matches the URL/Name,
	 * and if it exists, reads that data before overwriting it and returning the read data.
	 * If it doesn't exist, returns null.
	 */
	public Object[] readLastValueWriteNewValue(Object[] toWrite) {
		Object[] data = null;
		// Create the data file in whatever directory holds check_jvm.jar
		// Give it the name '.<application-server>'
		String jarDirectory = "";
		try {
			jarDirectory = new File(CheckParameters.class.getProtectionDomain().getCodeSource().getLocation().toURI()).getParent();
		}
		catch (SecurityException | URISyntaxException e) {
			assert true;
		}
		File theFile = new File(jarDirectory + "." + fileName);
		
		String lineToWrite = url + " ||| " + svcName;
		for(int i = 0; i < toWrite.length; i++) {
			lineToWrite += " ||| " + toWrite[i].toString();
		}

		List<String> linesToWrite;
		try {
			if (theFile.createNewFile()) {
				// File DNE previously.
				linesToWrite = Arrays.asList(lineToWrite);
			}
			else {
				// File already existed.
				List<String> oldLinesList= Files.readAllLines(theFile.toPath(), Charset.defaultCharset());
				boolean foundData = false;
				for(int i = 0; i < oldLinesList.size(); i++) {
					String[] oldData = oldLinesList.get(i).split(" \\|\\|\\| ");
					if (oldData.length <= 1) {
						continue;
					}
					if (oldData[1].equals(svcName) && oldData[0].equals(url)) {
						foundData = true;
						data = Arrays.copyOfRange(oldData, 2, oldData.length);
						oldLinesList.remove(i);
						oldLinesList.add(i, lineToWrite);
					}
				}
				if (!foundData) {
					oldLinesList.add(lineToWrite);
				}
				linesToWrite = oldLinesList;
			}
			Files.write(theFile.toPath(), linesToWrite, Charset.defaultCharset(), StandardOpenOption.TRUNCATE_EXISTING);
		}
		catch(IOException e) {
			throw new IllegalArgumentException("Failed to read/write to file: " + e.getMessage());
		}
		if (data == null || data.length == 0) {
			return new Object[0];
		}
		Object[] ret = new Object[data.length];
		for (int i = 0; i < ret.length; i++) {
			ret[i] = data[i]; // data is technically a String[].
		}
		return ret;
	}

	public boolean isMethod(int i) {
		return isMethod[i];
	}

	public String getMethAttrName(int i) {
		return methodsAndAttributes[i];
	}

	public String getKey(int i) {
		return compositeDataKeys[i];
	}

	public int getSize() {
		return objects.length;
	}

	public String getValueType() {
		return type;
	}

	public String getUnitSvc() {
		return unitSvc;
	}

	/* Takes a value and converts it to a "readable" form */
	public double negotiateValue(double value) {
		double valuePerf = value;
		if (unitPerf.equals("ms")) {
			if (value > 86400000.0) {
				double days = value / 86400000.0;
				unitSvc = "days";
				valueSvc = days;
			}
			else if (value > 3600000.0) {
				double hours = value / 3600000.0;
				unitSvc = "hours";
				valueSvc = hours;
			}
			else if (value > 60000.0) {
				double minutes = value / 60000.0;
				unitSvc = "minutes";
				valueSvc = minutes;
			}
			else {
				double seconds = value / 1000.0;
				unitSvc = "seconds";
				valueSvc = seconds;
			}
		}
		else if (unitPerf.equals("B")) {
			if (value > 1073741824.0) {
				double GiB = value / 1073741824.0;
				unitSvc = "GiB";
				valueSvc = GiB;
			}
			else if (value > 1048576.0) {
				double MiB = value / 1048576.0;
				unitSvc = "MiB";
				valueSvc = MiB;
			}
			else if (value > 1024.0) {
				double KiB = value / 1024.0;
				unitSvc = "KiB";
				valueSvc = KiB;
			}
			else {
				double B = value / 1.0;
				unitSvc = "B";
				valueSvc = B;
			}
		}
		else if (unitPerf.equals("%")) {
			unitSvc = "%";
			valueSvc = value * 100.0;
			valuePerf *= 100.0;
		}
		return valuePerf;
	}

	public String getUnitPerf() {
		return unitPerf;
	}

	public String getValueSvc(Object valuePerf) {
		if (valueSvc == null) {
			valueSvc = valuePerf;
		}
		if (valueSvc instanceof Double || valueSvc instanceof Float) {
			// Round to two decimals
			double tmp = 0;
			tmp = (double) valueSvc;
			tmp *= 100;
			tmp = Math.round(tmp);
			tmp /= 100.0;
			valueSvc = tmp;
		}
		return valueSvc.toString();
	}

	public String getName() {
		return svcName;
	}

	public String getPerfName() {
		return perfName;
	}

	public ObjectName getObjectName(int i) {
		return objects[i];
	}

	public boolean isTimestamp(int i) {
		return objects[i] == null && methodsAndAttributes[i].equals("time");
	}
}