/* Hi there. You can compile this program by running the following from plugins/src:
 * javac -cp . GenericASCheck.java -target 7 -source 7 -Xlint:unchecked -bootclasspath /usr/lib/jvm/java-1.7.0/jre/lib/rt.jar
 * jar cfm check_jvm.jar MANIFEST.mf *.class org/apache/commons/cli/*.class
 * mv check_jvm.jar ..
 * Note: You may need to change the bootclasspath and target/source versions depending on which JVM(s) you have installed.
 */


import org.apache.commons.cli.Options;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.DefaultParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.ParseException;

import java.io.PrintStream;

public class GenericASCheck {
	private JVMCheck query;
	private NagiosThresholds[] checkThresholds;
	private CheckParameters[] paramsList;

	public static String[] serverTypes = { "tomcat", "jboss", "jetty", "glassfish", "other"};
	public GenericASCheck(CommandLine cmd) {

		// Don't let server connection print to terminal.
		PrintStream stderr = System.err;
		PrintStream nullStream = new PrintStream(new NullStream());
		System.setErr(nullStream);

		// Now try connecting.
		try {
			query = new JVMCheck(cmd.getOptionValue("service-url"), cmd.getOptionValue("username"), cmd.getOptionValue("password"));
		}
		catch (ClassCastException e) {
			System.out.println("UNKNOWN: could not connect to provided service URL");
			System.exit(3);
		}

		// Now put it back.
		System.setErr(stderr);

		// Get the AS type, so we know which MBeans are built in.
		String serverType = "other";
		if (cmd.hasOption("server-type")) {
			serverType = cmd.getOptionValue("server-type");
			boolean matched = false;
			for (String type : serverTypes) {
				if (type.equals(serverType)) {
					matched = true;
				}
			}

			if (!matched) {
				serverType = "other";
			}
		}

		// Check information parsing.
		String[] checks_raw = cmd.getOptionValue("check-types").split(",");
		String[] warnings_raw;
		String[] criticals_raw;
		if (cmd.hasOption("warning")) {	
			warnings_raw = cmd.getOptionValue("warning").split(",", -1);
		}
		else {
			warnings_raw = new String[checks_raw.length];
			for (int i = 0; i < checks_raw.length; i++) {
				warnings_raw[i] = "";
			}
		}
		if (cmd.hasOption("critical")) {
			criticals_raw = cmd.getOptionValue("critical").split(",", -1);
		}
		else {
			criticals_raw = new String[checks_raw.length];
			for (int i = 0; i < checks_raw.length; i++) {
				warnings_raw[i] = "";
			}
		}

		if (warnings_raw.length < checks_raw.length) {
			String[] warnings_new = new String[checks_raw.length];
			int i = 0;
			for (i = 0; i < warnings_raw.length; i++) {
				warnings_new[i] = warnings_raw[i];
			}
			for (i = warnings_raw.length; i < warnings_new.length; i++) {
				warnings_new[i] = "";
			}
			warnings_raw = warnings_new;
		}

		if (criticals_raw.length < checks_raw.length) {
			String[] criticals_new = new String[checks_raw.length];
			int i = 0;
			for (i = 0; i < criticals_raw.length; i++) {
				criticals_new[i] = criticals_raw[i];
			}
			for (i = criticals_raw.length; i < criticals_new.length; i++) {
				criticals_new[i] = "";
			}
			criticals_raw = criticals_new;
		}

		populateCheckParameters(checks_raw, serverType, cmd.getOptionValue("service-url"));

		checkThresholds = new NagiosThresholds[checks_raw.length];
		for (int i = 0; i < checks_raw.length; i++) {
			checkThresholds[i] = new NagiosThresholds(warnings_raw[i], criticals_raw[i], paramsList[i].getValueType());
		}
	}

	public void runChecks() {
		Object[] values = query.getValues(paramsList);
		int highWaterMark = 0;
		int numHigh = 0;
		String longOut = "";
		String shortOut = "";
		String perfData = "";

		for(int i = 0; i < values.length; i++) {
			int result = checkThresholds[i].check(values[i]);
			if (result == highWaterMark) {
				numHigh += 1;
			}
			else if (result > highWaterMark) {
				highWaterMark = result;
				numHigh = 1;
			}

			longOut += makeLongOut(paramsList[i].getName(), result, paramsList[i].getValueSvc(values[i]), paramsList[i].getUnitSvc()) + "\n";
			perfData += makePerfData(paramsList[i].getPerfName(), values[i].toString(), paramsList[i].getUnitPerf(),
				checkThresholds[i].getRawThreshold(true), checkThresholds[i].getRawThreshold(false)) + " ";

		}

		shortOut = makeShortOut(highWaterMark, numHigh);

		String fullOutput = "";
		if (paramsList.length >= 2) {
			fullOutput = shortOut + " | " + perfData +  "\n" + longOut;
		}
		else {
			fullOutput = longOut.substring(0, longOut.length()-1) + " | " + perfData;
		}

		System.out.println(fullOutput);
		query.close();
		System.exit(highWaterMark);
		return;
	}

	/* Call only after query has been initialized.
	 * Returns the first element of potentialObjectNames which has an associated MBean name on the application server.
	 */
	public String findWhichObjectName(String[] potentialObjectNames) {
		for (int i = 0; i < potentialObjectNames.length; i++) {

			boolean matched = query.queryNames(potentialObjectNames[i]);
			if (matched) {
				return potentialObjectNames[i];
			}
		}
		return null;
	}

	public String getObjectNameFromPattern(String knownObjectNamePattern) {
		return query.getFirstObjectName(knownObjectNamePattern);
	}

	/* Caller and serviceURL are passed in so that file-writing parameters can guarantee uniqueness of data. */
	public void populateCheckParameters(String[] inputs, String caller, String serviceURL) {
		paramsList = new CheckParameters[inputs.length];
		for(int i = 0; i < inputs.length; i++) {
			String[] temp = inputs[i].split(":");
			temp[0] = temp[0].toLowerCase();
			String trueObjectName; // Apparently this gets hoisted out of the switch anyways...
			switch (temp[0]) {	
				case "uptime":
					paramsList[i] = new CheckParameters("Uptime", caller + "_uptime", "long", "java.lang:type=Runtime", "Uptime", "", "", "ms", "ms alive");
					break;
				case "memorysimplenonheap":
					paramsList[i] = new CheckParameters("Non-Heap-Allocated Memory", caller + "_non_heap", "long", "java.lang:type=Memory", "NonHeapMemoryUsage", "", "used", "B", "bytes allocated");
					break;
				case "memorysimpleheap":
					paramsList[i] = new CheckParameters("Heap-Allocated Memory", caller + "_heap", "long", "java.lang:type=Memory", "HeapMemoryUsage", "", "used", "B", "bytes allocated");
					break;
				case "memoryeden":
					trueObjectName = getObjectNameFromPattern("java.lang:type=MemoryPool,name=*Eden*");
					paramsList[i] = new CheckParameters("Eden Space (heap)", caller + "_Eden", "long", trueObjectName, "Usage", "", "used", "B", "bytes allocated");
					break;
				case "memorysurvivor":
					String [] potentialObjectNames = {"java.lang:type=MemoryPool,name=*Survivor*", "java.lang:type=MemoryPool,name=*Tenured*"};
					trueObjectName = findWhichObjectName(potentialObjectNames);
					trueObjectName = getObjectNameFromPattern(trueObjectName);
					paramsList[i] = new CheckParameters("Survivor/Tenured Space (heap)", caller + "_Survivor", "long", trueObjectName, "Usage", "", "used", "B", "bytes allocated");
					break;
				case "memoryold":
					trueObjectName = getObjectNameFromPattern("java.lang:type=MemoryPool,name=*Old Gen*");
					paramsList[i] = new CheckParameters("Old Space (heap)", caller + "_Old", "long", trueObjectName, "Usage", "", "used", "B", "bytes allocated");
					break;
				case "memorycodecache":
					paramsList[i] = new CheckParameters("Code Cache (non-heap)", caller + "_code_cache", "long", "java.lang:type=MemoryPool,name=Code Cache", "Usage", "", "used", "B", "bytes allocated");
					break;
				case "memorycompressedclass":
					paramsList[i] = new CheckParameters("Compressed Class Space (non-heap)", caller + "_compressed_class", "long", "java.lang:type=MemoryPool,name=Compressed Class Space", "Usage", "", "used", "B", "bytes allocated");
					break;
				case "memorymetaspace":
					paramsList[i] = new CheckParameters("Metaspace (non-heap)", caller + "_metaspace", "long", "java.lang:type=MemoryPool,name=Metaspace", "Usage", "", "used", "B", "bytes allocated");
					break;
				case "classcount":
					paramsList[i] = new CheckParameters("Class Count", caller + "_classes", "long", "java.lang:type=ClassLoading", "LoadedClassCount", "", "", "", "classes loaded");
					break;
				case "threadcount":
					paramsList[i] = new CheckParameters("Thread Count", caller + "_threads", "long", "java.lang:type=Threading", "ThreadCount", "", "", "", "threads running");
					break;
				case "processcpuusage":
					paramsList[i] = new CheckParameters("Process CPU usage", caller + "_process_cpu_usage", "double", "java.lang:type=OperatingSystem", "ProcessCpuLoad", "", "", "%", "%");
					break;
				case "systemcpuusage":
					paramsList[i] = new CheckParameters("System CPU usage", caller + "_system_cpu_usage", "double", "java.lang:type=OperatingSystem", "SystemCpuLoad", "", "", "%", "%");
					break;

				// Tomcat-specific checks.
				// All of these fall through to default if serverType wasn't set to "tomcat" in the options parsing.
				case "requestsperminute":
					if (caller.equals("tomcat")) {
						if (temp.length != 2) {
							throw new IllegalArgumentException("Please specify a request processor for RequestsPerMinute");
						}
						paramsList[i] = new CheckParameters(true, caller, serviceURL, "Requests Per Minute (" + temp[1] + ")", caller + "_requests_minute", "rate", 
							"Catalina:type=GlobalRequestProcessor,name=\"" + temp[1] + "\";time", 
							"requestCount,time", "", "", "", "requests");
						break;
					}

				case "bytesperminute":
					if (caller.equals("tomcat")) {
						if (temp.length != 2) {
							throw new IllegalArgumentException("Please specify a request processor for BytesPerMinute");
						}
						paramsList[i] = new CheckParameters(true, caller, serviceURL, "Bytes Per Minute (" + temp[1] + ")", caller + "_bytes_minute", "rate", 
							"Catalina:type=GlobalRequestProcessor,name=\"" + temp[1] + "\";time", 
							"bytesSent,time", "", "", "B", "bytes");
						break;
					}

				case "bytesperrequest":
					if (caller.equals("tomcat")) {
						if (temp.length != 2) {
							throw new IllegalArgumentException("Please specify a request processor for BytesPerRequest");
						}
						paramsList[i] = new CheckParameters(true, caller, serviceURL, "Bytes Per Request (" + temp[1] + ")", caller + "_bytes_request", "rate", 
							"Catalina:type=GlobalRequestProcessor,name=\"" + temp[1] + "\";Catalina:type=GlobalRequestProcessor,name=\"" + temp[1] + "\"", 
							"bytesSent,requestCount", ",,", "", "B", "bytes");
						break;
					}

				case "errorsperminute":
					if (caller.equals("tomcat")) {
						if (temp.length != 2) {
							throw new IllegalArgumentException("Please specify a request processor for ErrorsPerMinute");
						}
						paramsList[i] = new CheckParameters(true, caller, serviceURL, "Errors Per Minute (" + temp[1] + ")", caller + "_errors_minute", "rate", 
							"Catalina:type=GlobalRequestProcessor,name=\"" + temp[1] + "\";time", 
							"errorCount,time", "", "", "", "errors");
						break;
					}

				case "processingtimeperrequest":
					if (caller.equals("tomcat")) {
						if (temp.length != 2) {
							throw new IllegalArgumentException("Please specify a request processor for ProcessingTimePerRequest");
						}
						paramsList[i] = new CheckParameters(true, caller, serviceURL, "Processing Time Per Request (" + temp[1] + ")", caller + "_proctime_request", "rate",
							"Catalina:type=GlobalRequestProcessor,name=\"" + temp[1] + "\";Catalina:type=GlobalRequestProcessor,name=\"" + temp[1] + "\"", 
							"processingTime,requestCount", "", "", "ms", "milliseconds");
						break;
					}

				default:
					throw new IllegalArgumentException("Invalid Check Type " + temp[0]);
			}
		}
	}

	public static void main(String[] args) {
		try {
			CommandLine cmd = parseOptions(args); // sanity checking done inside
			GenericASCheck tc = new GenericASCheck(cmd);
			tc.runChecks();
		}
		catch (IllegalArgumentException e) {
			System.out.println("UNKNOWN: " + e.getMessage());
			System.exit(3);
		}
	}

	public static String makeShortOut(int highest, int numAtHighest) {
		String status = getStatus(highest);
		return status + ": " + numAtHighest + " checks returned " + status;
	}

	public static String makeLongOut(String name, int returnCode, Object value, String unitOfMeasure) {
		String status = getStatus(returnCode);
		return name + " returned " + status + " with " + value.toString() + " " + unitOfMeasure;
	}

	public static String makePerfData(String name, String value, String uom, String warning, String critical, String min, String max) {
		if (name == null || value == null) {
			return "";
		}
		return name + "=" + value + uom + ";" + warning + ";" + critical + ";" + min + ";" + max;
	}

	public static String makePerfData(String name, String value, String uom, String warning, String critical) {
		return makePerfData(name, value, uom, warning, critical, "", "");
	}

	public static String getStatus(int returnCode) {
		switch(returnCode) {
			case 0:
				return "OK";
			case 1:
				return "WARNING";
			case 2:
				return "CRITICAL";
			default:
				return "UNKNOWN";
		}
	}

	public static CommandLine parseOptions(String[] args) {
		Options options = makeOptions();
		CommandLineParser parser = new DefaultParser();
		CommandLine cmd = null;
		try {
			cmd = parser.parse(options, args);

			if (cmd.hasOption("version") || cmd.getOptions().length == 0) {
				System.out.println("check_jvm.jar, version 1.1.0, (c) 2017-2019 Nagios Enterprises, LLC");
				System.exit(0);
			}

			if (cmd.hasOption("help")) {
				HelpFormatter hf = new HelpFormatter();
				String header = "Monitor an application server (Tomcat, JBoss/Wildfly, Jetty, GlassFish) instance remotely via JMX.";
				String footer = "Please report issues at our support forums: https://support.nagios.com/forum/";
				hf.printHelp("check_jvm.jar", header, options, footer, true);
				System.exit(0);
			}

			if (!cmd.hasOption("service-url")) {
				throw new IllegalArgumentException("Please set the service URL for your JVM using the -s flag.");
			}

			if (!cmd.hasOption("check-types")) {
				throw new IllegalArgumentException("Please specify a list of checks using the -C flag.");
			}
		}
		catch(IllegalArgumentException | ParseException e) {
			throw new IllegalArgumentException("Failed to parse options: " + e.getMessage());
		}
		return cmd;
	}

	public static Options makeOptions() {
		Options options = new Options();

		options.addOption("w", "warning", true, 
			"A list of nagios warning thresholds. At least one of this and critical must be entered." /* +
			" You may alternatively enter a string of the form \"[@]s$SOMESTRING\", where $SOMESTRING is the string you do/do not want to match."*/);
		options.addOption("c", "critical", true,
			"A list of nagios critical threhsolds. At least one of this and warning must be entered." /* +
			" You may alternatively enter a string of the form \"[@]s$SOMESTRING\", where $SOMESTRING is the string you do/do not want to match."*/);
		options.addOption("s", "service-url", true,
			"The full JMX service URL of your JVM (defaults to localhost). This will be " +
			"similar to 'service:jmx:rmi:///jndi/rmi://<host>:<port>/jmxrmi'");
		options.addOption("C", "check-types", true,
			"A list of the checks that should be run. Valid checks: "
			+ "Uptime, MemorySimpleNonHeap, MemorySimpleHeap, MemoryEden, MemorySurvivor, MemoryOld, MemoryCodeCache, MemoryCompressedClass, MemoryMetaspace, "
			+ "ClassCount, ThreadCount, ProcessCPUUsage, and SystemCPUUsage. "
			+ "Tomcat servers also have the checks RequestsPerMinute, BytesPerMinute, BytesPerRequest, ErrorsPerMinute, and ProcessingTimePerRequest."
			+ " Note: RequestsPerMinute, BytesPerMinute, BytesPerRequest, ErrorsPerMinute, and ProcessingTimePerRequest also require the name of the relevant request processor. For example,"
			+ " \"RequestsPerMinute:http-nio-0.0.0.0-8080\". For rate-based checks, a negative value may be returned after a counter is reset.");
		options.addOption("t", "server-type", true,
			"The name of the server brand you're using. Should be one of {tomcat, jboss, jetty, glassfish, other}. Defaults to 'other'. Wildfly users may use 'jboss'.");
		options.addOption("u", "username", true,
			"A username to pass to the JMX connection (should be used in conjunction with the password option");
		options.addOption("p", "password", true,
			"A password to pass to the JMX connection (should be used in conjunction with the username option");
		options.addOption("h", "help", false, 
			"Print this message and exit.");
		options.addOption("v", "version", false,
			"Print the version information and exit.");
	
		return options;
	}
}