<?php
//
// SSH Proxy Config Wizard
// Copyright (c) 2010-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

windows_ssh_configwizard_init();

function windows_ssh_configwizard_init()
{
    $name = "windows-ssh";

    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.1.0",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor a remote Windows Machine using SSH."),
        CONFIGWIZARD_DISPLAYTITLE => _("Windows SSH"),
        CONFIGWIZARD_FUNCTION => "windows_ssh_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "windows_ssh.png",
        CONFIGWIZARD_FILTER_GROUPS => array('windows'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );

    register_configwizard($name, $args);
}

function validate_ip_address($ip_address)
{
    return filter_var($ip_address, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4) || filter_var($ip_address, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6);
}

function is_python_3()
{
    $versionString = shell_exec('python3 -c "import sys; print(sys.version_info[0])" 2>/dev/null');

    if ($versionString === null) {
        return false;
    }

    $pythonVersion = (int) $versionString;

    return ($pythonVersion >= 3);
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function windows_ssh_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    $wizard_name = "windows_ssh";

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:
            $address = grab_array_var($inargs, "ip_address", "");
            $ssh_username = grab_array_var($inargs, "ssh_username", "");
            $services = grab_array_var($inargs, "services", "");
            $services_serial = grab_array_var($inargs, "services_serial", "");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");
            $selectedhostconfig = grab_array_var($inargs, "selectedhostconfig", "");
            $operation = grab_array_var($inargs, "operation", "new");

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__ . '/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:
            $address = grab_array_var($inargs, "ip_address", "");
            $address = nagiosccm_replace_user_macros($address);
            $ssh_username = grab_array_var($inargs, "ssh_username", "");

            $errors = 0;
            $errmsg = array();

            if (have_value($address) == false) {
                $errmsg[$errors++] = _("No address specified.");
            } else if (validate_ip_address($address) == false) {
                $errmsg[$errors++] = _("Invalid address specified.");
            }

            if (have_value($ssh_username) == false) {
                $errmsg[$errors++] = _("No SSH username specified.");
            }

            if (is_python_3() == false) {
                $errmsg[$errors++] = _("Python 3 is required to run this wizard.");
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:
            $address = grab_array_var($inargs, "ip_address");
            $ssh_username = grab_array_var($inargs, "ssh_username", "");
            $selectedhostconfig = grab_array_var($inargs, "selectedhostconfig", "");
            $operation = grab_array_var($inargs, "operation", "new");

            # Try to get the "hostname", from the provided ip
            # What happens if we are given a hostname?
            $hostname_byaddr = @gethostbyaddr($address);

            $hostname = grab_array_var($inargs, "hostname", (!empty($hostname_byaddr) ? $hostname_byaddr : $address));

            $services = "";
            $services_serial = grab_array_var($inargs, "services_serial", "");

            if ($services_serial != "") {
                $services = json_decode(base64_decode($services_serial), true);
            }

            if (!is_array($services)) {
                $services_default = array(
                    "volume" => array(),
                    "cpu_usage" => array(),
                    "disk_usage" => array(),
                    "services" => array(),
                );

                $services_default["ping"]["monitor"] = "off";
                $services_default["tcp"]["monitor"] = "on";

                $services_default["volume"]["monitor"] = "on";
                $services_default["volume"][0]["drive"] = "C:";
                $services_default["volume"][0]["warning"] = "65";
                $services_default["volume"][0]["critical"] = "100";
                $services_default["volume"][0]["outputType"] = "GB";
                $services_default["volume"][0]["metric"] = "Used";
                $services_default["volume"][1]["warning"] = "65";
                $services_default["volume"][1]["critical"] = "100";
                $services_default["volume"][1]["outputType"] = "GB";
                $services_default["volume"][1]["metric"] = "Used";

                $services_default["cpu_usage"]["monitor"] = "on";
                $services_default["cpu_usage"]["warning"] = "80";
                $services_default["cpu_usage"]["critical"] = "90";

                $services_default["disk_usage"]["monitor"] = "on";
                $services_default["disk_usage"][0]["warning"] = "65";
                $services_default["disk_usage"][0]["critical"] = "100";
                $services_default["disk_usage"][0]["metric"] = "Total";
                $services_default["disk_usage"][0]["disk_number"] = "0";
                $services_default["disk_usage"][1]["warning"] = "65";
                $services_default["disk_usage"][1]["critical"] = "100";
                $services_default["disk_usage"][1]["metric"] = "Total";

                $services_default["windows_services"]["monitor"] = "on";
                $services_default["windows_services"][0]["service_name"] = "Spooler";
                $services_default["windows_services"][0]["display_name"] = "Print Spooler";
                $services_default["windows_services"][0]["expected_state"] = "Running";
                $services_default["windows_services"][1]["expected_state"] = "Running";

                $services_default["windows_processes"]["monitor"] = "on";
                $services_default["windows_processes"][0]["process_name"] = "notepad";
                $services_default["windows_processes"][0]["display_name"] = "Notepad";
                $services_default["windows_processes"][0]["outputType"] = "MB";
                $services_default["windows_processes"][0]["warning"] = "400";
                $services_default["windows_processes"][0]["critical"] = "500";
                $services_default["windows_processes"][0]["metric"] = "Memory";
                $services_default["windows_processes"][1]["outputType"] = "MB";
                $services_default["windows_processes"][1]["warning"] = "400";
                $services_default["windows_processes"][1]["critical"] = "500";
                $services_default["windows_processes"][1]["metric"] = "Memory";

                $services_default["memory_usage"]["monitor"] = "on";
                $services_default["memory_usage"]["warning"] = "1024";
                $services_default["memory_usage"]["critical"] = "512";
                $services_default["memory_usage"]["metric"] = "Available";
                $services_default["memory_usage"]["outputType"] = "MB";

                $services = grab_array_var($inargs, "services", $services_default);
            }

            $serviceargs = "";
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            # TODO: serviceargs is not used by this wizard.
            if ($serviceargs_serial != "") {
                $serviceargs = json_decode(base64_decode($serviceargs_serial), true);
            }

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__ . '/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $ssh_username = grab_array_var($inargs, "ssh_username");
            $hostname = nagiosccm_replace_user_macros($hostname);

            $services = "";
            $services_serial = grab_array_var($inargs, "services_serial");

            if ($services_serial != "") {
                $services = json_decode(base64_decode($services_serial), true);
            } else {
                $services = grab_array_var($inargs, "services");
            }

            $serviceargs = "";
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial");

            # TODO: serviceargs is not used in this wizard.
            if ($serviceargs_serial != "") {
                $serviceargs = json_decode(base64_decode($serviceargs_serial), true);
            } else {
                $serviceargs = grab_array_var($inargs, "serviceargs");
            }

            $errors = 0;
            $errmsg = array();
            // $errmsg[$errors++] = _("No address specified.");

            if (isset($services["volume"])) {
                foreach ($services["volume"] as $key => $value) {
                    if ($value == "on") {
                        continue;
                    }

                    if (empty($services["volume"][$key]["drive"])) {
                        continue;
                    }

                    if (empty($services["volume"][$key]["warning"]) || empty($services["volume"][$key]["critical"])) {
                        $errmsg[$errors++] = _("Volume Warning and Critical values are required if the drive is defined.");
                    }
                }
            }

            if (isset($services["disk_usage"])) {
                foreach ($services["disk_usage"] as $key => $value) {
                    if ($value == "on") {
                        continue;
                    }

                    if (empty($services["disk_usage"][$key]["disk_number"])) {
                        continue;
                    }

                    if (empty($services["disk_usage"][$key]["warning"]) || empty($services["disk_usage"][$key]["critical"])) {
                        $errmsg[$errors++] = _("Disk Warning and Critical values are required if the disk number is defined.");
                    }
                }
            }

            if (isset($services["memory_usage"])) {
                if (array_key_exists('monitor', $services['memory_usage']) && $services["memory_usage"]["monitor"] == "on") {
                    if ($services["memory_usage"]["warning"] === null || $services["memory_usage"]["warning"] === '' || $services["memory_usage"]["critical"] === null || $services["memory_usage"]["critical"] === '') {
                        $errmsg[$errors++] = _("Memory Warning and Critical values are required if Memory Usage is enabled.");
                    }

                    if (!is_numeric($services["memory_usage"]["warning"]) || !is_numeric($services["memory_usage"]["critical"])) {
                        $errmsg[$errors++] = _("Memory Warning and Critical values must be numeric.");
                    }

                    if ($services["memory_usage"]["warning"] < 0 || $services["memory_usage"]["critical"] < 0) {
                        $errmsg[$errors++] = _("Memory Warning and Critical values must be positive.");
                    }
                }
            }

            if (isset($services["cpu_usage"])) {
                if (array_key_exists('monitor', $services['cpu_usage']) && $services["cpu_usage"]["monitor"] == "on") {
                    if ($services["cpu_usage"]["warning"] === null || $services["cpu_usage"]["warning"] === '' || $services["cpu_usage"]["critical"] === null || $services["cpu_usage"]["critical"] === '') {
                        $errmsg[$errors++] = _("CPU Warning and Critical values are required if CPU Usage is enabled.");
                    }

                    if (!is_numeric($services["cpu_usage"]["warning"]) || !is_numeric($services["cpu_usage"]["critical"])) {
                        $errmsg[$errors++] = _("CPU Warning and Critical values must be numeric.");
                    }

                    if ($services["cpu_usage"]["warning"] < 0 || $services["cpu_usage"]["critical"] < 0) {
                        $errmsg[$errors++] = _("CPU Warning and Critical values must be positive.");
                    }
                }
            }

            // for windows services, if service is set but display name is not, set display name to service name
            if (isset($services["windows_services"])) {
                foreach ($services["windows_services"] as $key => $value) {
                    if ($value == "on") {
                        continue;
                    }

                    if (empty($services["windows_services"][$key]["service_name"])) {
                        continue;
                    }

                    if (empty($services["windows_services"][$key]["display_name"])) {
                        $services["windows_services"][$key]["display_name"] = $services["windows_services"][$key]["service_name"];
                    }
                }
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE3HTML:
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $ssh_username = grab_array_var($inargs, "ssh_username");
            $hostname = nagiosccm_replace_user_macros($hostname);
            $operation = grab_array_var($inargs, "operation");
            $selectedhostconfig = grab_array_var($inargs, "selectedhostconfig");
            $config_serial = grab_array_var($inargs, "config_serial");

            $services = "";
            $services_serial = grab_array_var($inargs, "services_serial");

            if ($services_serial != "") {
                $services = json_decode(base64_decode($services_serial), true);
            } else {
                $services = grab_array_var($inargs, "services");
            }

            $serviceargs = "";
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial");

            if ($serviceargs_serial != "") {
                $serviceargs = json_decode(base64_decode($serviceargs_serial), true);
            } else {
                $serviceargs = grab_array_var($inargs, "serviceargs");
            }

            if (isset($services["volume"])) {
                foreach ($services["volume"] as $key => $value) {
                    if ($value == "on") {
                        continue;
                    }

                    $unset = true;

                    foreach ($value as $key2 => $value2) {
                        if ($value2 != "") {
                            $unset = false;
                        }
                    }

                    if ($unset) {
                        unset($services["volume"][$key]);
                    }
                }
            }

            if (isset($services["disk_usage"])) {
                foreach ($services["disk_usage"] as $key => $value) {
                    if ($value == "on") {
                        continue;
                    }

                    $unset = true;

                    foreach ($value as $key2 => $value2) {
                        if ($value2 != "") {
                            $unset = false;
                        }
                    }

                    if ($unset) {
                        unset($services["disk_usage"][$key]);
                    }
                }
            }

            if (isset($services["windows_services"])) {
                foreach ($services["windows_services"] as $key => $value) {
                    if ($value == "on") {
                        continue;
                    }

                    $unset = true;

                    foreach ($value as $key2 => $value2) {
                        if ($value2 != "") {
                            $unset = false;
                        }
                    }

                    if ($unset) {
                        unset($services["windows_services"][$key]);
                    }
                }
            }

            if (isset($services["windows_processes"])) {
                foreach ($services["windows_processes"] as $key => $value) {
error_log("services[windows_processes] key [$key] value ".var_export($value, true));
                    if ($value == "on") {
                        continue;
                    }

                    $unset = true;

                    foreach ($value as $key2 => $value2) {
                        if ($value2 != "") {
                            $unset = false;
                        }
                    }

                    if ($unset) {
                        unset($services["windows_processes"][$key]);
                    }
                }
            }

            ob_start();
?>
        <input type="hidden" id="ip_address" name="ip_address" value="<?= $address ?>" />
        <input type="hidden" id="hostname" name="hostname" value="<?= $hostname ?>" />
        <input type="hidden" id="ssh_username" name="ssh_username" value="<?= $ssh_username ?>" />
        <input type="hidden" id="services_serial" name="services_serial" value="<?= base64_encode(json_encode($services)) ?>" />
        <input type="hidden" id="serviceargs_serial" name="serviceargs_serial" value="<?= base64_encode(json_encode($serviceargs)) ?>" />
        <input type="hidden" id="operation" name="operation" value="<?= encode_form_val($operation) ?>"/>
        <input type="hidden" id="selectedhostconfig" name="selectedhostconfig" value="<?= encode_form_val($selectedhostconfig) ?>"/>
        <input type="hidden" id="config_serial" name="config_serial" value="<?= encode_form_val($config_serial) ?>" />
<?php
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:
            $services_serial = grab_array_var($inargs, "services_serial");
            $services = json_decode(base64_decode($services_serial), true);

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:

            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:
            $hostname = grab_array_var($inargs, "hostname", "");
            $address = grab_array_var($inargs, "ip_address", "");

            $ssh_username = grab_array_var($inargs, "ssh_username", "");
            $hostaddress = $address;

            $services_serial = grab_array_var($inargs, "services_serial", "");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            $services = json_decode(base64_decode($services_serial), true);
            $serviceargs = json_decode(base64_decode($serviceargs_serial), true);

            # TODO: What is the point of this???
            $meta_arr = array();
            $meta_arr["hostname"] = $hostname;
            $meta_arr["ip_address"] = $address;
            $meta_arr["ssh_username"] = $ssh_username;
            $meta_arr["services"] = $services;
            $meta_arr["serviceargs"] = $serviceargs;
            save_configwizard_object_meta($wizard_name, $hostname, "", $meta_arr);

            $objs = array();

            if (!host_exists($hostname)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_linuxserver_host",
                    "host_name" => $hostname,
                    "address" => $hostaddress,
                    "icon_image" => "windows_ssh.png",
                    "statusmap_image" => "windows_ssh.png",
                    "_xiwizard" => $wizard_name,
                );
            }

            foreach ($services as $svc => $svcstate) {
                switch ($svc) {
                    case "tcp":
                        if (array_key_exists("ping", $services) && $services["ping"]["monitor"] == "on") {
                            break;
                        }

                        $port = 22;

                        $objs[] = array(
                            "type" => OBJECTTYPE_HOST,
                            "use" => "xiwizard_website_host",
                            "host_name" => $hostname,
                            "icon_image" => "www_server.png",
                            "statusmap_image" => "www_server.png",
                            "check_command" => "check_tcp!$port!",
                            "_xiwizard" => $wizard_name,
                        );

                    break;

                    case "ping":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Ping",
                            "use" => "xiwizard_linuxserver_ping_service",
                            "_xiwizard" => $wizard_name,
                        );

                    break;

                    case "volume":
                        if (($services["volume"]["monitor"] != "on") || (!array_key_exists("volume", $services))) {
                            break;
                        }

                        foreach ($services["volume"] as $key => $value) {
                            if ($key === "monitor") {
                                continue;
                            }

                            if (empty($services["volume"][$key]["drive"])) {
                                continue;
                            }

                            $checkcommand = "check_volume_by_ssh! ";
                            $checkcommand .= "-u " . $ssh_username . " ";
                            $checkcommand .= "-a '-volumename " . $services["volume"][$key]["drive"] . "\ -outputType " . $services["volume"][$key]["outputType"] . " -metric " . $services["volume"][$key]["metric"] . "' ";


                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $hostname,
                                "service_description" => "Volume " . $services["volume"][$key]["drive"],
                                "use" => "generic-service",
                                "check_command" => $checkcommand,
                                "_xiwizard" => $wizard_name,
                            );
                        }

                        break;

                    case "cpu_usage":
                        if (($services["cpu_usage"]["monitor"] != "on") || (!array_key_exists("cpu_usage", $services))) {
                            break;
                        }

                        $checkcommand = "check_cpu_usage_by_ssh! ";
                        $checkcommand .= "-u " . $ssh_username . " ";
                        $checkcommand .= "-a '-warning " . $services["cpu_usage"]["warning"] . " -critical " . $services["cpu_usage"]["critical"] . "' ";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "CPU Usage",
                            "use" => "generic-service",
                            "check_command" => $checkcommand,
                            "_xiwizard" => $wizard_name,
                        );

                        break;

                    case "disk_usage":
                        if (($services["disk_usage"]["monitor"] != "on") || (!array_key_exists("disk_usage", $services))) {
                            break;
                        }

                        foreach ($services["disk_usage"] as $key => $value) {
                            if ($key === "monitor") {
                                continue;
                            }

                            if (!array_key_exists("disk_number", $services["disk_usage"][$key]) || empty($services["disk_usage"][$key]["disk_number"])) {
                                continue;
                            }

                            $checkcommand = "check_disk_usage_by_ssh! ";
                            $checkcommand .= "-u " . $ssh_username . " ";
                            $checkcommand .= "-a '-metric " . $services["disk_usage"][$key]["metric"] . " -diskNum " . $services["disk_usage"][$key]["disk_number"] . " -warning " . $services["disk_usage"][$key]["warning"] . " -critical " . $services["disk_usage"][$key]["critical"] . "' ";

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $hostname,
                                "service_description" => "Disk Number: " . $services["disk_usage"][$key]["disk_number"],
                                "use" => "generic-service",
                                "check_command" => $checkcommand,
                                "_xiwizard" => $wizard_name,
                            );
                        }

                        break;

                    case "windows_services":
                        if (($services["windows_services"]["monitor"] != "on") || (!array_key_exists("windows_services", $services))) {
                            break;
                        }

                        foreach ($services["windows_services"] as $key => $value) {
                            if ($key === "monitor") {
                                continue;
                            }

                            if (empty($services["windows_services"][$key]["service_name"])) {
                                continue;
                            }

                            $checkcommand = "check_windows_services_by_ssh! ";
                            $checkcommand .= "-u " . $ssh_username . " ";
                            $checkcommand .= "-a '-expectedstate " . $services["windows_services"][$key]["expected_state"] . " -servicename " . $services["windows_services"][$key]["service_name"] . "' ";

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $hostname,
                                "service_description" => $services["windows_services"][$key]["display_name"],
                                "use" => "generic-service",
                                "check_command" => $checkcommand,
                                "_xiwizard" => $wizard_name,
                            );
                        }

                        break;

                    case "memory_usage":
                        if (($services["memory_usage"]["monitor"] != "on") || (!array_key_exists("memory_usage", $services))) {
                            break;
                        }

                        $checkcommand = "check_windows_memory_by_ssh! ";
                        $checkcommand .= "-u " . $ssh_username . " ";
                        $checkcommand .= "-a '-outputType " . $services["memory_usage"]["outputType"] . " -metric " . $services["memory_usage"]["metric"] . " -warning " . $services["memory_usage"]["warning"] . " -critical " . $services["memory_usage"]["critical"] . "' ";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Memory Usage",
                            "use" => "generic-service",
                            "check_command" => $checkcommand,
                            "_xiwizard" => $wizard_name,
                        );

                        break;

                    case "windows_processes":
                        if (($services["windows_processes"]["monitor"] != "on") || (!array_key_exists("windows_processes", $services))) {
                            break;
                        }

                        foreach ($services["windows_processes"] as $key => $value) {
                            if ($key === "monitor") {
                                continue;
                            }

                            if (empty($services["windows_processes"][$key]["process_name"])) {
                                continue;
                            }

                            $checkcommand = "check_windows_processes_by_ssh! ";
                            $checkcommand .= "-u " . $ssh_username . " ";
                            $checkcommand .= "-a '-processname " . $services["windows_processes"][$key]["process_name"] . " -metric " . $services["windows_processes"][$key]["metric"];

                            if ($services["windows_processes"][$key]["metric"] == "Memory") {
                                $checkcommand .= " -outputType " . $services["windows_processes"][$key]["outputType"];
                            }

                            $checkcommand .= " -warning " . $services["windows_processes"][$key]["warning"] . " -critical " . $services["windows_processes"][$key]["critical"] . "' ";

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $hostname,
                                "service_description" => $services["windows_processes"][$key]["display_name"],
                                "use" => "generic-service",
                                "check_command" => $checkcommand,
                                "_xiwizard" => $wizard_name,
                            );
                        }

                        break;

                    default:

                        break;
                }
            }
            // Keep this for debugging
            // echo "<pre>";
            // print_r($objs);
            // echo "</pre>";
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;
            break;

        default:

            break;
    }

    return $output;
}