<?php

/**
 * SSO Tenant Management Functions
 * 
 * Functions for managing OAuth client/tenant configuration
 * 
 * @package SSO
 */

// require_once(dirname(__FILE__) . '/../createprovider.php');
// require_once(dirname(__FILE__) . '/sso-utilities.php');

function get_tenants()
{
    try {
        $tenants = get_tenants_credentials_helper();
        foreach ($tenants as $tenant_name => $tenant) {
            unset($tenants[$tenant_name]['client_secret']);
        }
        echo json_encode(array('status' => 'success', 'data' => $tenants));
        exit();
    } catch (Exception $e) {
    $msg = 'Failed to get tenants.';
    log_sso_error($msg . ' ' . $e->getMessage());
    echo json_encode(array('status' => 'error', 'message' => $msg));
    exit();
    }
}

function validate_aad_credentials($client_id = '', $tenant_id = '', $client_secret = '')
{
    global $request;
    $client_id = grab_request_var('client_id', $client_id);
    $tenant_id = grab_request_var('tenant_id', $tenant_id);
    $client_secret = grab_request_var('client_secret', $client_secret);

    foreach (['client_id', 'tenant_id', 'client_secret'] as $field) {
        if (!$$field) {
            $msg = 'Missing ' . $field . ' in tenant details.';
            log_sso_error($msg);
            echo json_encode(array('status' => 'error', 'message' => $msg));
            exit();
        }
    }

    $provider = getProvider('azure', $params = array(
        'tenantId' => $tenant_id,
        'clientId' => $client_id,
        'clientSecret' => $client_secret
    ));

    $scopes = array('https://graph.microsoft.com/.default');

    try {
        $token = $provider->getAccessToken('client_credentials', ['scope' => implode(' ', $scopes)]);
        if (strlen($token->getToken()) > 0) {
            echo json_encode(array('status' => 'success', 'message' => 'Successfully authenticated with Azure AD.'));
            exit();
        } else {
            log_sso_error('Failed to authenticate with Azure AD.');
            throw new Exception('Failed to authenticate with Azure AD.');
        }
    } catch (Exception $e) {
        log_sso_error($e->getMessage());
        echo json_encode(array('status' => 'error', 'message' => $e->getMessage()));
        exit();
    }
}

function save_tenant()
{
    try {
        $tenant = grab_request_var('tenant', array());
        $tenant = json_decode($tenant, true);
        $client_id = $tenant['client_id'];
        $client_secret = $tenant['client_secret'] ?? '';

        $current_tenants = get_tenants_credentials_helper();
        $current_tenants[$client_id] = array_merge($current_tenants[$client_id] ?? [], $tenant);

        if (!array_key_exists('name', $current_tenants[$client_id]) && ($client_secret ?? false)) { // get tenant name from MSFT AAD
            $more_tenant_details = get_tenant_details_helper($tenant['client_id'], $tenant['tenant_id'], $client_secret);
            foreach ($more_tenant_details as $key => $detail) {
                $current_tenants[$client_id][$key] = $detail;
            }
        }
        if (!array_key_exists('enabled', $current_tenants[$client_id])) {
            $current_tenants[$client_id]['enabled'] = 1;
        }
        // remove empty keys that may have arisen from initial setup - if I had more time, I'd figure out why this occasionally happens, but for now, this is a quick fix
        if (array_key_exists('', $current_tenants)) {
            unset($current_tenants['']);
        }

        $sso_enabled = false;
        foreach($current_tenants as $key => $tenant) {
            if ($tenant['enabled'] == 1) {
                $sso_enabled = true;
                break;
            }
        }

        $tenants_encrypted = encrypt_data(json_encode($current_tenants));

        $file_path = '/usr/local/nagiosxi/etc/components/oauth2/providers/AAD/tenants';
        if (!file_exists(dirname($file_path))) {
            mkdir(dirname($file_path), 0755, true);
        }
        file_put_contents($file_path, $tenants_encrypted);
        set_option('MSFT_SSO', $sso_enabled);        
        echo json_encode(array('status' => 'success', 'message' => 'Successfully saved tenant credentials.'));
        exit();
    } catch (Exception $e) {
        log_sso_error('Failed to save tenant credentials: ' . $e->getMessage());
        echo json_encode(array('status' => 'error', 'message' => 'Failed to save tenant credentials.'));
        exit();
    }
}

// grab tenant details from MSFT AAD via client credentials flow
// currently only returns the tenant name
function get_tenant_details_helper($client_id, $tenant_id, $client_secret)
{
    try {
        if (!file_exists('/usr/local/nagiosxi/etc/components/oauth2/providers/AAD/tenants')) {
            file_put_contents('/usr/local/nagiosxi/etc/components/oauth2/providers/AAD/tenants', encrypt_data(json_encode(array())));
        }

        $provider = getProvider('azure', $params = array(
            'tenantId' => $tenant_id,
            'clientId' => $client_id,
            'clientSecret' => $client_secret
        ));

        $scopes = array('https://graph.microsoft.com/.default');

        $token = $provider->getAccessToken('client_credentials', ['scope' => implode(' ', $scopes)]);
        if (strlen($token->getToken()) > 0) {
            $extra_tenant_details = $provider->request('GET', 'https://graph.microsoft.com/v1.0/applications(appId=\'' . $client_id . '\')?$select=displayName', $token);
            $tenant_details = array(
                'name' => $extra_tenant_details['displayName']
            );
            return $tenant_details;
        } else {
            throw new Exception('Failed to authenticate with Azure AD.');
        }
    } catch (Exception $e) {
        throw new Exception('Failed to get tenant details from Azure AD.');
    }
}

/**
 * Get tenant details from the file -- if client_id is passed, only return the details for that tenant
 */
function get_tenants_credentials_helper($client_id = '')
{
    try {
        if (!file_exists('/usr/local/nagiosxi/etc/components/oauth2/providers/AAD/tenants')) {
            file_put_contents('/usr/local/nagiosxi/etc/components/oauth2/providers/AAD/tenants', encrypt_data(json_encode(array())));
        }

        $tenant_details_encrypted = file_get_contents('/usr/local/nagiosxi/etc/components/oauth2/providers/AAD/tenants');
        $tenant_details = json_decode(decrypt_data($tenant_details_encrypted), true);
        if ($client_id) {
            return $tenant_details[$client_id];
        }
        return $tenant_details;
    } catch (Exception $e) {
        throw new Exception('Failed to get tenant details.');
    }
}

function delete_tenant($client_id)
{
    try {
        $tenants = get_tenants_credentials_helper();
        if (!array_key_exists($client_id, $tenants)) {
            echo json_encode(array('status' => 'error', 'message' => 'Tenant not found.'));
            return;
        }
        unset($tenants[$client_id]);
        $tenants_encrypted = encrypt_data(json_encode($tenants));
        file_put_contents('/usr/local/nagiosxi/etc/components/oauth2/providers/AAD/tenants', $tenants_encrypted);
        delete_option($client_id . '_sso_enabled');
        remove_user_meta_aad_helper_by_client_id($client_id);
        echo json_encode(array('status' => 'success', 'message' => 'Successfully deleted tenant.'));
        exit();
    } catch (Exception $e) {
        log_sso_error('Failed to delete tenant: ' . $e->getMessage());
        echo json_encode(array('status' => 'error', 'message' => 'Something went wrong while deleting the tenant.'));
        exit();
    }
}

