#!/bin/bash -e

#################################################################################################
#
# Purpose:
#
# Takes parameters from the snmpwalk api and starts an snmpwalk job using those parameters
#
# Author: Justin Simon
#
#################################################################################################

# Default values
DEVICE_ADDRESS=""
DEVICE_PORT="161"
SNMP_VERSION="2c"
SNMP_COMMUNITY="public"
OID_SCAN_OPTIONS=""
MIB_SCAN_OPTIONS=""
TIMEOUT="15"
OUTPUT_DIR="/usr/local/nagiosxi/tmp/"

# Parse arguments
for ARG in "$@"; do
    case $ARG in
        --device_address=*)
            DEVICE_ADDRESS="${ARG#*=}"
            ;;
        --device_port=*)
            DEVICE_PORT="${ARG#*=}"
            ;;
        --snmp_version=*)
            SNMP_VERSION="${ARG#*=}"
            ;;
        --snmp_community=*)
            SNMP_COMMUNITY="${ARG#*=}"
            ;;
        --oid_scan_options=*)
            OID_SCAN_OPTIONS="${ARG#*=}"
            ;;
        --timeout=*)
            TIMEOUT="${ARG#*=}"
            ;;
        --max_results=*)
            MAX_RESULTS="${ARG#*=}"
            ;;
        --output_dir=*)
            OUTPUT_DIR="${ARG#*=}"
            ;;
        --mib_scan_options=*)
            MIB_SCAN_OPTIONS="${ARG#*=}"
            ;;
        --v3_security_level=*)
            SECURITY_LEVEL="${ARG#*=}"
            ;;
        --v3_username=*)
            USERNAME="${ARG#*=}"
            ;;
        --v3_auth_password=*)
            AUTH_PASSWORD="${ARG#*=}"
            ;;
        --v3_auth_protocol=*)
            AUTH_PROTOCOL="${ARG#*=}"
            ;;
        --v3_privacy_password=*)
            PRIVACY_PASSWORD="${ARG#*=}"
            ;;
        --v3_privacy_protocol=*)
            PRIVACY_PROTOCOL="${ARG#*=}"
            ;;
        *)
            echo "Unknown option: $ARG"
            exit 1
            ;;
    esac
done

# Check required parameters
if [[ -z "$DEVICE_ADDRESS" ]]; then
    echo "Error: --device_address is required"
    exit 1
fi

# Generate output file path
TIMESTAMP=$(date +%s)
OUTPUT_FILE="${OUTPUT_DIR}/snmpwalk_${DEVICE_ADDRESS}_${TIMESTAMP}.log"
PID_FILE="${OUTPUT_DIR}/snmpwalk_${DEVICE_ADDRESS}_${TIMESTAMP}.pid"

# Build SNMPWALK command based on SNMP version
if [[ "$SNMP_VERSION" == "3" ]]; then
    SNMPWALK_CMD="snmpwalk -v 3 -l $SECURITY_LEVEL -u $USERNAME"

    if [[ "$SECURITY_LEVEL" == "authNoPriv" || "$SECURITY_LEVEL" == "authPriv" ]]; then
        SNMPWALK_CMD="$SNMPWALK_CMD -a $AUTH_PROTOCOL -A $AUTH_PASSWORD"
    fi

    if [[ "$SECURITY_LEVEL" == "authPriv" ]]; then
        SNMPWALK_CMD="$SNMPWALK_CMD -x $PRIVACY_PROTOCOL -X $PRIVACY_PASSWORD"
    fi

    SNMPWALK_CMD="$SNMPWALK_CMD -t $TIMEOUT -r 1 $DEVICE_ADDRESS:$DEVICE_PORT $OID_SCAN_OPTIONS"
else
    # SNMPv1 or v2c
    SNMPWALK_CMD="snmpwalk -v $SNMP_VERSION -c $SNMP_COMMUNITY -t $TIMEOUT -r 1 $DEVICE_ADDRESS:$DEVICE_PORT $OID_SCAN_OPTIONS"
fi

# Sanitize passwords in scan output
# This is a good idea but for now we're getting the community string and passwword value for this file, so probably need to leave it unredacted for now
# SANITIZED_CMD=$(echo "$SNMPWALK_CMD" | \
#   sed -E \
#     -e 's/(-c[[:space:]]+)[^[:space:]]+/\1[REDACTED]/g' \
#     -e 's/(-A[[:space:]]+)[^[:space:]]+/\1[REDACTED]/g' \
#     -e 's/(-X[[:space:]]+)[^[:space:]]+/\1[REDACTED]/g'
# )

# Launch command in background and limit output
(
    if [ -n "$MAX_RESULTS" ] && [ "$MAX_RESULTS" -gt 0 ]; then
        echo "Running: $SNMPWALK_CMD with Max Results set to: $MAX_RESULTS"
        $SNMPWALK_CMD 2>&1 | head -n "$MAX_RESULTS"
    else
        echo "Running: $SNMPWALK_CMD"
        $SNMPWALK_CMD 2>&1
    fi
) > "$OUTPUT_FILE" &

# Save the PID
PID=$!
echo "$PID" > "$PID_FILE"

echo "SNMP walk job started asynchronously."
echo "PID: $PID"
echo "Output will be written to: $OUTPUT_FILE"
echo "PID file: $PID_FILE"

# Detach cleanup watcher in background
nohup bash -c "
    while kill -0 $PID 2>/dev/null; do sleep 1; done
    rm -f \"$PID_FILE\"
" >/dev/null 2>&1 & disown

exit 0
