<?php

require_once(dirname(__FILE__) . '/../../common.inc.php');

// Graphs will not generate if error messaging turned on
ini_set('display_errors', 'off');

// Initialization stuff
pre_init();
init_session();

// Grab GET or POST variables and check pre-reqs
grab_request_vars();
check_prereqs();
check_authentication(false);

route_request();

// Comparison functions for use in usort - prob a nicer way to do this
function cmp_ok($a, $b) {
    return $b['ok'] - $a['ok'];
}

function cmp_warning($a, $b) {
    return $b['warning'] - $a['warning'];
}

function cmp_critical($a, $b) {
    return $b['critical'] - $a['critical'];
}

function cmp_unknown($a, $b) {
    return $b['unknown'] - $a['unknown'];
}

function route_request() {
    $mode = grab_request_var("mode", "");
    switch ($mode) {
        case "getbetterchart":
            get_better_highcharts();
            break;    
        case "coolmodeactivated":
            get_cool_highcharts();
            break;
        default:
            break;
    }
}

// Core function that defines $output, building the actual highchart
function build_highchart($services, $host_names, $link_labels, $title, $subtitle = "") {
    
    $divId = grab_request_var("divId", "");
    $width = grab_request_var("width", 400);
    $height = grab_request_var("height", 300) - 30;
    $charttype = grab_request_var("graphChoiceSelect", "column");
    $animation = grab_request_var("animation", "false");

    if ($title == "" && ($charttype == "pie" || $charttype == "donut")){
        $title = "All Hosts";
    }
    
    $oks = $services[0];
    $warnings = $services[1];
    $criticals = $services[2];
    $unknowns = $services[3];

    $donut = 0;
    $radial = 0;
    $stacked = 0;

    if ($charttype == 'donut') {
        $charttype = 'pie';
        $donut = 1;
    }

    else if ($charttype == 'radial') {
        $charttype = 'column';
        $radial = 1;
    }

    else if ($charttype == 'barstacked') {
        $charttype = 'bar';
        $stacked = 1;
    }

    else if ($charttype == 'columnstacked') {
        $charttype = 'column';
        $stacked = 1;
    }

    $color = "#000000";
    $gridLineColor = "var(--border)";
    $colors = '["#b2ff5f", "#FEFF5F", "#FFC45F", "#FF795F"]';
    $theme = get_theme();
    if ($theme == "xi5dark") {
        $color = '#EEEEEE';
    }
    else if ($theme == "colorblind") {
        $colors = '["#56B4E9", "#D55E00", "#F0E442", "#CC79A7"]';
    }
    else if (is_neptune()) {
        // $color = "#e1e7ef";
        $color = 'var(--foreground)';
        // colors needs to be changed below
        $colors =  '["#52ef5a", "#eaff05", "#eba71e" , "#f0050e"]';
    }

    // Base chart options regardless of type
    $output = '<script type="text/javascript">
    Highcharts.setOptions({
        exporting: {
            buttons: {
                contextButton: {
                    menuItems: [
                        "downloadPNG",
                        "downloadJPEG",
                        "downloadPDF",
                        "downloadSVG",
                        "separator",
                        "downloadCSV",
                        "downloadXLS"
                    ]
                }
            }
        },
        colors: ' . $colors . '
    });
    var chart_' . $divId . ';
    var options = {
        legend: {
            backgroundColor: "transparent"
        },
        exporting: {
            buttons: {
                contextButton: {
                    menuItems: [
                        "printChart",
                        "separator",
                        "downloadPNG",
                        "downloadJPEG",
                        "downloadPDF",
                        "downloadSVG",
                        "separator",
                        "downloadCSV",
                        "downloadXLS"
                    ]
                }
            }
        },

        chart: {
            backgroundColor: "transparent",
            renderTo: "' . $divId . '",
            width: ' . $width . ',
            height: ' . $height . ',
            type: "'. $charttype . '"';

    if ($radial) {
        $output .= 
        ', 
        polar: true,
        inverted: true';
    }
    $output .= 
        '},
        credits: {
            enabled: false
        },
        title: {
            text: "' . $title . '"
        },
        subtitle: {
            text: "' . $subtitle . '"
        },';
    if ($stacked) {
        $output .= '
        plotOptions: {
            series: {
                stacking: "normal"
            }
        },';
    }
    // Organize data to fit pie / donut charts
    if ($charttype == 'pie') {

        $data = array(0, 0, 0, 0);
        $num_services = array_sum($oks) + array_sum($warnings) + array_sum($criticals) + array_sum($unknowns);
        $num_hosts = count($host_names);
        
        for ($x = 0; $x < $num_hosts; $x++) {
            $data[0] += $oks[$x];
            $data[1] += $warnings[$x];
            $data[2] += $unknowns[$x];
            $data[3] += $criticals[$x];
        }

        for ($x = 0; $x < 4; $x++) {
            if ($num_services > 0) {
                $data[$x] = $data[$x] / $num_services * 100;
            } else {
                $data[$x] = 0;
            }
        }

        $formatted_data = array(
            array("name" => "Ok", "y" => round($data[0], 3)),
            array("name" => "Warning", "y" => round($data[1], 3)),
            array("name" => "Unknown", "y" => round($data[2], 3)),
            array("name" => "Critical", "y" => round($data[3], 3))
        );
        
        $output .= 
        'plotOptions: {
            pie: {
                dataLabels: {
                    useHTML: true,
                    enabled: false,
                    color: "'.$color.'",
                    connectorColor: "'.$color.'",
                    format: "{point.name} {point.percentage:.2f}%",
                    distance: 10
                }
            }
        },
        series: [{';
            
        // Add innersize if donut chart
        if ($donut) {
            $output .= 
            'innerSize: "65%",';
        }
            
        $output .= 
        'data: '.json_encode($formatted_data).',
        animation: '.($animation).',
        borderWidth: 0
        }]';

    // If not pie / donut, use data as is - sort according to user's selection
    } else {
        // Encode arrays to be used in highchart
        $json_host_names = json_encode($host_names);
        $json_oks = json_encode($oks);
        $json_warnings = json_encode($warnings);
        $json_criticals = json_encode($criticals);
        $json_unknowns = json_encode($unknowns);
        $output .=
            'plotOptions: {
                series: {
                    borderWidth: 0,
                    borderRadius: 1.5
                }
            },
            xAxis: {
                categories: ' . $json_host_names . ',
                title: {
                    text: null
                },
                gridLineWidth: 0,
                lineWidth: 0,
                ';
        if ($link_labels){
            $base_url = '/nagiosxi/includes/components/xicore/status.php?show=';
            if (grab_request_var("hostgroupChoiceSelect", "default") == "all_hostgroups" ||
                grab_request_var("hostgroup", "") != "" && strstr(grab_request_var("hostgroup", ""), ",")) {
                $base_url .= 'hosts&hostgroup=';
            }
            else if (grab_request_var("servicegroup", "") != "" && strstr(grab_request_var("servicegroup", ""), ",")) {
                $base_url .= 'services&servicegroup=';
            } else {
                $base_url .= 'hostdetail&host=';
            }
            $output .= 
                'labels: {
                    formatter: function () {
                        return "<a href=' . $base_url . '" + this.value + " target=' . '_blank' . '>" + this.value + "</a>"
                    },
                    useHTML: true
                }';
        }
        $output .= 
            '},
            yAxis: {
                title: {
                    text: null
                },
                gridLineColor: "'.$gridLineColor.'"
            },
            series: [{
                name: "Ok",
                data: ' . $json_oks . ',
                animation: '.($animation).',
                groupPadding: 0.0
            }, {
                name: "Warning",
                data: ' . $json_warnings . ',
                animation: '.($animation).',
                groupPadding: 0.0
            }, {
                name: "Unknown",
                data: ' . $json_unknowns . ',
                animation: '.($animation).',
                groupPadding: 0.0
            }, {
                name: "Critical",
                data: ' . $json_criticals . ',
                animation: '.($animation).',
                groupPadding: 0.0
            }]';
    }
    $output .= '
    }
    chart_' . $divId . ' = new Highcharts.Chart(options);
    </script>';
    return $output;
}

function get_host_data(&$host_names) {

    $num_hosts = sizeof($host_names);
    $sorted = array();
	$service_statuses = array(); // 2D array of service states indexed by hosts
	
	for ($x = 0; $x < $num_hosts; $x++) {
		$services_by_name = get_data_service_status(array('host_name' => $host_names[$x])); // slow, could be improved
		$temp = array();
		for ($y = 0; $y < sizeof($services_by_name); $y++) {
			array_push($temp, $services_by_name[$y]['current_state']);
		}
        $ok = count(array_keys($temp, 0));
		$warning = count(array_keys($temp, 1));
		$critical = count(array_keys($temp, 2));
		$unknown = count(array_keys($temp, 3));
        $sort_array = array('name' => $host_names[$x], 'ok' => $ok, 'warning' => $warning, 'critical' => $critical, 'unknown' => $unknown);
        array_push($sorted, $sort_array);
	}
    return $sorted;
}

function get_hostgroup_data($hostgroups, &$hostgroup_names) {
    $sorted = array();
    $hostgroup_ids_str = "";
    $y = 0;
    $ag = [];
    if (is_array($hostgroups)) {
        foreach ($hostgroups as $hg) {
            $hgid = get_hostgroup_id($hg);
            if ($y > 0)
                $hostgroup_ids_str .= ",";
            $hostgroup_ids_str .= $hgid;
            $y++;
        }
        $ag["hostgroup_id"] = "in:" . $hostgroup_ids_str;
    }

    $xmlhg = get_xml_hostgroup_objects($ag);
    foreach($xmlhg->hostgroup as $hostgroup) {
        $ok = 0;
        $warning = 0;
        $critical = 0;
        $unknown = 0;
        $hargs = [];
        $hargs["hostgroup_name"] = strval($hostgroup->hostgroup_name);
        $hxml = get_xml_hostgroup_member_objects($hargs);
        foreach($hxml->hostgroup->members->host as $host) {
            $temp = array();
            $services_by_name = get_data_service_status(array('host_name' => strval($host->host_name)));
            for ($y = 0; $y < sizeof($services_by_name); $y++) {
                array_push($temp, $services_by_name[$y]['current_state']);    
            }
            $ok += count(array_keys($temp, 0));
            $warning += count(array_keys($temp, 1));
            $critical += count(array_keys($temp, 2));
            $unknown += count(array_keys($temp, 3));
        }
        $sort_array = array('name' => strval($hostgroup->hostgroup_name), 'ok' => $ok, 'warning' => $warning, 'critical' => $critical, 'unknown' => $unknown);
        array_push($sorted, $sort_array);
        array_push($hostgroup_names, strval($hostgroup->hostgroup_name));
    }

    return $sorted;
}

// Gets and organizes internal data for use in a custom dashlet
function get_better_highcharts() {

    $sort_choice = grab_request_var("sortChoiceSelect", "");
    $host_choice = grab_request_var("hostChoiceSelect", "");

    // This is tragically required for backwards compatability
    $hostgroup_choice = grab_request_var("hostgroupChoiceSelect", "");

    // New vars to support limiting by specific hostgroups / servicegroups
    $hostgroup = grab_request_var("hostgroup", "");
    $servicegroup = grab_request_var("servicegroup", "");
    $group = grab_request_var("group_select", "");

    // New var for limiting to specific host
    $host = grab_request_var("host", "");

    $unselected_output = "";
    if ($group == "all") {
        $host_choice = "total";
    } else if ($group == "host" && !$host) {
        print _("You have chosen to limit by host(s) but did not select any host(s).");
        die();
    } else if ($group == "hostgroup" && !$hostgroup) {
        print _("You have chosen to limit by hostgroup(s) but did not select any hostgroup(s).");
        die();
    } else if ($group == "servicegroup" && !$servicegroup) {
        print _("You have chosen to limit by servicegroup(s) but did not select any servicegroup(s).");
        die();
    } else if ($hostgroup) {
        $hostgroup_choice = $hostgroup;
    } else if ($host) {
        $host_choice = $host;
    }

    $title = "";

    $names = array();
    $sorted = array();

    if ($hostgroup_choice == "all_hostgroups") {
        $title = "All Hostgroups";
        $sorted = get_hostgroup_data($hostgroup, $names);
    } else if ($hostgroup_choice) {
        $title = $hostgroup;
        if (strstr($hostgroup, ",")) {
            $hostgroups = explode(", ", $hostgroup);
            $sorted = get_hostgroup_data($hostgroups, $names);
        } else {
            $hargs = [];
            $hargs["hostgroup_name"] = strval($hostgroup);
            $hxml = get_xml_hostgroup_member_objects($hargs);
            foreach ($hxml->hostgroup->members->host as $host) {
                $names[] = strval($host->host_name);
            }
            $sorted = get_host_data($names);
        }
    } else if ($servicegroup) {
        $title = $servicegroup;
        $single_servicegroup = false;
        if (!strstr($servicegroup, ",")) {
            $single_servicegroup = true;
        }
        $servicegroups = explode(", ", $servicegroup);
        foreach ($servicegroups as $servicegroup) {
            $service_ids = [];
            $service_ids_str = "";
            $service_ids = get_servicegroup_member_ids($servicegroup);
            $y = 0;
            foreach ($service_ids as $sid) {
                if ($y > 0)
                    $service_ids_str .= ",";
                $service_ids_str .= $sid;
                $y++;
            }
            $backendargs = array();
            $backendargs["cmd"] = "getservicestatus";
            $backendargs["limitrecords"] = false; // don't limit records
            $backendargs["combinedhost"] = true; // get host status too
            $backendargs["brevity"] = 1; // we don't need everything
            // service id limiters
            if ($service_ids_str != "") {
                $backendargs["service_id"] = "in:" . $service_ids_str;
            }
            // order by host name, service description
            $backendargs["orderby"] = "host_name:a,service_description:a";

            $temp = [0, 0, 0, 0];
            $lasthost = "";
            $xml = get_xml_service_status($backendargs);
            $count = 1;
            foreach ($xml->servicestatus as $ss) {
                $servicename = strval($ss->name); 
                if ($single_servicegroup) { // Organize data by host names
                    $thishost = strval($ss->host_name);
                    if ($thishost != $lasthost || $count == $xml->recordcount) {
                        if ($lasthost != "") {
                            $sorted[] = array(
                                "name" => $lasthost,
                                "ok" => $temp[0],
                                "warning" => $temp[1],
                                "critical" => $temp[2],
                                "unknown" => $temp[3]
                            );
                            $names[] = $lasthost;
                            $temp = [0, 0, 0, 0];
                        } 
                    }
                    $lasthost = $thishost;
                    $temp[intval($ss->current_state)]++;
                } else { // Organize data by servicegroup
                    $temp[intval($ss->current_state)]++;
                }
                $count++;
            }
            if (!$single_servicegroup) {
                $sorted[] = array(
                    "name" => $servicegroup,
                    "ok" => $temp[0],
                    "warning" => $temp[1],
                    "critical" => $temp[2],
                    "unknown" => $temp[3]
                );
                $names[] = $servicegroup;
                $temp = [0, 0, 0, 0];
            }
        }
    } else if ($host_choice == "total") {
        $title = "All Hosts";
        $all_hosts = get_data_host(array());
        foreach($all_hosts as $host) {
            array_push($names, $host['host_name']);
        }
        $sorted = get_host_data($names);
    } else if ($host_choice) {
        $title = $host;
        $names = array($host);
        $sorted = get_host_data($names);
    } else {
        $output = _("No data to display.");
        print $output;
        die();
    }

    $num_hosts = count($names);

	// Organize the number of ok, warning, and critical services into arrays indexed by hosts
	$oks = array();
	$warnings = array();
	$criticals = array();
	$unknowns = array();
    
    switch($sort_choice) {
        case('total_ok'):
            usort($sorted, "cmp_ok");
            break;
        case('total_warning'):
            usort($sorted, "cmp_warning");
            break;
        case('total_unknown'):
            usort($sorted, "cmp_unknown");
            break;
        case('total_critical'):
            usort($sorted, "cmp_critical");
            break;
        default:
            break;
    }
   
    for ($x = 0; $x < $num_hosts; $x++) {
		array_push($oks, $sorted[$x]['ok']);
		array_push($warnings, $sorted[$x]['warning']);
		array_push($criticals, $sorted[$x]['critical']);
		array_push($unknowns, $sorted[$x]['unknown']);
	}

    // This block exists solely because it felt weird to pass this many variables to a function
    $service_arr = array();
    array_push($service_arr, $oks);
    array_push($service_arr, $warnings);
    array_push($service_arr, $criticals);
    array_push($service_arr, $unknowns);
    
    $output = build_highchart($service_arr, $names, 1, $title);
    print $output;
    die();
}

// Hard-codes fake data for use in the cool dashlets for the cool dashboard
function get_cool_highcharts() {

    $iteration = grab_request_var("coolMode", "");
    $service_arr = array();
    $host_names = array();
    $title = "";
    $subtitle = "";

    if ($iteration == 1) {
        $service_arr = array(
            array(10, 4, 3, 1, 11, 12, 5, 7, 6, 15, 12, 2, 9, 4, 7),
            array(4, 1, 2, 3, 1, 5, 4, 3, 0, 0, 5, 5, 1, 6, 1),
            array(2, 3, 4, 2, 1, 0, 0, 0, 3, 1, 1, 0, 2, 6, 7),
            array(1, 0, 0, 1, 0, 0, 1, 0, 1, 0, 2, 1, 3, 0, 0)
        );
        $host_names = array("Host1", "Host2", "Host3", "Host4", "Host5", "Host6", "Host7", "Host8", "Host9", "Host10", "Host11", "Host12", "Host13", "Host14", "Host15");
    }

    else if ($iteration == 2) {
        $service_arr = array(
            array(10, 4, 3, 1, 11, 12, 5, 7, 6, 15, 12, 2, 9, 4, 7),
            array(4, 1, 2, 3, 1, 5, 4, 3, 0, 0, 5, 5, 1, 6, 1),
            array(7, 6, 4, 3, 3, 2, 2, 2, 1, 1, 1, 0, 0, 0, 0),
            array(1, 0, 0, 1, 0, 0, 1, 0, 1, 0, 2, 1, 3, 0, 0)
        );
        $host_names = array("Host1", "Host2", "Host3", "Host4", "Host5", "Host6", "Host7", "Host8", "Host9", "Host10", "Host11", "Host12", "Host13", "Host14", "Host15");
    }

    else if ($iteration == 3) {
        $service_arr = array(
            array(10),
            array(4),
            array(2),
            array(1)
        );
        $host_names = array("Host1");
        $title = "Host7";
    }

    else if ($iteration == 4) {
        $service_arr = array(
            array(7, 2, 3, 1, 11, 12, 5, 7, 9, 15, 12, 22, 9, 4, 7),
            array(4, 1, 2, 3, 12, 5, 4, 3, 4, 0, 5, 5, 1, 6, 1),
            array(2, 3, 4, 2, 1, 0, 0, 2, 3, 1, 1, 0, 2, 6, 7),
            array(1, 0, 0, 1, 0, 0, 1, 1, 1, 0, 2, 1, 3, 0, 1)
        );
        $host_names = array("Host1", "Host2", "Host3", "Host4", "Host5", "Host6", "Host7", "Host8", "Host9", "Host10", "Host11", "Host12", "Host13", "Host14", "Host15");
        $title = "All Hosts";
    }

    else if ($iteration == 5) {
        $service_arr = array(
            array(7, 9, 8),
            array(4, 1, 2),
            array(2, 3, 4),
            array(1, 0, 0)
        );
        $host_names = array("Group1", "Group2", "Group3");
    }

    else if ($iteration == 6) {
        $service_arr = array(
            array(10, 4, 3, 12, 10, 12, 5, 10, 6, 15, 12, 18, 9, 4, 7, 4, 19, 18, 1, 10, 11, 3, 4, 6, 17, 10),
            array(4, 1, 2, 2, 1, 5, 4, 3, 8, 0, 5, 5, 1, 6, 1, 6, 5, 3, 2, 1, 6, 7, 8, 0, 0, 1),
            array(7, 6, 1, 3, 6, 2, 2, 1, 1, 1, 1, 0, 0, 0, 0, 2, 5, 6, 1, 2, 7, 8, 1, 0, 1, 1),
            array(1, 0, 0, 1, 0, 0, 1, 0, 1, 0, 2, 1, 3, 0, 0, 1, 2, 4, 0, 0, 1, 0, 0, 0, 1, 0)
        );
        $host_names = array("A", "B", "C", "D", "E", "F", "G", "H", "I", "J", "K", "L", "M", "N", "O", "P", "Q", "R",
            "S", "T", "U", "V", "W", "X", "Y", "Z");
    }
    
    $output = build_highchart($service_arr, $host_names, 0, $title, $subtitle);
    print $output;
    die();
}
