<?php
//
// OpenAI Usage Wizard
// Copyright (c) 2023-2023 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

openai_usage_configwizard_init();

// This function is used to ensure the input API key is valid by making a request to the OpenAI API
function is_valid_admin_key(string $apiKey){
    $startTime = time(); // current UNIX timestamp
    $url = "https://api.openai.com/v1/organization/costs?start_time=$startTime";

    $ch = curl_init($url);

    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HTTPHEADER => [
            "Authorization: Bearer $apiKey"
        ],
        CURLOPT_TIMEOUT => 10
    ]);

    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);

    curl_close($ch);

    if ($curlError) {
        return false;
    }

    return $httpCode == 200;
}

function openai_usage_configwizard_init()
{
    $name = "openai_usage";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.0.0",
        CONFIGWIZARD_IS_PREMIUM => TRUE,
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor your usage on OpenAI."),
        CONFIGWIZARD_DISPLAYTITLE => _("OpenAI Usage"),
        CONFIGWIZARD_FUNCTION => "openai_usage_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "openai.png",
        CONFIGWIZARD_FILTER_GROUPS => array(''),
        CONFIGWIZARD_REQUIRES_VERSION => 60030
    );
    register_configwizard($name, $args);
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function openai_usage_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    // Grab core wizard info
    $wizard_name = grab_array_var($inargs, "wizard");
    $wiz_args = get_configwizard_by_name($wizard_name);
    $wizard_title = isset($wiz_args[CONFIGWIZARD_DISPLAYTITLE]) ? $wiz_args[CONFIGWIZARD_DISPLAYTITLE] : '';

    // Gotta have a sesh
    if (!isset($_SESSION[$wizard_name])) {
        $_SESSION[$wizard_name] = [];
    }

    // Give session a simple name for convenience
    $sess =& $_SESSION[$wizard_name];

    // $inargs keys that needn't/can't be encoded with htmlentities()
    $noEncodeList = [];

    // Array elements that may have user macros
    $userMacroKeys = [];

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    $services_default = [
        "total_cost" => [
            'monitor' => 'on',
            'warning' => '',
            'critical' => '',
            'time_units' => 1,
            'time_unit' => 'day',
        ],

        "requests" => [
            'monitor' => 'on',
            'warning' => '',
            'critical' => '',
            'time_units' => 1,
            'time_unit' => 'day',
        ],

        "input_tokens" => [
            'monitor' => 'on',
            'warning' => '',
            'critical' => '',
            'time_units' => 1,
            'time_unit' => 'day',
        ],

        "cached_input_tokens" => [
            'monitor' => 'on',
            'warning' => '',
            'critical' => '',
            'time_units' => 1,
            'time_unit' => 'day',
        ],

        "output_tokens" => [
            'monitor' => 'on',
            'warning' => '',
            'critical' => '',
            'time_units' => 1,
            'time_unit' => 'day',
        ],

    ];

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            // Clear session if this is the first wizard run
            if ($_POST == []) {
                unset($_SESSION[$wizard_name]);
                $_SESSION[$wizard_name] = [];
                $sess =& $_SESSION[$wizard_name];
            }

            // Session page numbers are used to determine direction to prevent validation onback button
            $sess['page'] = 1;
            encode_form_all($inargs, $noEncodeList); // Encode all user vars


            // Remove values from session that aren't applicable here (for back-button response)
            $retain = [];
            // clear_sess_vars($sess, $retain);


            $hostname = grab_in_var($inargs, "hostname", "", $sess);
            $admin_api_key = grab_in_var($inargs, "api_key", "", $sess);

            // Retain Step 2 data after back button
            $services = grab_in_var($inargs, "services", $services_default, $sess);

            ob_start();
            include __DIR__ . '/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:
            $hostname = grab_in_var($inargs, "hostname", "", $sess);
            $admin_api_key = grab_in_var($inargs, "api_key", "", $sess);

            $errors = [];

            // Only validate if going forward. No need to do so going backward.
            if ($sess['page'] == 1) {
                if (have_value($hostname) == false) {
                    $errors[] = _("No Hostname specified.");
                }
                if (is_valid_host_name($hostname) == false) {
                    $errmsg[] = "Invalid host name.";
                }
                if (have_value($admin_api_key) == false) {
                    $errors[] = _("No Admin API Key specified.");
                }
                else if (!is_valid_admin_key($admin_api_key)) {
                    $errors[] = _("Invalid Admin API Key.");
                }        

            }

            if (count($errors)) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errors;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:
            $sess['page'] = 2;
            encode_form_all($inargs, $noEncodeList); // Encode all user vars

            $hostname = grab_in_var($inargs, "hostname", "", $sess);
            $admin_api_key = grab_in_var($inargs, "api_key", "", $sess);
            $services = grab_in_var($inargs, "services", $services_default, $sess);

            ob_start();
            include __DIR__ . '/steps/step2.php';
            $output = ob_get_clean();
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:
            $hostname = grab_in_var($inargs, "hostname", "", $sess);
            $admin_api_key = grab_in_var($inargs, "api_key", "", $sess);
            $services = grab_in_var($inargs, "services", "", $sess);

            $errors = [];

            // Only validate if going forward. No need to do so going backward.
            if ($sess['page'] == 2) {

                $has_service = false;
                foreach($services as $service) {
                    if (isset($service['monitor']) && $service['monitor'] == 'on') {
                        $has_service = true;
                        break;
                    }
                }
                if (!$has_service) {
                    $errors[] = _("You must select at least one service for monitoring.");
                }

                foreach ($services as $service => $args) {
                    // Make key pretty for use in warnings
                    $service_pretty = ucwords(str_replace('_', ' ', $service));
                    if (isset($args["monitor"]) && $args["monitor"] == "on") {

                        if (!have_value($args['time_units'])) {
                            $errors[] = _("No Time Units specified.");
                        }
                        else if (!is_numeric($args['time_units'])) {
                            $errors[] = _("Time Units must be numeric.");
                        }
                        else if (($args['time_units']) < 1) {
                            $errors[] = _("Time Units must be positive.");
                        }
        
                        if ((!is_numeric($args["warning"]) && $args["warning"] != "") || (!is_numeric($args["critical"]) && $args["critical"] != "")) {
                            $errors[] = _("$service_pretty: Warning and Critical values must be numeric.");
                        }
                        else if (($args["warning"] < 0 && $args["warning"] != "") || ($args["critical"] < 0 && $args["critical"] != "")) {
                            $errors[] = _("$service_pretty: Warning and Critical values must be positive.");
                        }

                        if ($service == "total_cost" && $args["time_unit"] == "day" && $args["time_units"] > 180 || $args["time_unit"] == "week" && $args["time_units"] > 25 || $args["time_unit"] == "month" && $args["time_units"] > 6) {
                            $errors[] = _("$service_pretty: Time Units must be less than or equal to 180 days.");
                        }
                        else if ($service != "total_cost" && $args["time_unit"] == "day" && $args["time_units"] > 31 || $args["time_unit"] == "week" && $args["time_units"] > 4) {
                            $errors[] = _("$service_pretty: Time Units must be less than or equal to 31 days.");
                        }
                        
                    }
                }
            }

            if (count($errors)) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errors;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE3HTML:
            $sess['page'] = 3;
            encode_form_all($inargs, $noEncodeList); // Encode all user vars

            $hostname = grab_in_var($inargs, "hostname", "", $sess);
            $api_key = grab_in_var($inargs, "api_key", "", $sess);
            $services = grab_in_var($inargs, "services", "", $sess);

            $output = '';

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:

            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:
            $hostname = encode_form_val(grab_in_var($inargs, "hostname", "", $sess));
            $api_key = encode_form_val(grab_in_var($inargs, "api_key", "", $sess));
            $services = grab_in_var($inargs, "services", "", $sess);

            foreach ($services as $service => $args) {
                if ($args["monitor"] == "on") {
                    if ($args["time_unit"] != "day" || $args["time_unit"] != "week" || $args["time_unit"] != "month") {
                        $args["time_unit"] = "day";
                    }
                    if (!is_numeric($args["time_units"])){
                        $args["time_units"] = 1;
                    }
                    if (!is_numeric($args["warning"])){
                        $args["warning"] = "";
                    }
                    if (!is_numeric($args["critical"])){
                        $args["critical"] = "";
                    }
                }
            }


            $meta_arr = array();
            $meta_arr["hostname"] = $hostname;
            $meta_arr["api_key"] = $api_key;
            $meta_arr["services"] = $services;
            save_configwizard_object_meta($wizard_name, $hostname, "", $meta_arr);

            $objs = array();

            if (!host_exists($hostname)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_linuxserver_host",
                    "host_name" => $hostname,
                    "address" => 'openai.com',
                    "icon_image" => "openai.png",
                    "statusmap_image" => "openai.png",
                    "_xiwizard" => $wizard_name,
                );
            }

            $service_definitions = [
                "total_cost" => ["description" => "Total Cost", "command" => "check_openai!-C COST"],
                "requests" => ["description" => "Requests", "command" => "check_openai!-C NUM_REQUESTS"],
                "input_tokens" => ["description" => "Input Tokens", "command" => "check_openai!-C INPUT_TOKENS"],
                "cached_input_tokens" => ["description" => "Cached Input Tokens", "command" => "check_openai!-C CACHED_TOKENS"],
                "output_tokens" => ["description" => "Output Tokens", "command" => "check_openai!-C OUTPUT_TOKENS"]
            ];

            foreach ($services as $service => $args) {
                if ($args["monitor"] == "on"){
                    $objs[] = array(
                        "type" => OBJECTTYPE_SERVICE,
                        "host_name" => $hostname,
                        "service_description" => $service_definitions[$service]["description"],
                        "check_command" => $service_definitions[$service]["command"] . " -k " . $api_key . " -w " . $args["warning"] . " -c " . $args["critical"] . " -t " . $args["time_units"] . " -p " . $args["time_unit"],
                        "use" => "xiwizard_generic_service",
                        "_xiwizard" => $wizard_name,
                    );
                }
            }

            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;
            break;

        default:

            break;
    }

    return $output;
}
