<?php
//
// Email Delivery Config Wizard
// Copyright (c) 2010-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

email_delivery_configwizard_init();

function email_delivery_configwizard_init()
{
    $name = "email-delivery";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "3.0.1",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Test mail servers reception and simulated users inspection of email messages."),
        CONFIGWIZARD_DISPLAYTITLE => _("Email Delivery"),
        CONFIGWIZARD_FUNCTION => "email_delivery_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "email-delivery.png",
        CONFIGWIZARD_FILTER_GROUPS => array('email'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );
    register_configwizard($name, $args);
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function email_delivery_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    $wizard_name = "email-delivery";

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            $address = grab_array_var($inargs, "ip_address", "");

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address", "");
            $address_replaced = nagiosccm_replace_user_macros($address);

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (have_value($address_replaced) == false) {
                $errmsg[$errors++] = _("No address specified.");
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");

            $ha = @gethostbyaddr($address);

            if (empty($ha)) {
                $ha = $address;
            }

            $hostname = grab_array_var($inargs, "hostname", $ha);

            $servicedesc = grab_array_var($inargs, "servicedesc", 'Email Delivery');
            $smtpto = grab_array_var($inargs, "smtpto");
            $smtpfrom = grab_array_var($inargs, "smtpfrom");
            $smtp_address = grab_array_var($inargs, "smtp_address");
            $smtp_username = grab_array_var($inargs, "smtp_username");
            $smtp_password = grab_array_var($inargs, "smtp_password");
            $smtp_port = grab_array_var($inargs, "smtp_port", '25');
            $smtp_tls = grab_array_var($inargs, "smtp_tls");
            $imap_address = grab_array_var($inargs, "imap_address");
            $imap_username = grab_array_var($inargs, "imap_username");
            $imap_password = grab_array_var($inargs, "imap_password");
            $imap_port = grab_array_var($inargs, "imap_port", '143');
            $imap_ssl = grab_array_var($inargs, "imap_ssl");

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $hostname_replaced = nagiosccm_replace_user_macros($hostname);
            $servicedesc = grab_array_var($inargs, "servicedesc");
            $smtpto = grab_array_var($inargs, "smtpto");
            $smtpfrom = grab_array_var($inargs, "smtpfrom");
            $smtp_address = grab_array_var($inargs, "smtp_address");
            $smtp_username = grab_array_var($inargs, "smtp_username");
            $smtp_password = grab_array_var($inargs, "smtp_password");
            $smtp_port = grab_array_var($inargs, "smtp_port");
            $smtp_tls = grab_array_var($inargs, "smtp_tls");
            $imap_address = grab_array_var($inargs, "imap_address");
            $imap_username = grab_array_var($inargs, "imap_username");
            $imap_password = grab_array_var($inargs, "imap_password");
            $imap_port = grab_array_var($inargs, "imap_port");
            $imap_ssl = grab_array_var($inargs, "imap_ssl");

            // check for errors
            $errors = 0;
            $errmsg = array();

            if (is_valid_host_name($hostname_replaced) == false)
                $errmsg[$errors++] = _("Invalid host name.");
            if (is_valid_service_name($servicedesc) == false)
                $errmsg[$errors++] = _("Invalid service description.");
            if ($smtp_address == "")
                $errmsg[$errors++] = _("SMTP address cannot be blank.");
            if ($smtp_port == "")
                $errmsg[$errors++] = _("SMTP port cannot be blank.");
            if ($imap_address == "")
                $errmsg[$errors++] = _("IMAP address cannot be blank.");
            if ($imap_username == "")
                $errmsg[$errors++] = _("IMAP username cannot be blank.");
            if ($imap_password == "")
                $errmsg[$errors++] = _("IMAP password cannot be blank.");
            if ($imap_port == "")
                $errmsg[$errors++] = _("IMAP port cannot be blank.");

            if ($smtpto == "")
                $errmsg[$errors++] = _("To address cannot be blank.");
            if ($smtpfrom == "")
                $errmsg[$errors++] = _("From address cannot be blank.");

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $servicedesc = grab_array_var($inargs, "servicedesc");
            $smtpto = grab_array_var($inargs, "smtpto");
            $smtpfrom = grab_array_var($inargs, "smtpfrom");
            $smtp_address = grab_array_var($inargs, "smtp_address");
            $smtp_username = grab_array_var($inargs, "smtp_username");
            $smtp_password = grab_array_var($inargs, "smtp_password");
            $smtp_port = grab_array_var($inargs, "smtp_port");
            $smtp_tls = grab_array_var($inargs, "smtp_tls");
            $imap_address = grab_array_var($inargs, "imap_address");
            $imap_username = grab_array_var($inargs, "imap_username");
            $imap_password = grab_array_var($inargs, "imap_password");
            $imap_port = grab_array_var($inargs, "imap_port");
            $imap_ssl = grab_array_var($inargs, "imap_ssl");

            $output = '

        <input type="hidden" name="ip_address" value="' . encode_form_val($address) . '">
        <input type="hidden" name="hostname" value="' . encode_form_val($hostname) . '">
        <input type="hidden" name="servicedesc" value="' . encode_form_val($servicedesc) . '">
        <input type="hidden" name="smtpto" value="' . encode_form_val($smtpto) . '">
        <input type="hidden" name="smtpfrom" value="' . encode_form_val($smtpfrom) . '">
        <input type="hidden" name="smtp_address" value="' . encode_form_val($smtp_address) . '">
        <input type="hidden" name="smtp_username" value="' . encode_form_val($smtp_username) . '">
        <input type="hidden" name="smtp_password" value="' . encode_form_val($smtp_password) . '">
        <input type="hidden" name="smtp_port" value="' . encode_form_val($smtp_port) . '">
        <input type="hidden" name="smtp_tls" value="' . encode_form_val($smtp_tls) . '">
        <input type="hidden" name="imap_address" value="' . encode_form_val($imap_address) . '">
        <input type="hidden" name="imap_username" value="' . encode_form_val($imap_username) . '">
        <input type="hidden" name="imap_password" value="' . encode_form_val($imap_password) . '">
        <input type="hidden" name="imap_port" value="' . encode_form_val($imap_port) . '">
        <input type="hidden" name="imap_ssl" value="' . encode_form_val($imap_ssl) . '">

            ';
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:


            $output = '<p></p>';
            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $servicedesc = grab_array_var($inargs, "servicedesc");
            $smtpto = grab_array_var($inargs, "smtpto");
            $smtpfrom = grab_array_var($inargs, "smtpfrom");
            $smtp_address = grab_array_var($inargs, "smtp_address");
            $smtp_username = grab_array_var($inargs, "smtp_username");
            $smtp_password = grab_array_var($inargs, "smtp_password");
            $smtp_port = grab_array_var($inargs, "smtp_port");
            $smtp_tls = grab_array_var($inargs, "smtp_tls");
            $imap_address = grab_array_var($inargs, "imap_address");
            $imap_username = grab_array_var($inargs, "imap_username");
            $imap_password = grab_array_var($inargs, "imap_password");
            $imap_port = grab_array_var($inargs, "imap_port");
            $imap_ssl = grab_array_var($inargs, "imap_ssl");

            // save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["hostname"] = $hostname;
            $meta_arr["ip_address"] = $address;
            $meta_arr["servicedesc"] = $servicedesc;
            $meta_arr["smtpto"] = $smtpto;
            $meta_arr["smtpfrom"] = $smtpfrom;
            $meta_arr["smtp_address"] = $smtp_address;
            $meta_arr["smtp_username"] = $smtp_username;
            $meta_arr["smtp_password"] = $smtp_password;
            $meta_arr["smtp_port"] = $smtp_port;
            $meta_arr["smtp_tls"] = $smtp_tls;
            $meta_arr["imap_address"] = $imap_address;
            $meta_arr["imap_username"] = $imap_username;
            $meta_arr["imap_password"] = $imap_password;
            $meta_arr["imap_port"] = $imap_port;
            $meta_arr["imap_ssl"] = $imap_ssl;
            save_configwizard_object_meta($wizard_name, $hostname, "", $meta_arr);

            $objs = array();

            if (!host_exists($hostname)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_generic_host",
                    "host_name" => $hostname,
                    "address" => $address,
                    "icon_image" => "email-delivery.png",
                    "statusmap_image" => "email-delivery.png",
                    "_xiwizard" => $wizard_name,
                );
            }

            // email service
            $emailargs = "--mailto $smtpto --mailfrom $smtpfrom -H $address ";
            $emailargs .= "--smtp-server $smtp_address --smtp-username '$smtp_username' --smtp-password '$smtp_password' --smtp-port $smtp_port ";
            $emailargs .= "--imap-server $imap_address --username '$imap_username' --password '$imap_password' --imap-port $imap_port";
            if ($smtp_tls)
                $emailargs .= " --smtptls";
            if ($imap_ssl)
                $emailargs .= " --imapssl";


            $objs[] = array(
                "type" => OBJECTTYPE_SERVICE,
                "host_name" => $hostname,
                "service_description" => $servicedesc,
                "use" => "xiwizard_generic_service",
                "check_command" => "check_email_delivery!" . $emailargs,
                "_xiwizard" => $wizard_name,
            );

            // Return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:
            break;
    }

    return $output;
}
