<?php
//
// Mass Notification Component
// Copyright (c) 2011-2023 Nagios Enterprises, LLC. All rights reserved.
//

require_once(dirname(__FILE__) . '/../../common.inc.php');

// Initialization stuff
pre_init();
init_session();

// Grab GET or POST variables and do prereq/auth checks
grab_request_vars();
check_prereqs();
check_authentication(false);

$title = _("Mass Toggle Notifications");
do_page_start(array("page_title" => $title), true);
?>

<script type="text/javascript">

var allChecked = false;
var checked = [];

function checkAll(host, obj) {
    if (host == 'all' && allChecked == false) {
        $('.select-all-for-host').each(function() {
            var host = $(this).data('host');
            $('input.'+host).prop('checked', true);
            checked[host] = true;
        });
        $('#checkAllButton').val('<?php echo _("Uncheck All Items"); ?>');
        allChecked = true;
        $('.hostcheck').prop('checked', true);
    } else if (host == 'all' && allChecked == true) {
        $('.select-all-for-host').each(function() {
            var host = $(this).data('host');
            $('input.'+host).prop('checked', false);
            checked[host] = false;
        });
        $('#checkAllButton').val('<?php echo _("Check All Items"); ?>');
        allChecked = false;
        $('.hostcheck').prop('checked', false);
    }
}

$(document).ready(function() {
    // toggle services for host
    $('.select-all-for-host').click(function() {
        var host = $(this).data('host');
        if (checked[host] == false || checked[host] == undefined) {
            $('input.'+host).prop('checked', true);
            checked[host] = true;
        } else {
            $('input.'+host).prop('checked', false);
            checked[host] = false;
        }
    });

    if (!is_neptune()) {
        $('#hostList').searchable({maxMultiMatch: 9999});
        $('#hostgroupList').searchable({maxMultiMatch: 9999});
        $('#servicegroupList').searchable({maxMultiMatch: 9999});
    }

    $('#hostList').change(function () {
        $('#servicegroupList').val('');
        $('#hostgroupList').val('');
    });

    $('#hostgroupList').change(function () {
        $('#servicegroupList').val('');
        $('#hostList').val('');
    });

    $('#servicegroupList').change(function () {
        $('#hostgroupList').val('');
        $('#hostList').val('');
    });

});

</script>
<form method="get">
    <div class="well report-options form-inline">
        <div class="input-group" style="margin-right: 10px;">
            <label class="input-group-addon"><?php echo _("Limit To"); ?></label>
            <select name="host" id="hostList" style="width: 150px;" class="rounded-l-none form-control">
                <option value=""><?php echo _("Host"); ?>:</option>
                <?php
                $args = array('brevity' => 1, 'orderby' => 'host_name:a');
                $oxml = get_xml_host_objects($args);
                if ($oxml) {
                    foreach ($oxml->host as $hostobject) {
                        $name = strval($hostobject->host_name);
                        $is_selected = (isset($host)) ? is_selected($host, $name) : "";
                        echo "<option value='" . $name . "' " . $is_selected . ">$name</option>\n";
                    }
                }
                ?>
            </select>        
            <select name="hostgroup" id="hostgroupList" style="width: 150px;" class="rounded-l-none form-control">
                <option value=""><?php echo _("Hostgroup"); ?>:</option>
                <?php
                $args = array('orderby' => 'hostgroup_name:a');
                $oxml = get_xml_hostgroup_objects($args);
                if ($oxml) {
                    foreach ($oxml->hostgroup as $hg) {
                        $name = strval($hg->hostgroup_name);
                        $is_selected = (isset($hostgroup)) ? is_selected($hostgroup, $name) : "";
                        echo "<option value='" . $name . "' " . $is_selected . ">$name</option>";
                    }
                }
                ?>
            </select>
            <select name="servicegroup" id="servicegroupList" style="width: 175px;" class="rounded-l-none form-control">
                <option value=""><?php echo _("Servicegroup"); ?>:</option>
                <?php
                $args = array('orderby' => 'servicegroup_name:a');
                $oxml = get_xml_servicegroup_objects($args);
                if ($oxml) {
                    foreach ($oxml->servicegroup as $sg) {
                        $name = strval($sg->servicegroup_name);
                        $is_selected = (isset($servicegroup)) ? is_selected($servicegroup, $name) : "";
                        echo "<option value='" . $name . "' " . $is_selected . ">$name</option>";
                    }
                }
                ?>
            </select>
        </div>
        <div class="input-group" style="margin-right: 10px;">
            <input type="submit" style="margin-left: 10px;" id="runButton" class='btn btn-sm btn-primary' name='runButton' value="<?php echo _("Update"); ?>">
        </div>
    </div>
</form>

<?php
$submitted = grab_request_var('submitted', false);
$feedback = '';
$hosts = get_hosts();
$services = get_services();

// Display output from command submissions 
if ($submitted) {
    $exec_errors = 0;
    $error_string = '';
    $feedback = massnotifications_core_commands();
}


function mn_comp($a, $b) {
    return strnatcmp($a['service_description'], $b['service_description']);
}

function mn_host_comp($a, $b) {
    return strnatcmp($a, $b);
}

foreach ($services as &$p) {
    usort($p, 'mn_comp');
}

// Sort the hostnames alphabetically
uksort($hosts, 'mn_host_comp');

if (is_readonly_user(0)) {
    $html = _("You are not authorized for this component.");
} else {
    $html = massnotifications_build_html($hosts, $services, $feedback);
}

print $html;

/////////////////FUNCTIONS/////////////////////////////////

function massnotifications_build_html($hosts, $services, $feedback)
{
    $html = "
        <div id='massnotif_wrapper'>
        <h1>" . _('Mass Toggle Notifications') . "</h1>
        {$feedback}
        <div id='massnotif_info'>
            <p class='text-medium'>" . _("Use this tool to enable or disable notifications for a large groups of hosts and/or services.") . "</p>
        </div>

        <div id='massack'>";

    $html .= '<form id="form_massnotif" action="index.php" method="post">';

    $html .= "<div class='well flex items-center gap-2 rounded-md nowrap mtn-width' style='margin: 10px 0 0 0;'>
                <input type='hidden' id='submitted' name='submitted' value='true' />
                <label class='text-medium' for='massack_type'>" . _('Command Type') . "</label>

                <select name='type' id='mass_type' class='form-control'>
                    <option value='enable'>" . _("Enable Notifications") . "</option>
                    <option value='disable'>" . _("Disable Notifications") . "</option>
                </select> &nbsp;

                <button type='submit' class='btn btn-sm btn-primary' style='margin-left: 10px; vertical-align: top;' id='submit'>" . _("Submit Commands") . "</button>
            </div>

            <div class='fl'>
                <div style='padding: 20px 0 10px 0;'>
                    <input type='button' class='btn btn-sm btn-default fl' id='checkAllButton' onclick='checkAll(\"all\")' title='"._('Check All Hosts and Services')."' value='" . _("Check All Items") . "'>
                    
                    <a href='index.php' class='btn btn-sm btn-default fr' title='"._('Update List')."'><i class='fa fa-refresh l'></i> " . _("Update List") . "</a>
                    <div class='clear'></div>
                </div>

                <table class='table table-condensed table-striped table-bordered table-auto-width'>
                    <thead>
                        <tr>
                            <th>" . _("Host Name") . "</th>
                            <th>" . _("Unhandled Service Problems") . "</th>
                            <th>" . _("Service Status") . "</th>
                            <th>" . _("Notification settings") . "</th>
                        </tr>
                    </thead>
                    <tbody>";

    $hostcount = 0;
    foreach ($hosts as $host) {

        //html variables
        $host_notifications = ($host['notifications'] == 1) ? _('Enabled') : _('Disabled');
        $host_checkbox = "<input type='checkbox' class='hostcheck' name='hosts[]' value='{$host['host_name']}' />";
        $checkAll = (isset($services[$host['host_name']])) ? '<a class="select-all-for-host" data-host="host'.$hostcount.'" >'. _("Toggle checkboxes for this Host").'</a>' : '&nbsp;';
        $host_class = host_class($host['host_state']);

        if(is_neptune()){
            $html .= "<tr>
            <td><div>{$host_checkbox} <span class='status-dot dot-10 {$host_class}'></span> {$host['host_name']} </div></td>
            <td class='aligncenter'> {$checkAll} </td>
            <td> &nbsp; </td>
            <td> {$host_notifications} </td>";
        } else {
            $html .= "<tr>
            <td class='{$host_class}'>{$host_checkbox} {$host['host_name']} </td>
            <td class='aligncenter'> {$checkAll} </td>
            <td> &nbsp; </td>
            <td> {$host_notifications} </td>";
        }

        if (isset($services[$host['host_name']])) {
            foreach ($services[$host['host_name']] as $service) {
                $service_notifications = ($service['notifications'] == 1) ? _('Enabled') : _('Disabled');
                if(is_neptune()){
                    $html .= "
                <tr>                             
                    <td> &nbsp; </td>
                    <td class='alignleft'>
                        <div class='checkbox'>
                            <label>
                                <input class='host{$hostcount} servicecheck' type='checkbox' name='services[]' value='{$service['host_name']};{$service['service_description']}'>
                                <div class='flex items-center checkbox-status-dot-align'>
                                <span class='status-dot dot-10 " . service_class($service['current_state']) . "'></span>
                                {$service['service_description']}
                                </div>
                            </label>
                        </div>
                    </td>
                    <td><div class='plugin_output scrollable-30'>{$service['plugin_output']}</div></td>
                    <td> {$service_notifications} </td>
                </tr>"; 
                } else {
                    $html .= "
                    <tr>
                        <td> &nbsp; </td>
                        <td class='alignleft " . service_class($service['current_state']) . "'>
                            <div class='checkbox'>
                                <label>
                                    <input class='host{$hostcount} servicecheck' type='checkbox' name='services[]' value='{$service['host_name']};{$service['service_description']}'>
                                    {$service['service_description']}
                                </label>
                            </div>
                        </td>
                        <td><div class='plugin_output'>{$service['plugin_output']}</div></td>
                        <td> {$service_notifications} </td>
                    </tr>";
                }

            }
            $hostcount++;
        }

    }
    $html .= "</tbody></table>
    </div><div class='clear'></div></form>";
    $html .= "</div></div>";

    return $html;
}

function massnotifications_core_commands()
{
    global $exec_errors;
    global $error_string;

    $hosts = grab_request_var('hosts', array());
    $services = grab_request_var('services', array());
    $mode = grab_request_var('type', 'enable');

    //bail if missing required values
    if (count($hosts) == 0 && count($services) == 0)
        return feedback_message('You must specify at least one host or service', true);

        //loop through any host specific commands   
        foreach ($hosts as $host) {
            
            $args = array(
                "host_name" => $host
            );
            
            $cmd = NAGIOSCORE_CMD_ENABLE_HOST_NOTIFICATIONS;
            if ($mode === 'disable') {
                 $cmd = NAGIOSCORE_CMD_DISABLE_HOST_NOTIFICATIONS;
            }

            $core_cmd = core_get_raw_command($cmd, $args);
            core_submit_command($core_cmd, $output);
        }

        //loop through service specific commands 
        foreach ($services as $service) {
            $service = explode(';', $service);

            $args = array(
                "host_name" => $service[0],
                "service_name" => $service[1]
            );

            $cmd = NAGIOSCORE_CMD_ENABLE_SVC_NOTIFICATIONS;
            if ($mode === 'disable') {
                $cmd = NAGIOSCORE_CMD_DISABLE_SVC_NOTIFICATIONS;
           }

            $core_cmd = core_get_raw_command($cmd, $args);
            core_submit_command($core_cmd, $output);
        }

    //return feedback for front-end
    if ($exec_errors == 0)
        return feedback_message(_('Commands processed successfully! Your command submissions may take a few moments to update in the display.'));
    else
        return feedback_message("$exec_errors " . _("errors were encountered while processing these commands") . " <br />$error_string", true);
}

function feedback_message($msg, $error = false)
{
    $class = ($error) ? 'errorMessage' : 'actionMessage';
    $icon = "<img src='" . theme_image("info_small.png") . "'>";

    if ($error) {
        $icon = "<img src='" . theme_image("critical_small.png") . "'>";
    }

    $html = "<div class='{$class} standalone'>
                {$icon} {$msg}
            </div>";
    return $html;
}

function host_class($code, $has_been_checked=1)
{
    if ($has_been_checked != 1)
        return '';
    switch ($code) {
        case 0:
            return "hostup";
        case 1:
            return 'hostdown';
        default:
            return 'hostunreachable';
    }
}

function service_class($code, $has_been_checked=1)
{
    if ($has_been_checked != 1)
        return '';
    switch ($code) {
        case 0:
            return "serviceok";
        case 1:
            return 'servicewarning';
        case 2:
            return 'servicecritical';
        default:
            return 'serviceunknown';
    }
}

function get_hosts()
{
    $hostgroup = grab_request_var("hostgroup", "");
    $servicegroup = grab_request_var("servicegroup", "");
    $host = grab_request_var("host", "");
    $backendargs = array();
    $hosts = array();
    $host_ids = array();

    if (!empty($host)) {
        $hosts[] = $host;
    }

    if (!empty($hostgroup)) {
        $host_ids = get_hostgroup_member_ids($hostgroup);
    }

    if (!empty($servicegroup)) {
        $host_ids = get_servicegroup_host_member_ids($servicegroup);
    }

    if (!empty($hosts)) {
        $backendargs['host_name'] = "in:".implode(',', $hosts);
    }

    if (!empty($host_ids)) {
        $backendargs['host_id'] = "in:".implode(',', $host_ids);
    }

    $xml = get_xml_host_status($backendargs);
    $hosts = array();
    if ($xml) {
        foreach ($xml->hoststatus as $x) {

            $hosts["$x->name"] = array(
                'host_state' => "$x->current_state", 
                'host_name' => "$x->name",
                'plugin_output' => "$x->status_text",
                'host_id' => "$x->host_id",
                'notifications' => "$x->notifications_enabled"
            );
        }
    } 

    return $hosts;
}

function get_services()
{
    $servicegroup = grab_request_var("servicegroup", "");
    $host = grab_request_var("host", "");
    $backendargs = array();

    if (!empty($host)) {
        $backendargs["host_name"] = $host;
    }

    if (!empty($servicegroup)) {
        $service_ids = get_servicegroup_member_ids($servicegroup);
    }

    if (!empty($service_ids)) {
        $backendargs['service_id'] = "in:".implode(',', $service_ids);
    }

    $xml = get_xml_service_status($backendargs);
    if ($xml) {
        foreach ($xml->servicestatus as $x) {
            $host_state = intval($x->host_current_state);
            $service = array(
                'host_name' => "$x->host_name",
                'service_description' => "$x->name",
                'current_state' => "$x->current_state",
                'plugin_output' => "$x->status_text",
                'notifications' => "$x->notifications_enabled",
                'service_id' => "$x->service_id"
            );
            $services["$x->host_name"][] = $service;
        }
    }

    return $services;  
}