<?php
//
// SNMP Walk Config Wizard
// Copyright (c) 2008-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

snmpwalk_configwizard_init();

function snmpwalk_configwizard_init()
{
    $name = "snmpwalk";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "3.0.2",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Scan an SNMP-enabled device for elements to monitor."),
        CONFIGWIZARD_DISPLAYTITLE => _("SNMP Walk"),
        CONFIGWIZARD_FUNCTION => "snmpwalk_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "snmp.png",
        CONFIGWIZARD_FILTER_GROUPS => array('network'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );
    register_configwizard($name, $args);
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function snmpwalk_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    global $cfg;

    $wizard_name = "snmpwalk";

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            // clear pre-existing snmpwalk wizard values for new wizard session
            // Remove any existing session variables
            unset($_SESSION['snmpwalk_snmp_version']);
            unset($_SESSION['snmpwalk_community_string']);
            unset($_SESSION['snmpwalk_v3_security_level']);
            unset($_SESSION['snmpwalk_v3_username']);
            unset($_SESSION['snmpwalk_v3_auth_password']);
            unset($_SESSION['snmpwalk_v3_auth_proto']);
            unset($_SESSION['snmpwalk_v3_privacy_password']);
            unset($_SESSION['snmpwalk_v3_priv_proto']);
            unset($_SESSION['snmpwalk_port']);
            unset($_SESSION['snmpwalk_hostname']);
            unset($_SESSION['snmpwalk_community_string']);
            unset($_SESSION['snmpwalk_snmp_version']);
            unset($_SESSION['oid_data']);
            unset($_SESSION['snmpwalk_oid_data']);
            unset($_SESSION['snmpwalk_address']);
            unset($_SESSION['snmpwalk_job']);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            // Get variables that were passed to us
            $job = $_SESSION['snmpwalk_job'] ?? grab_array_var($inargs, "job") ?? '';

            $errors = 0;
            $errmsg = array();
            if (have_value($job) == true) {
                $job_values = explode('_', $job);
                $address = $job_values[0];
                $_SESSION['snmpwalk_job'] = $job;
                $_SESSION['snmpwalk_address'] = $address;
            } else {
                $errmsg[$errors++] = "Job not specified.";
            }

            $_SESSION['oid_group_mode'] = grab_array_var($inargs, "oid_group_mode") ?? "mib";

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            if (!isset($_SESSION['snmpwalk_oid_data'])) {
                // Get job values from job value passed from step 1
                $job = $_SESSION['snmpwalk_job'];
                if (!empty($job)) {
                    $job_values = explode('_', $job);
                    $address = $job_values[0];
                }
                $ha = @gethostbyaddr($address);

                if ($ha == "") {
                    $ha = $address;
                }

                $hostname = grab_array_var($inargs, "hostname", $ha);
                $hostname = nagiosccm_replace_user_macros($hostname);
                $_SESSION['snmpwalk_hostname'] = $hostname;

                // Build file name from job name
                $job_file_dir = get_tmp_dir();
                $filename = $job_file_dir . '/snmpwalk_' . $job . '.log';

                // Get job contents, break down into rows, isolate command row
                $fcontents = file_get_contents($filename);
                $rows = explode("\n", $fcontents);
                $first_line = trim($rows[0]);
                if (preg_match('/^Running:\s*snmpwalk\s+(.*)$/i', $first_line, $matches)) {
                    $command = $matches[1];
                    $tokens = preg_split('/\s+/', $command);

                    for ($i = 0; $i < count($tokens); $i++) {
                        switch ($tokens[$i]) {
                            case '-v':
                                $_SESSION['snmpwalk_snmp_version'] = $tokens[$i + 1] ?? null;
                                $i++;
                                break;
                            case '-c':
                                $_SESSION['snmpwalk_community_string'] = $tokens[$i + 1] ?? null;
                                $i++;
                                break;
                            case '-l':
                                $_SESSION['snmpwalk_v3_security_level'] = $tokens[$i + 1] ?? null;
                                $i++;
                                break;
                            case '-u':
                                $_SESSION['snmpwalk_v3_username'] = $tokens[$i + 1] ?? null;
                                $i++;
                                break;
                            case '-A':
                                $_SESSION['snmpwalk_v3_auth_password'] = $tokens[$i + 1] ?? null;
                                $i++;
                                break;
                            case '-a':
                                $_SESSION['snmpwalk_v3_auth_proto'] = $tokens[$i + 1] ?? null;
                                $i++;
                                break;
                            case '-X':
                                $_SESSION['snmpwalk_v3_privacy_password'] = $tokens[$i + 1] ?? null;
                                $i++;
                                break;
                            case '-x':
                                $_SESSION['snmpwalk_v3_priv_proto'] = $tokens[$i + 1] ?? null;
                                $i++;
                                break;
                            default:
                                // Check for host:port pattern
                                if (preg_match('/^([a-zA-Z0-9\.\-]+):(\d+)$/', $tokens[$i], $host_matches)) {
                                    $_SESSION['snmpwalk_port'] = $host_matches[2];
                                }
                                break;
                        }
                    }
                }
                $all_oids = [];

                foreach ($rows as $row) {
                    // Get mib
                    $parts = explode("::", $row);
                    $mib = $parts[0];

                    array_shift($parts);
                    $newrow = implode("::", $parts);

                    // Get oid
                    $parts = explode(" = ", $newrow);
                    $theoid = $parts[0];

                    array_shift($parts);
                    $newrow = implode(" = ", $parts);

                    // Get type
                    $parts = explode(":", $newrow);
                    $type = $parts[0];

                    array_shift($parts);
                    $newrow = implode(":", $parts);

                    // Get value
                    $val = $newrow;

                    // Skip if there is no value and no type
                    if (empty($val) && strpos($type, "No more variables left in this MIB View") !== false) {
                        continue;
                    }

                    // Make sure we have all the data
                    if ($mib == "" || $theoid == "" || $type == "") {
                        continue;
                    }

                    $oid_name = substr($theoid, 0, strrpos($theoid, '.'));
                    $oid_index = substr($theoid, strrpos($theoid, '.') + 1, strlen($theoid));

                    // only show OIDs that are useful for monitoring
                    $oid_type_whitelist = [
                        'integer',
                        'string',
                        'gauge32',
                        'counter32',
                        'counter64',
                        'unsigned32',
                        'timeticks'
                    ];

                    // only load translated OIDs
                    if (preg_match('/^[a-zA-Z0-9_-]+$/', $oid_name) === 1) {
                        if (in_array(strtolower($type), $oid_type_whitelist)) {
                            // $oid_data[$oid_index][$theoid] = array(
                            $all_oids[] = array(
                                "oid" => $theoid,
                                "type" => $type,
                                "val" => $val,
                                "name" => "",
                                "label" => "",
                                "units" => "",
                                "matchtype" => "",
                                "warning" => "",
                                "critical" => "",
                                "string" => "",
                                "mib" => $mib,
                                "oid_index" => $oid_index,
                                "oid_name" => $oid_name,
                            );
                        }
                    }
                }

                // store data in both structures now to make toggling quick
                $grouped_by_interface = [];
                $grouped_by_mib = [];

                foreach ($all_oids as $entry) {
                    $grouped_by_interface[$entry['oid_index']][$entry['oid']] = $entry;
                    $grouped_by_mib[$entry['mib']][$entry['oid']] = $entry;
                }

                // load args into session
                // $_SESSION['snmpwalk_oid_data'] = $oid_data;
                $_SESSION['snmpwalk_oid_data']['grouped_by_interface'] = $grouped_by_interface;
                $_SESSION['snmpwalk_oid_data']['grouped_by_mib'] = $grouped_by_mib;
            }

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // Get variables that were passed to us
            $address =  $_SESSION['snmpwalk_address'];
            $hostname =  $_SESSION['snmpwalk_hostname'];
            $port = $_SESSION['snmpwalk_port'];
            $snmpcommunity = $_SESSION['snmpwalk_community_string'];
            $snmpversion = $_SESSION['snmpwalk_snmp_version'];
            $oid_data = grab_array_var($inargs, "oid_data", NULL) ?? $_SESSION['snmpwalk_oid_data'] ?? [];

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (is_valid_host_name($hostname) == false) {
                $errmsg[$errors++] = "Invalid host name: " . $hostname;
            }

            $selected_row_count = 0;
            foreach ($oid_data as $group) {
                foreach ($group as $oid => $args) {
                    if (isset($args['checked']) && $args['checked'] == 'on') {
                        $selected_row_count++;
                        switch ($args["matchtype"]) {
                            case "numeric":
                                if ($args["warning"] == "")
                                    $errmsg[$errors++] = 'Invalid warning numeric range for OID (' . encode_form_val($args['oid']) . ')';
                                if ($args["critical"] == "")
                                    $errmsg[$errors++] = 'Invalid critical numeric range for OID (' . encode_form_val($args['oid']) . ')';
                                break;
                            case "string":
                                if ($args["string"] == "")
                                    $errmsg[$errors++] = 'Threshold is empty for OID (' . encode_form_val($args['oid']) . ')';
                            default:
                                break;
                        }
                    }
                }
            }

            // If no rows were checked, advise the user
            if ($selected_row_count == 0) {
                $errmsg[$errors++] = "You have not selected any OIDs to monitor.";
            }
            
            // update session values with validated form values
            $_SESSION['snmpwalk_oid_data'] = $oid_data;

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;


        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:

            $output = '

            ';
            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            $address =  $_SESSION['snmpwalk_address'];
            $hostname =  $_SESSION['snmpwalk_hostname'];
            $port = $_SESSION['snmpwalk_port'];
            $snmpcommunity = $_SESSION['snmpwalk_community_string'];
            $snmpversion = $_SESSION['snmpwalk_snmp_version'];
            $oid_data = $_SESSION['snmpwalk_oid_data'];

            $objs = array();

            if (!host_exists($hostname)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_genericnetdevice_host",
                    "host_name" => $hostname,
                    "address" => $address,
                    "icon_image" => "snmp.png",
                    "statusmap_image" => "snmp.png",
                    "_xiwizard" => $wizard_name,
                );
            }
            // loop through OIDs to create services
            foreach ($oid_data as $interface) {
                foreach ($interface as $oid_label => $oid_record) {
                    // only add services for checked OIDs
                    if (isset($oid_record['checked']) && $oid_record['checked'] == "on") {
                        $oid = $oid_record["oid"];
                        $name = $oid_record["name"];
                        $label = $oid_record["label"];
                        $units = $oid_record["units"];
                        $matchtype = $oid_record["matchtype"];
                        $warning = $oid_record["warning"];
                        $critical = $oid_record["critical"];
                        $string = $oid_record["string"];
                        $mib = $oid_record["mib"];

                        // Strip whitespace from units
                        $units = str_replace(" ", "", $units);

                        $service_description = (!empty($name)) ? $name : $oid;

                        $cmdargs = "";
                        // port
                        $cmdargs .= " -p " . intval($port);
                        // oid
                        if ($oid != "")
                            $cmdargs .= " -o " . escapeshellarg($oid);
                        // snmp community
                        if ($snmpcommunity != "" && $snmpversion != "3")
                            $cmdargs .= " -C " . escapeshellarg_nowrap($snmpcommunity);
                        // snmp version
                        if ($snmpversion != "")
                            $cmdargs .= " -P " . $snmpversion;
                        // snmp v3 stuff
                        if ($snmpversion == "3") {
                            if ($_SESSION['snmpwalk_v3_security_level'] != "")
                                $cmdargs .= " --seclevel=" . $_SESSION['snmpwalk_v3_security_level'];
                            if ($_SESSION['snmpwalk_v3_username'] != "")
                                $cmdargs .= " --secname=" . escapeshellarg($_SESSION['snmpwalk_v3_username']);
                            if ($_SESSION['snmpwalk_v3_auth_proto'] != "")
                                $cmdargs .= " --authproto=" . $_SESSION['snmpwalk_v3_auth_proto'];
                            if ($_SESSION['snmpwalk_v3_auth_password'] != "")
                                $cmdargs .= " --authpasswd=" . escapeshellarg_nowrap($_SESSION['snmpwalk_v3_auth_password']);
                            if ($_SESSION['snmpwalk_v3_privacy_password'] != "")
                                $cmdargs .= " --privpasswd=" . escapeshellarg_nowrap($_SESSION['snmpwalk_v3_privacy_password']);
                            if ($_SESSION['snmpwalk_v3_priv_proto'] != "")
                                $cmdargs .= " -x " . $_SESSION['snmpwalk_v3_priv_proto'];
                        }
                        // label
                        if ($label != "")
                            $cmdargs .= " -l " . escapeshellarg($label);
                        // units
                        if ($units != "")
                            $cmdargs .= " -u " . escapeshellarg($units);
                        // mib
                        if ($mib != "")
                            $cmdargs .= " -m " . $mib;
                        // match type...
                        switch ($matchtype) {
                            case "numeric":
                                if ($warning != "")
                                    $cmdargs .= " -w " . $warning;
                                if ($critical != "")
                                    $cmdargs .= " -c " . $critical;
                                break;
                            case "string":
                                if ($string != "")
                                    $cmdargs .= " -r " . escapeshellarg($string);
                                break;
                            default:
                                break;
                        }

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => $service_description,
                            "use" => "xiwizard_snmp_service",
                            "check_command" => "check_xi_service_snmp!" . $cmdargs,
                            "_xiwizard" => $wizard_name,
                        );
                    }
                }
            }

            //echo "OBJECTS:<BR>";
            //print_r($objs);
            //exit();

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:
            break;
    }

    return $output;
}


