#!/bin/bash -e
#
# Script to automatically install, upgrade, and configure NagVis
#
# NagVis Integration Component - Install
# Copyright (c) 2011-2025 - Nagios Enterprises, LLC. All rights reserved.
#

# Load configuration file

BASEDIR=$(dirname $(readlink -f $0))
cfgfile="$BASEDIR/../../../../xi-sys.cfg"
cfgfile_installed="/usr/local/nagiosxi/var/xi-sys.cfg"

if [ -f $cfgfile ]; then
    . $cfgfile
elif [ -f $cfgfile_installed ]; then
    . $cfgfile_installed
fi

# Set config options

MODE="xi"
NAGVIS_VER="1.9.47c"
NAGIOS_NAGVIS_VER="$NAGVIS_VER-nagios"

# Print usage information
usage() {
    echo "Script to automatically install and configure NagVis"
    echo ""
    echo "Copyright:  2011-2024 - Nagios Enterprises, LLC"
    echo "Author:     Nagios Development Team"
    echo ""
    echo "Options:"
    echo "    -h | --help"
    echo "           Display this help text"
    echo "    -v | --version"
    echo "           Display the version of NagVis this script will install"
    echo ""
}

# Parse parameters
while [ $# -gt 0 ]; do
    case "$1" in
        -h | --help)
            usage
            exit 0
            ;;
        -v | --version)
            echo "Will install NagVis version $NAGIOS_NAGVIS_VER"
            echo ""
            exit 0
            ;;
        -s | --short-version)
            echo "$NAGIOS_NAGVIS_VER"
            exit 0
            ;;
        *)
            echo "Unknown argument: $1" >&2
            usage >&2
            exit 0
            ;;
    esac
shift
done

# Check if we are able to install cleanly
if [ -f /usr/local/nagvis/share/index.php ]; then
    echo "NagVis is already installed... upgrading..."
    $BASEDIR/upgrade.sh
    exit 0
fi

# Check whether we have sufficient privileges
if [ $(whoami) != "root" ]; then
    echo "This script needs to be run as root/superuser." >&2
    exit 0
fi

# Install nagvis
(
    cd $BASEDIR
    tar xf nagvis-$NAGIOS_NAGVIS_VER.tar.gz
    cd nagvis-$NAGIOS_NAGVIS_VER
    ./install.sh -b /usr/bin -u $apacheuser -g $apachegroup -w "$httpdconfdir" -i ndo2db -a y -q
    cd ..
    rm -rf nagvis-$NAGIOS_NAGVIS_VER
)

# Run script to adjust configuration (may remove this in the future?)
$BASEDIR/scripts/adjust_config.py --mode xi

# Make password correct
sed -i "s/$(grep 'dbpass=' '/usr/local/nagvis/etc/nagvis.ini.php')/dbpass=\"$ndoutilspass\"/" /usr/local/nagvis/etc/nagvis.ini.php

# Remove livestatus backend
sed -i 's/\[backend_live_1\]/;\[backend_live_1\]/' /usr/local/nagvis/etc/nagvis.ini.php
sed -i 's/backendtype="mklivestatus"/;backendtype="mklivestatus"/' /usr/local/nagvis/etc/nagvis.ini.php

# Update ndomy_1 to nagiosxi
sed -i 's/\[backend_ndomy_1\]/\[backend_nagiosxi\]/' /usr/local/nagvis/etc/nagvis.ini.php
sed -i 's/backend="ndomy_1"/backend="nagiosxi"/' /usr/local/nagvis/etc/nagvis.ini.php

# Reload Apache configuration
if [ `command -v a2enconf` ]; then
    a2enconf nagvis.conf
fi

if [ `command -v systemctl` ]; then
    systemctl reload $httpd
else
    service $httpd reload
fi

echo ""
echo "============================="
echo "NagVis Installation Complete!"
echo "============================="
