<?php
//
// Minefield Component
// Copyright (c) 2010-2024 Nagios Enterprises, LLC. All rights reserved.
//

require_once(dirname(__FILE__) . '/../componenthelper.inc.php');
include_once(dirname(__FILE__) . '/dashlet.inc.php');

$minemap_component_name = "minemap";
minemap_component_init();

////////////////////////////////////////////////////////////////////////
// COMPONENT INIT FUNCTIONS
////////////////////////////////////////////////////////////////////////

function minemap_component_init()
{
    global $minemap_component_name;

    $versionok = minemap_component_checkversion();

    $desc = "";
    if (!$versionok)
        $desc = "<b>" . _("Error: This component requires Nagios XI 2024R1.2 or later.") . "</b>";

    $args = array(
        COMPONENT_NAME => $minemap_component_name,
        COMPONENT_AUTHOR => "Nagios Enterprises, LLC",
        COMPONENT_DESCRIPTION => _("Displays a grid-like view of host and service status. ") . $desc,
        COMPONENT_TITLE => _("Minemap"),
        COMPONENT_VERSION => '2.1.1'
    );

    register_component($minemap_component_name, $args);

    // add a menu link
    if ($versionok)
        register_callback(CALLBACK_MENUS_INITIALIZED, 'minemap_component_addmenu');

    // register a dashlet
    $args = array();
    $args[DASHLET_NAME] = "minemap";
    $args[DASHLET_TITLE] = "Minemap";
    $args[DASHLET_FUNCTION] = "minemap_dashlet_func";
    $args[DASHLET_DESCRIPTION] = _("Displays a grid-like view of host and service status.");
    $args[DASHLET_FILTER_GROUPS] = array("status");
    $args[DASHLET_WIDTH] = "360";
    $args[DASHLET_HEIGHT] = "260";
    $args[DASHLET_INBOARD_CLASS] = "minemap_map_inboard";
    $args[DASHLET_OUTBOARD_CLASS] = "minemap_map_outboard";
    $args[DASHLET_CLASS] = "minemap_map";
    $args[DASHLET_AUTHOR] = "Nagios Enterprises, LLC";
    $args[DASHLET_COPYRIGHT] = "Dashlet Copyright &copy; 2010-2018 Nagios Enterprises. All rights reserved.";
    $args[DASHLET_HOMEPAGE] = "http://www.nagios.com";
    $args[DASHLET_SHOWASAVAILABLE] = true;
    $args[DASHLET_ISCUSTOM] = false;
    register_dashlet($args[DASHLET_NAME], $args);
}


///////////////////////////////////////////////////////////////////////////////////////////
// MISC FUNCTIONS
///////////////////////////////////////////////////////////////////////////////////////////

function minemap_component_checkversion()
{

    if (!function_exists('get_product_release'))
        return false;
    //requires greater than 2024R1.2
    if (get_product_release() < 60200)
        return false;

    return true;
}

function minemap_component_addmenu($arg = null)
{
    global $minemap_component_name;

    $mi = find_menu_item(MENU_HOME, "menu-home-section-maps", "id");
    if ($mi == null)
        return;

    $order = grab_array_var($mi, "order", "");
    if ($order == "")
        return;

    $neworder = $order + 1;

    add_menu_item(MENU_HOME, array(
        "type" => "link",
        "title" => _("Minemap"),
        "id" => "menu-home-minemap",
        "order" => $neworder,
        "opts" => array(
            "href" => get_base_url() . 'includes/components/minemap/',
            "icon" => "fa-circle"
        )
    ));

}

// You can either pass in a service object or a host object
function get_minemap_host_img($hs = null, $ss = null, $hostname = "") {
    $img = "";
    // host status
    if ($hs) {
        $hosttext = strval($hs->status_text);
        $hoststate = intval($hs->current_state);
        $hosthbc = intval($hs->has_been_checked);
        $hostack = intval($hs->problem_acknowledged);
        $hostsdd = intval($hs->scheduled_downtime_depth);
    } else if ($ss) {
        $hosttext = strval($ss->host_status_text);
        $hoststate = intval($ss->host_current_state);
        $hosthbc = intval($ss->host_has_been_checked);
        $hostack = intval($ss->host_problem_acknowledged);
        $hostsdd = intval($ss->host_scheduled_downtime_depth);
    }
    else {
        return $img;
    }
    switch ($hoststate) {
        case 0:
            if ($hosthbc == 1) {
                $imgtitle = "" . encode_form_val($hostname) . " " . _("Host Up") . ": " . encode_form_val($hosttext);
                $img = '<span class="material-symbols-outlined md-18 md-ok md-400 md-middle md-button md-padding tt-bind" title="' . $imgtitle . '">check_circle</span>';
            } else {
                $imgtitle = encode_form_val($hostname) . " " . _("Host Pending");
                $img = '<span class="material-symbols-outlined md-18 md-pending md-400 md-middle md-button md-padding tt-bind" title="' . $imgtitle . '">schedule</span>';
            }
            break;
        case 1:
            $imgtitle = "" . encode_form_val($hostname) . " " . _("Host Down") . ": " . encode_form_val($hosttext);
            $img = '<span class="material-symbols-outlined md-18 md-critical md-400 md-middle md-button md-padding tt-bind" title="' . $imgtitle . '">cancel</span>';
            break;
        case 2:
            $imgtitle = "" . encode_form_val($hostname) . " " . _("Host Unreachable") . ": " . encode_form_val($hosttext);
            $img = '<span class="material-symbols-outlined md-18 md-unknown md-400 md-middle md-button md-padding tt-bind" title="' . $imgtitle . '">help</span>';
            break;
        default:
            $img = "";
            break;
    }

    // check for acknowledgements, scheduled downtime
    if ($hoststate != 0) {
        if ($hostack == 1) {
            $imgtitle = "" . encode_form_val($hostname) . " " . _("Host Acknowledged") . ": " . encode_form_val($hosttext);
            $img = '<span class="material-symbols-outlined md-18 md-pending md-400 md-middle md-button md-padding tt-bind" title="' . $imgtitle . '">build_circle</span>';
        } else if ($hostsdd > 0) {
            $imgtitle = "" . encode_form_val($hostname) . " " . _("Scheduled Downtime") . ": " . encode_form_val($hosttext);
            $img = '<span class="material-symbols-outlined md-18 md-pending md-400 md-middle md-button md-padding tt-bind" title="' . $imgtitle . '">build_circle</span>';
        }
    }
    return $img;
}

function get_minemap_service_img($ss, $servicename = "") {
    // service status
    $servicestate = intval($ss->current_state);
    $servicetext = strval($ss->status_text);
    $servicehbc = intval($ss->has_been_checked);
    $serviceack = intval($ss->problem_acknowledged);
    $servicesdd = intval($ss->scheduled_downtime_depth);
    $hoststate = intval($ss->host_current_state);
    $hostack = intval($ss->host_problem_acknowledged);
    $hostsdd = intval($ss->host_scheduled_downtime_depth);
    $img = "";
    switch ($servicestate) {
        case 0:
            if ($servicehbc == 1) {
                $imgtitle = "" . encode_form_val($servicename) . " " . _("Ok") . ": " . encode_form_val($servicetext);
                $img = '<span class="material-symbols-outlined md-18 md-ok md-400 md-middle md-button tt-bind" title="' . $imgtitle . '">check_circle</span>';
            } else {
                $imgtitle = "" . encode_form_val($servicename) . " " . _("Pending") . ": " . encode_form_val($servicetext);
                $img = '<span class="material-symbols-outlined md-18 md-pending md-400 md-middle md-button tt-bind" title="' . $imgtitle . '">build_circle</span>';
            }
            break;
        case 1:
            $imgtitle = "" . encode_form_val($servicename) . " " . _("Warning") . ": " . encode_form_val($servicetext);
            $img = '<span class="material-symbols-outlined md-18 md-warning md-400 md-middle md-button tt-bind" title="' . $imgtitle . '">warning</span>';
            break;
        case 2:
            $imgtitle = "" . encode_form_val($servicename) . " " . _("Critical") . ": " . encode_form_val($servicetext);
            $img = '<span class="material-symbols-outlined md-18 md-critical md-400 md-middle md-button tt-bind" title="' . $imgtitle . '">error</span>';
            break;
        case 3:
            $imgtitle = "" . encode_form_val($servicename) . " " . _("Unknown") . ": " . encode_form_val($servicetext);
            $img = '<span class="material-symbols-outlined md-18 md-unknown md-400 md-middle md-button tt-bind" title="' . $imgtitle . '">help</span>';
            break;
        default:
            $img = "";
            break;
    }

    // check for acknowledgements, scheduled downtime
    if ($servicestate != 0) {
        if ($serviceack == 1) {
            $imgtitle = "" . encode_form_val($servicename) . " " . _("Acknowledged") . ": " . encode_form_val($servicetext);
            $img = '<span class="material-symbols-outlined md-18 md-pending md-400 md-middle md-button tt-bind" title="' . $imgtitle . '">build_circle</span>';
        } else if ($servicesdd > 0) {
            $imgtitle = "" . encode_form_val($servicename) . " " . _("Scheduled Downtime") . ": " . encode_form_val($servicetext);
            $img = '<span class="material-symbols-outlined md-18 md-pending md-400 md-middle md-button tt-bind" title="' . $imgtitle . '">build_circle</span>';
        }
    }

    // check for host acknowledgements, scheduled downtime
    if ($hoststate != 0) {
        if ($hostack == 1) {
            $imgtitle = "" . encode_form_val($servicename) . " " . _("Host Problem Acknowledged") .": " . encode_form_val($servicetext);
            $img = '<span class="material-symbols-outlined md-18 md-pending md-400 md-middle md-button tt-bind" title="' . $imgtitle . '">build_circle</span>';
        } else if ($hostsdd > 0) {
            $imgtitle = "" . encode_form_val($servicename) . " " . _("Host in Scheduled Downtime") . ": " . encode_form_val($servicetext);
            $img = '<span class="material-symbols-outlined md-18 md-pending md-400 md-middle md-button tt-bind" title="' . $imgtitle . '">build_circle</span>';
        }
    }
    return $img;
}

///////////////////////////////////////////////////////////////////////////////////////////
// AJAX FUNCTIONS
///////////////////////////////////////////////////////////////////////////////////////////

function xicore_ajax_get_minemap_dashlet_html($args = null)
{

    $host = grab_array_var($args, "host", "");
    $hostgroup = grab_array_var($args, "hostgroup", "");
    $servicegroup = grab_array_var($args, "servicegroup", "");
    $hostgroups = [];
    $servicegroups = [];

    // special "all" stuff
    if ($hostgroup == "all")
        $hostgroup = "";
    if ($servicegroup == "all")
        $servicegroup = "";
    if ($host == "all")
        $host = "";

    // can do hostgroup(s) OR servicegroup(s) OR host
    if (strstr($hostgroup, ",")) {
        $hostgroups = explode(", ", $hostgroup);
        $hostgroup = "";
    } else if ($hostgroup != "") {
        $servicegroup = "";
        $host = "";
    } else if (strstr($servicegroup, ",")) {
        $servicegroups = explode(", ", $servicegroup);
        $servicegroup = "";
    } else if ($servicegroup != "") {
        $host = "";
    }

    //  limit hosts by hostgroup or host
    $host_ids = array();
    $host_ids_str = "";
    //  limit by hostgroup
    if ($hostgroup != "") {
        $host_ids = get_hostgroup_member_ids($hostgroup);
    } //  limit by host
    else if ($host != "") {
        $host_ids[] = get_host_id($host);
    }
    $y = 0;
    foreach ($host_ids as $hid) {
        if ($y > 0)
            $host_ids_str .= ",";
        $host_ids_str .= $hid;
        $y++;
    }
    //  limit service by servicegroup
    $service_ids = array();
    $service_ids_str = "";
    if ($servicegroup != "") {
        $service_ids = get_servicegroup_member_ids($servicegroup);
    }
    $y = 0;
    foreach ($service_ids as $sid) {
        if ($y > 0)
            $service_ids_str .= ",";
        $service_ids_str .= $sid;
        $y++;
    }

    // get service status from backend
    $xml = null;
    $first_col_title = _('Hosts');
    $second_col_title = _('Services');
    if ($hostgroups) {
        $first_col_title = _('Hostgroups');
        $second_col_title = _('Hosts');
    } else if ($servicegroups) {
        $first_col_title = _('Servicegroups');
        $second_col_title = _('Services');
    } else {        
        $backendargs = array();
        $backendargs["cmd"] = "getservicestatus";
        $backendargs["limitrecords"] = false; // don't limit records
        $backendargs["combinedhost"] = true; // get host status too
        $backendargs["brevity"] = 1; // we don't need everything
        // host id limiters
        if ($host_ids_str != "")
            $backendargs["host_id"] = "in:" . $host_ids_str;
        // service id limiters
        if ($service_ids_str != "")
            $backendargs["service_id"] = "in:" . $service_ids_str;
        // order by host name, service description
        $backendargs["orderby"] = "host_name:a,service_description:a";

        $xml = get_xml_service_status($backendargs);
    }

    $output = "
    <style>
    .minemap_map_inboard {
        overflow: auto;
    }
    .minemap_dashlet {
        width: 100% !important;
    }
    </style>
    ";

    $pretitle = "";
    if ($host != "")
        $pretitle = "Host '$host' ";
    else if ($hostgroup != "") {
        $pretitle = "Hostgroup '$hostgroup' ";
    }
    else if ($servicegroup != "") {
        $pretitle = "Servicegroup '$servicegroup' ";
    }

    $output .= '
    <style>
        .minemap_map_inboard .infotable_title {
            padding: 0px 0px 1px;
            font-size: 0.75rem;
        }
        .minemap_map {
            display: flex;
            flex-direction: column;
        }
        .minemap_map_inboard {
            display: flex;
            flex-direction: column;
            gap: 2px;
        }
        .minemap_map_inboard .content-wrapper {
            display: flex;
            flex-direction: column;
            gap: 1px;
        }
            
        .minemap_map_inboard .table-container {
            overflow: hidden;
            position: relative;
        }
        .minemap_map_inboard .table-container table {
            transform-origin: 0 0;
            position: relative;
        }
        .minemap_map_inboard td, 
        .minemap_map_inboard th {
            white-space: nowrap;
            padding: 2px 4px;
        }
        .minemap_map_inboard .material-symbols-outlined {
            font-size: 16px;
        }
        .minemap_map_inboard .ajax_date {
            padding: 2px 4px;
            font-size: 0.75rem;
        }
    </style>
    <script src="' . get_base_url() . 'includes/components/minemap/minemap.js"></script>
    <div class="content-wrapper">
        <div class="infotable_title">' . $pretitle . _('Status Grid') . '</div>
        <div class="table-container">
            <table class="' . $tableclass . '">
    <thead>
    <tr><th>' . $first_col_title . '</th><th>' . $second_col_title . '</th></tr>
    </thead>
    <tbody>
    ';

    if ($xml) {
        
        $base_url = get_base_url();
        $status_url = $base_url . "includes/components/xicore/status.php";
        $image_url = $base_url . "includes/components/minemap/images/";

        $lasthost = "";
        foreach ($xml->servicestatus as $ss) {
            //$output.='<tr><td>SVC</td></tr>';

            $thishost = strval($ss->host_name);

            if ($thishost != $lasthost) {

                // end last row
                if ($lasthost != "") {
                    $output .= '</td></tr>';
                }

                $img = get_minemap_host_img(null, $ss);

                // start new row
                $output .= '<tr><td nowrap>';
                $output .= '<a class="align-items-center-flex" href="' . $status_url . '?show=hostdetail&host=' . urlencode($thishost) . '" target="_blank">' . $img . $thishost . '</a>';
                $output .= '</td><td>';
            }

            // remember last host
            $lasthost = $thishost;
            $servicename = strval($ss->name);

            $img = get_minemap_service_img($ss, $servicename);

            $output .= '<a href="' . $status_url . '?show=servicedetail&host=' . urlencode($thishost) . '&service=' . urlencode($servicename) . '" target="_blank">' . $img . '</a>';
        }

        // end last row
        if ($lasthost != "") {
            $output .= '</td></tr>';
        }
    } else if ($hostgroups) {
        $status_url = get_base_url() . "includes/components/xicore/status.php";
        foreach ($hostgroups as $hostgroup) {
            $output .= '<tr><td nowrap>';
            $output .= '<a href="' . $status_url . '?show=hosts&hostgroup=' . urlencode($hostgroup) . '" target="_blank">' . $hostgroup . '</a>';
            $output .= '</td><td>';
            $host_ids = [];
            $host_ids_str = "";
            $host_ids = get_hostgroup_member_ids($hostgroup);
            $y = 0;
            foreach ($host_ids as $hid) {
                if ($y > 0)
                    $host_ids_str .= ",";
                $host_ids_str .= $hid;
                $y++;
            }
            $backendargs = array();
            $backendargs["cmd"] = "gethoststatus";
            $backendargs["limitrecords"] = false; // don't limit records
            $backendargs["brevity"] = 1; // we don't need everything
            // service id limiters
            if ($host_ids_str != "")
                $backendargs["host_id"] = "in:" . $host_ids_str;
            // order by host name, service description
            $backendargs["orderby"] = "host_name:a";

            $xml = get_xml_host_status($backendargs);
            foreach ($xml->hoststatus as $hs) {

                $hostname = strval($hs->name);
                $img = get_minemap_host_img($hs, null, $hostname);

                $output .= '<a href="' . $status_url . '?show=hostdetail&host=' . urlencode($hostname) . ' target="_blank">' . $img . '</a>';
            }
            $output .= '</td></tr>';
        }
    } else if ($servicegroups) {
        $status_url = get_base_url() . "includes/components/xicore/status.php";
        foreach ($servicegroups as $servicegroup) {
            $output .= '<tr><td nowrap>';
            $output .= '<a href="' . $status_url . '?show=services&servicegroup=' . urlencode($servicegroup) . '" target="_blank">' . $servicegroup . '</a>';
            $output .= '</td><td>';
            $service_ids = [];
            $service_ids_str = "";
            $service_ids = get_servicegroup_member_ids($servicegroup);
            $y = 0;
            foreach ($service_ids as $sid) {
                if ($y > 0)
                    $service_ids_str .= ",";
                $service_ids_str .= $sid;
                $y++;
            }
            $backendargs = array();
            $backendargs["cmd"] = "getservicestatus";
            $backendargs["limitrecords"] = false; // don't limit records
            $backendargs["combinedhost"] = true; // get host status too
            $backendargs["brevity"] = 1; // we don't need everything
            // service id limiters
            if ($service_ids_str != "")
                $backendargs["service_id"] = "in:" . $service_ids_str;
            // order by host name, service description
            $backendargs["orderby"] = "host_name:a,service_description:a";

            $xml = get_xml_service_status($backendargs);
            foreach ($xml->servicestatus as $ss) {

                $servicename = strval($ss->name);
                $hostname = strval($ss->host_name);
                $img = get_minemap_service_img($ss, $servicename);

                $output .= '<a href="' . $status_url . '?show=servicedetail&host=' . urlencode($hostname) . '&service=' . urlencode($servicename) . '" target="_blank">' . $img . '</a>';
            }
            $output .= '</td></tr>';
        }

    } else {
        $output .= '<tr><td colspan="2">No data to display</td></tr>';
    }

    $output .= '
    </tbody>
    </table>
    </div>
        <div class="ajax_date">'._('Last Updated').': ' . get_datetime_string(time()) . '</div>
    </div>
    ';

    return $output;
}

?>