#!/usr/bin/env python3
import winrm
import argparse
import getpass
import os
from sys import argv

try:
    # Parse command-line arguments
    parser = argparse.ArgumentParser(description='winrm_check_cpu.py uses WinRM to obtain CPU statistics on a remote, Windows-based system.')
    parser.add_argument('-H', '--host', required=True, type=str, help='IP address or host name of the Windows system.')
    parser.add_argument('-u', '--user', required=True, type=str, help='Username for connecting to the Windows system.')
    parser.add_argument('-a', '--auth', required=True, type=str, choices=['basic-http', 'basic-https'], help='Authentication mechanism for the Windows system. Strongly recommended you avoid basic-http.')
    parser.add_argument('-p', '--password', required=True, type=str, help='Password for connecting to the Windows system.')
    parser.add_argument('-C', '--certvalidation', required=False, action="count", default=0)

    parser.add_argument('-m', '--metric', required=False, type=str, choices=['memory', 'cpu', 'count'], help='Metric', default="memory")
    parser.add_argument('-f', '--format', required=False, type=str, choices=['KB', 'MB', 'GB'], help='Format you want your metrics measured in. (KB, MB, GB)', default="KB")
    parser.add_argument('-n', '--name', required=True, type=str, help='Name of the process you are checking.')
    parser.add_argument('-w', '--warning', required=False, type=float, help='Percent usage at which a warning alert with be sent.', default=0)
    parser.add_argument('-c', '--critical', required=False, type=float, help='Percent usage at which a critical alert with be sent', default=0)

    args = parser.parse_args(argv[1:])

except argparse.ArgumentError as e:
    print(f"\nArgument error: {str(e)}")
    exit(1)
except Exception as e:
    print(f"\nAn error occurred during argument parsing: {str(e)}")
    exit(1)

try:
    authentication = None

    if args.auth == 'basic-http':
        # Use basic-http Authentication
        authentication = 'basic'

    elif args.auth == 'basic-https':
        # Use basic-https Authentication
        authentication = 'ssl'

    elif args.auth == 'cert':
        # Use Certificate Authentication (TLS transport with SSL enabled)
        authentication = winrm.transport.TlsTransport(ssl=True)


    # service status powershell script
    ps_process = f"""
    $strComputer = $Host
    Clear
    
    switch ('{args.metric}')
    {{
        'cpu' {{
            $CpuCores = (Get-WMIObject Win32_ComputerSystem).NumberOfLogicalProcessors
            $CpuSamples = (Get-Counter "\Process({args.name}*)\% Processor Time").CounterSamples
            return $CpuSamples | Measure-Object -Property CookedValue -Sum | ForEach-Object {{ [math]::Round(($_.Sum / $CpuCores), 2) }}
            break;
        }}
        default {{
            return (Get-Process -Name {args.name}).WorkingSet | ConvertTo-Json -Compress
            break;
        }}
    }}
    """


    # Create a WinRM session with the provided host, user, password, and authentication method
    if args.certvalidation > 0:
        winrmsession = winrm.Session(args.host, auth=(args.user, args.password), transport=authentication)
    else:
        winrmsession = winrm.Session(args.host, auth=(args.user, args.password), transport=authentication, server_cert_validation='ignore')


    # run the ps script and decode the output
    r = winrmsession.run_ps(ps_process)
    final_dictionary = eval(r.std_out.decode('utf-8'))

    status_dict = {
        0: "OK",
        1: "WARNING",
        2: "CRITICAL",
        3: "UNKNOWN"
    }
    byte_format = {
        "TB": 1099511627776,
        "GB": 1073741824,
        "MB": 1048576,
        "KB": 1024
    }

    status_code = 0
    message = ""
    if args.metric == 'cpu':
        process_cpu_usage = final_dictionary

        if args.warning and process_cpu_usage >= args.warning:
            status_code = 1
        if args.critical and process_cpu_usage >= args.critical:
            status_code = 2

        message = f"{str(status_dict[status_code])}: Process '{str(args.name)}' cpu usage is {process_cpu_usage}%"

    elif args.metric == 'count':
        # calculate the count of specified process
        process_total_count = 1
        # if the process has multiple instances return the total, otherwise just return the value
        try:
            process_total_count = len(final_dictionary)
        except Exception as e:
            pass

        if args.warning and process_total_count >= args.warning:
            status_code = 1
        if args.critical and process_total_count >= args.critical:
            status_code = 2

        message = f"{str(status_dict[status_code])}: Process '{str(args.name)}' count is {process_total_count}"

    else:
        # calculate the proceses total memory usage
        process_memory_usage = 0
        # if the process has multiple instances total them, otherwise just return the value
        try:
            total_process_memory_usage = len(final_dictionary)
            for memory_usage in final_dictionary:
                process_memory_usage += memory_usage
        except Exception as e:
            process_memory_usage = final_dictionary

        process_memory_usage = round(int(process_memory_usage)/byte_format[args.format])

        if args.warning and process_memory_usage >= args.warning:
            status_code = 1
        if args.critical and process_memory_usage >= args.critical:
            status_code = 2

        message = f"{str(status_dict[status_code])}: Process '{str(args.name)}' memory usage at {process_memory_usage} {args.format}"
    
    # return status message and code
    print(message)
    exit(status_code)

except winrm.exceptions.WinRMTransportError as e:
    print(f"WinRM transport error: {str(e)}")
    exit(4)
except Exception as e:
    print(f"An error occurred during WinRM session setup or command execution: {str(e)}")
    exit(4)