<?php
//
// Map component for viewing hosts, services, hostgroups and servicegroups on a world map
// User must specify coordinates in the 'notes' section of the CCM for hosts/services
// Copyright (c) 2019-2020 Nagios Enterprises, LLC. All rights reserved.
//

require_once(dirname(__FILE__).'/../../../config.inc.php');
require_once(dirname(__FILE__) . '/../../common.inc.php');

// Initialization stuff
pre_init();
init_session();

// Grab GET or POST variables, check prereqs, and authorization
grab_request_vars();
check_prereqs();
check_authentication(false);
do_page_start(array('page_title' => _('World Map')), true);

// Get map popup info
$map_stop_info_popup = get_user_meta(0, 'map_stop_info_popup', 0);

// Get default coords
$default_coords = get_user_meta(0, 'map_default_coords', '');
if (strpos($default_coords, ',') === false) {
    $coords = array(44.9537, -93.0900);
} else {
    $tmp = explode(',', $default_coords);
    $coords = array(floatval($tmp[0]), floatval($tmp[1]));
}
?>

<head>
    <link rel="stylesheet" href="library/leaflet/css/leaflet.css">
    <link rel="stylesheet" href="library/leaflet/css/leaflet.search.css">
    <link rel="stylesheet" href="library/markercluster/MarkerCluster.Default.css">
    <link rel="stylesheet" href="library/markercluster/MarkerCluster.css">
    <link rel="stylesheet" href="styles/style.css">
    <?php
    if (is_neptune()) {
        $neptune_palette = get_neptune_palette(false);
        ?>
        <link rel="stylesheet" href="styles/<?php echo $neptune_palette; ?>">
        <link rel="stylesheet" href="styles/neptune.style.css">
        <?php } ?>
    <style>
        .isaac {
            background-color: purple;
        }
    </style>
</head>
<body>
    <div id="map_container">
    <!-- Map -->
    </div>
    
    <!-- library -->
    <script src="library/leaflet/js/leaflet.js"></script>
    <script src="library/leaflet/js/leaflet.search.js"></script>
    <script src="library/markercluster/leaflet.markercluster.js"></script>
    <script src="library/markercluster/leaflet.markercluster.layersupport.js"></script>

    <!-- main script -->
    <script src="scripts/main.js"></script>
    
    <script type="text/javascript">
        var tile_selected = '<?php if (get_theme() == 'xi5dark' || get_theme() == 'neptune') { echo 'dark'; } else { echo 'light'; } ?>';
        var default_coords = <?php echo json_encode($coords); ?>;
        var map = null;

        $(document).ready(function() {

            <?php if (!$map_stop_info_popup && empty($_SESSION['shown_map_popup']) && !is_readonly_user($_SESSION['user_id'])):?>
                show_info_popup(true);
            <?php endif; ?>
            
            // Popup was shown, hide it on refresh
            <?php $_SESSION['shown_map_popup'] = 1; ?>

            $('.info-popup.import .btn-close').click(function() {
                clear_whiteout();
                $('.info-popup.import').hide();

                var dont_show = $('.dont-show').is(':checked');
                if (dont_show) {
                    var optsarr = {
                        "keyname": "map_stop_info_popup",
                        "keyvalue": 1
                    };
                    var opts = JSON.stringify(optsarr);
                    var result = get_ajax_data("setusermeta", opts);
                }
            });

            // Set up the setting gear
            if ($('#open-dtform').length > 0) {
                var p = $('#open-dtform').position();
                $('#settings-dropdown').css('left', '30px');
                $('#settings-dropdown').css('top', '0px');

                $("#open-dtform").click(function () {
                    var d = $("#settings-dropdown").css("display");
                    if (d == "none") {
                        $("#settings-dropdown").fadeIn("fast");
                    } else {
                        $("#settings-dropdown").fadeOut("fast");
                    }
                });
            }

            $('.btn-close-settings').click(function() {
                $("#settings-dropdown").fadeOut("fast");
            });

            $('.use-current').click(function() {
                var current = map.getCenter();
                $('.default-coords').val(current.lat + ', ' + current.lng);
            });

            $('.btn-update-settings').click(function() {
                var coords = $('.default-coords').val();
                var optsarr = {
                    "keyname": "map_default_coords",
                    "keyvalue": coords
                };
                var opts = JSON.stringify(optsarr);
                var result = get_ajax_data("setusermeta", opts);
                $("#settings-dropdown").fadeOut("fast");
            });

            $(window).resize(function() {
                $('.info-popup.import').center().css('top', '140px');
            });
            
            $('.import-info').click(function() {
                show_info_popup(false);
            });

        });

        // Initial info popup screen
        function show_info_popup(show_stop) {
            whiteout();
            $('.info-popup.import').show().center().css('top', '140px');
            if (show_stop) {
                $('.dont-display').show();
            } else {
                $('.dont-display').hide();
            }
        }
    </script>

    <style type="text/css">
    div#settings-dropdown:before { display: none; }
    </style>

    <div class="map-gear-margin bg-transparent neptune-map" style="position: absolute; top: 28px; left: 60px; z-index: 1000; background-color: #FFF; padding: 4px; border-radius: 5px;">
        <a class="tt-bind settings-dropdown" id="open-dtform" title="<?php echo _('Map settings'); ?>"><span class="bg-transparent material-symbols-outlined md-button bg-none">settings</span></a>
        <div id="settings-dropdown" style="z-index: 1001;">
            <div class="content">
                <div class="input-group" style="width: 300px;">
                    <label class="input-group-addon"><?php echo _('Map center'); ?></label>
                    <input type="text" class="default-coords form-control" value="<?php echo $default_coords; ?>" placeholder="44.9537, -93.0900">
                    <div class="input-group-addon"><a class="use-current tt-bind" title="<?php echo _('Use current map center'); ?>"><i class="fa fa-14 fa-map-marker"></i></a></div>
                </div>
                <div style="margin-top: 10px">
                    <button type="button" class="btn btn-update-settings btn-xs btn-primary fl"><i class="fa fa-check fa-l"></i> <?php echo _('Update'); ?></button>
                    <button type="button" class="btn btn-close-settings btn-xs btn-default fr"><i class="fa fa-times"></i> <?php echo _('Close'); ?></button>
                    <div class="clear"></div>
                </div>
            </div>
        </div>
    </div>

    <div class="info-popup import hide">
        <h2 style="margin-top: 0; padding-top: 0;"><?php echo _('How to Use This Map'); ?></h2>
        <!-- TODO: Uncomment this and fill in href once documentation is complete to show link to doc 

        <P><?php echo _("Please review the following documents before you use this tool"); ?>:</P>

        <p>
            <i class="fa fa-external-link"></i> <a target="_blank" href=""><?php echo _("How to Use the Nagios XI World Map"); ?></a>
        </p>
         -->

         <!-- Getting Started -->
        <p><strong><?php echo _("Getting Started"); ?></strong></p>
        <p><?php echo _('To view hosts and services on this map, you must specify their geographical coordinates in the form of longitude and latitude.'); ?></p>
        <p><?php echo _('This is done by specifying the coordinates in the "Notes" field of the config you wish to be mapped. To navigate to the proper location in the CCM, use the following path'); ?>:</p>
        <p>
            <em><?php echo _('Core Configuration Manager'); ?> <i class="fa fa-arrow-right"></i> <?php echo _('Hosts/Services'); ?> <i class="fa fa-arrow-right"></i> <?php echo _('Select a Config'); ?> <i class="fa fa-arrow-right"></i> <?php echo _('Misc Settings') ?> <i class="fa fa-arrow-right"></i> <?php echo _('Notes'); ?></em>
        </p>
        <p><?php echo _('You can specify the coordinates with one of two formats; alone or alongside other notes. The syntax for each is as follows, respectively'); ?>:</p>
            <pre style="width: 80%; margin: 3px auto;"><?php echo _("44.972894, -93.153935"); ?></pre>
            <pre style="width: 80%; margin: 3px auto;"><?php echo _("These are some example notes (44.972894, -93.153935)"); ?></pre>
        <p class="alert alert-info text-center" style="padding: 10px; margin-top: 10px;"><strong><?php echo _("Note:</strong> You must surround the coordinates in parenthesis when alongside other notes!"); ?></p>

        <!-- How to Use the Map -->
        <p><strong><?php echo _('Using the Map'); ?></strong></p>
        <p><?php echo _('To change the theme or toggle the display of hosts or services, use the menu in the upper right corner.'); ?></p>
        <p><?php echo _('To view the details of any given host or service on the map, click the corresponding pin on the map. This will show the current status and a link to the detail page.'); ?></p>
        <p><strong><?php echo _('Pin Clustering'); ?></strong></p>
        <p><?php echo _('When pins become too close to each other on the map, they will cluster and collapse into a circle. To view the clustered hosts and services, click the circle and the clustered objects will explode into view.'); ?></p>

        <div class="flex items-center" style="margin-top: 15px;">
            <button type="button" class="btn-close btn btn-sm btn-default"><?php echo _('Close'); ?></button>
            <span class="checkbox centered-checkbox dont-display" style="display: inline-block; margin-left:  <?php if (!is_neptune()) { echo "15px;"; } else { echo "30px;"; } ?>">
                <label class="flex items-center">
                    <input type="checkbox" class="dont-show"> <?php echo _("Don't show this message again"); ?>
                </label>
            </span>
        </div>
    </div>
</body>
<?php do_page_end(true); ?>
