<?php
//
// Map Component
// Copyright (c) 2019-2020 Nagios Enterprises, LLC. All rights reserved.
//

require_once(dirname(__FILE__) . '/../componenthelper.inc.php');

pre_init();
init_session();

grab_request_vars();
check_prereqs();
check_authentication();
route_request();

/**
 *  Route the request call from script main.js
 */
function route_request()
{
    // grab request values
    $mode = grab_request_var('mode');

    switch ( $mode ) {
        
    case 'hosts':
        generate_hosts();
        break;

    case 'services':
        generate_services();
        break;

    default:
        break;
    }
}

/*
 *  Generate hosts data from XML objects and return as JSON
 */
function generate_hosts()
{
    global $map_component_name;
    $url = get_component_url_base($map_component_name) . "/";
    $hosts = array();

    $host_objects = get_data_host_status(array('notes' => 'ne:'));

    // Extract id, name and notes from the objects
    foreach ($host_objects as $obj) {

        $notes = $obj['notes'];
        if (!empty($notes)) {

            $id = intval($obj['host_object_id']);
            $host_name = $obj['host_name'];

            $name = $obj['display_name'];
            if (empty($name)) {
                $name = $host_name;
            }

            $coordinates = extract_coordinates($notes);
            if ($coordinates) {
                $hosts[$id]['host_id'] = $id;
                $hosts[$id]['name'] = $name;
                $hosts[$id]['host_name'] = $host_name;
                $hosts[$id]['latitude']  = $coordinates['latitude'];
                $hosts[$id]['longitude'] = $coordinates['longitude'];
            } else {
                continue;
            }

            $status_text = $obj['output'];
            $current_state = $obj['current_state'];

            switch ($current_state) {

                // Up
                case 0:
                    $state_text = _('UP');
                    $state_class = 'label-up';
                    $icon = $url . 'images/marker-icon-green.png';
                    break;

                // Down
                case 1:
                    $state_text = _('DOWN');
                    $state_class = 'label-down';
                    $icon = $url . 'images/marker-icon-red.png';
                    break;

                // Unreachable
                case 2:
                    $state_text = _('UNREACHABLE');
                    $state_class = 'label-unreachable';
                    $icon = $url . 'images/marker-icon-yellow.png';
                    break;

                // Pending (default)
                default:
                    $state_text = _('PENDING');
                    $state_class = 'label-pending';
                    $icon = $url . 'images/marker-icon-gray.png';
                    break;
            }
            
            // Check to see if the host is in scheduled downtime.
            $in_downtime = false;
            if ($obj['scheduled_downtime_depth'] > 0) {
                $icon = $url . 'images/marker-icon-blue.png';
                $in_downtime = true;
            }

            $hosts[$id]['status_text'] = $status_text;
            $hosts[$id]['current_state'] = $current_state;
            $hosts[$id]['state_text'] = $state_text;
            $hosts[$id]['state_class'] = $state_class;
            $hosts[$id]['icon'] = $icon;
            $hosts[$id]['in_downtime'] = $in_downtime;
        }

    }

    $json = json_encode($hosts);
    echo $json;
}


/*
 *  Generate service data from XML objects and return as JSON
 */
function generate_services()
{
    global $map_component_name;
    $url = get_component_url_base($map_component_name) . "/";
    $services = array();

    $service_objects = get_data_service_status(array('notes' => 'ne:'));

    // Extract id, name and notes from the objects
    foreach ($service_objects as $obj) {

        $notes = $obj['notes'];
        if (!empty($notes)) {

            $id = intval($obj['service_object_id']);
            $host_name = $obj['host_name'];
            $service_description = $obj['service_description'];

            $name = $obj['display_name'];
            if (empty($name)) {
                $name = $service_description;
            }
            
            $coordinates = extract_coordinates($notes);
            if ($coordinates) {
                $services[$id]['service_id'] = $id;
                $services[$id]['name']= $name;
                $services[$id]['host_name'] = $host_name;
                $services[$id]['service_description'] = $service_description;
                $services[$id]['latitude'] = $coordinates['latitude'];
                $services[$id]['longitude'] = $coordinates['longitude'];
            } else {
                continue;
            }

            $status_text = $obj['output'];
            $current_state = $obj['current_state'];

            switch ($current_state) {

                // Ok
                case 0:
                    $state_text = _('OK');
                    $state_class = 'label-ok';
                    $icon = $url . 'images/marker-icon-green.png';
                    break;

                // Warning
                case 1:
                    $state_text = _('WARNING');
                    $state_class = 'label-warning';
                    $icon = $url . 'images/marker-icon-yellow.png';
                    break;

                // Critical
                case 2:
                    $state_text = _('CRITICAL');
                    $state_class = 'label-critical';
                    $icon = $url . 'images/marker-icon-red.png';
                    break;

                // Unknown
                case 3:
                    $state_text = _('UNKNOWN');
                    $state_class = 'label-unknown';
                    $icon = $url . 'images/marker-icon-orange.png';
                    break;

                // Pending (default)
                default:
                    $state_text = _('PENDING');
                    $state_class = 'label-pending';
                    $icon = $url . 'images/marker-icon-gray.png';
                    break;
            }

            $services[$id]['status_text'] = $status_text;
            $services[$id]['current_state'] = $current_state;
            $services[$id]['state_text'] = $state_text;
            $services[$id]['state_class'] = $state_class;
            $services[$id]['icon'] = $icon;
        }

    }

    // return the object
    $json = json_encode($services);
    echo $json;
}

/*
 * Used to extract coordinates from the notes section of a host/service object
 * 
 * ON SUCCESS: Returns dictionary with keys latitude & longitude
 * ON FAILURE: Returns false
 *
 */

function extract_coordinates($object_notes)
{
    $matches = array();
    $coordinates = array();

    // Check if we have coordinates inside paranthesis such as (10, 20)
    $regex_result = preg_match('/\((.*),(.*)\)/', $object_notes, $matches);
    if ($regex_result) {

        $latitude = floatval(trim($matches[1]));
        $longitude = floatval(trim($matches[2]));

        if (is_float($latitude) && is_float($longitude)) {
            $coordinates = array(
                'latitude'  => $latitude,
                'longitude' => $longitude
            );
            return $coordinates;
        }

    } else {

        if (strpos($object_notes, ',') !== false) {

            $raw_coordinates = explode(',', $object_notes);
            $latitude = floatval(trim($raw_coordinates[0]));
            $longitude = floatval(trim($raw_coordinates[1]));

            if (is_float($latitude) && is_float($longitude)) {
                $coordinates = array(
                    'latitude'  => $latitude,
                    'longitude' => $longitude
                );
                return $coordinates;
            }

        }                
    }

    return false;
}

