<?php
//
// Custom RSS Dashlet
// Copyright (c) 2008-2019 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__).'/../dashlethelper.inc.php');


/* 
 * CONSTANTS - redefine these in your script to change the
 * behaviour of fetch_rss() currently, most options effect the cache
 *
 * MAGPIE_CACHE_ON - Should Magpie cache parsed RSS objects? 
 * For me a built in cache was essential to creating a "PHP-like" 
 * feel to Magpie, see rss_cache.inc for rationale
 *
 *
 * MAGPIE_CACHE_DIR - Where should Magpie cache parsed RSS objects?
 * This should be a location that the webserver can write to.   If this 
 * directory does not already exist Mapie will try to be smart and create 
 * it.  This will often fail for permissions reasons.
 *
 *
 * MAGPIE_CACHE_AGE - How long to store cached RSS objects? In seconds.
 *
 *
 * MAGPIE_CACHE_FRESH_ONLY - If remote fetch fails, throw error
 * instead of returning stale object?
 *
 * MAGPIE_DEBUG - Display debugging notices?
 *
*/

rss_dashlet_init();

define('MAGPIE_DEBUG', false);
define('MAGPIE_CACHE_ON', true);
define('MAGPIE_CACHE_DIR', '/usr/local/nagiosxi/tmp');

function rss_dashlet_init()
{
    $name = "rss_dashlet";

    $args = array(
        DASHLET_NAME => $name,

        DASHLET_VERSION => "1.1.8",
        DASHLET_DATE => "04/02/2024",
        DASHLET_AUTHOR => "Nagios Enterprises, LLC",
        DASHLET_DESCRIPTION => _("Scrolling RSS Feed. Define your own RSS feed, with up to 5 URLs"),
        DASHLET_FILTER_GROUPS => array("other"),
        DASHLET_COPYRIGHT => "Copyright (c) 2010-2024 Nagios Enterprises",
        DASHLET_LICENSE => "BSD",
        DASHLET_HOMEPAGE => "https://www.nagios.com",

        DASHLET_FUNCTION => "rss_dashlet_func",

        DASHLET_TITLE => _("RSS Dashlet"),

        DASHLET_OUTBOARD_CLASS=> "rss_dashlet_outboardclass",
        DASHLET_INBOARD_CLASS => "rss_dashlet_inboardclass",
        DASHLET_PREVIEW_CLASS => "rss_dashlet_previewclass",

        DASHLET_CSS_FILE => "rss_dashlet.css",

        DASHLET_WIDTH => "300",
        DASHLET_HEIGHT => "400",
        DASHLET_OPACITY => "0.8",
        DASHLET_ISCUSTOM => false
    );

    register_dashlet($name, $args);
}


function rss_dashlet_func($mode = DASHLET_MODE_PREVIEW, $id = "", $args = null)
{
    $output = "";

    switch ($mode) {

        case DASHLET_MODE_GETCONFIGHTML:
            $output = '

            <div class="form-group">
                <label for="url1">RSS 1</label>
                <div>
                    <input type="text" class="form-control" style="width: 100%;" name="url1" id="url1" value="">
                </div>
            </div>

            <div class="form-group">
                <label for="url2">RSS 2</label>
                <div>
                    <input type="text" class="form-control" style="width: 100%;" name="url2" id="url2" value="">
                </div>
            </div>

            <div class="form-group">
                <label for="url3">RSS 3</label>
                <div>
                    <input type="text" class="form-control" style="width: 100%;" name="url3" id="url3" value="">
                </div>
            </div>

            <div class="form-group">
                <label for="url4">RSS 4</label>
                <div>
                    <input type="text" class="form-control" style="width: 100%;" name="url4" id="url4" value="">
                </div>
            </div>

            <div class="form-group">
                <label for="url5">RSS 5</label>
                <div>
                    <input type="text" class="form-control" style="width: 100%;" name="url5" id="url5" value="">
                </div>
            </div>

            <div class="form-group">
                <label for="opacity">' . _('Opacity') . '</label>
                <div>
                    <select name="opacity" class="form-control" id="opacity">
                        <option value="1.0">100%</option>
                        <option value=".75" selected>75%</option>
                        <option value=".50">50%</option>
                        <option value=".25">25%</option>
                    </select>
                </div>
            </div>
            ';
            break;

        case DASHLET_MODE_OUTBOARD:
        case DASHLET_MODE_INBOARD:
            $height = grab_array_var($args, 'height', 400); 
            $width = grab_array_var($args, 'width', 300);
            $opacity = $args["opacity"];

            // Grab RSS data 
            require_once('magpierss/rss_fetch.inc');

            $rss = array(); 
            $count = 1;

            foreach (array($args['url1'], $args['url2'], $args['url3'], $args['url4'], $args['url5']) as $arg) {

                $title = isset($args['title'.$count]) ? $args['title'.$count] : _("Feed") . " $count";

                if (!isset($arg) || $arg == '') {
                    break;
                } else {
                    $rssObject = @fetch_rss(urldecode($arg));
                }

                $newarray = array_slice($rssObject->items, 0, 10);
                foreach ($newarray as $a) {
                    $a['source'] = $title;
                }

                $rss = array_merge($rss, $newarray);
                $count++;

            }

             // Validate RSS data
             if (empty($rss)) {
                $output = "<div class='alert alert-danger'>Error: No RSS data found</div>";
                break;
            }

            $rss2 = array();
            foreach ($rss as $i => $item) {
                $date = 0;
                if (isset($item['dc']['date'])) $date = strtotime($item['dc']['date']); 
                else if (isset($item['published'])) $date = strtotime($item['published']);
                else if (isset($item['pubdate'])) $date = strtotime($item['pubdate']); 
                else if (isset($item['pubDate'])) $date = strtotime($item['pubDate']);
                else if (isset($item['updated'])) $date = strtotime($item['updated']);
                else if (isset($item['timestamp'])) $date = $item['timestamp'];

                $mtime = ($date * 1000) + $i;
                $rss2[$mtime] = $item;
            }
            krsort($rss2);

            // Default dashlet settings
            $rand = rand();

            // HTML output
            $output .= "
            <div class='rss_dashlet_inboardclass' id='rssdashlet{$rand}'>";
            $output .= "<div id='news-feed{$rand}'>"; 
            $output .= "<div class='rss_inboardclass_rssList' id='theList'>";

            $r = 178;
            $g = 255;
            $b = 95;

            foreach ($rss2 as $item) {

                $source = isset($item['source']) ? $item['source'] : _('No Source'); 
                $rgb = "$r,$g,$b";
                $output.= '<div style="background-color: rgb('.$rgb.')" class="rss_dashlet_inboardclass_rssListItem">
                                    <a class="rss_dashlet_inboardclass_rssLink" title="'.encode_form_val($item['description']).'" target="_blank" href="'.encode_form_val($item['link']).'">'.$item['title'].'</a>
                                            </div>';

                // Make the RSS feed gradient coloring
                if ($r < 253) $r += 3; else $r = 255;
                if ($g < 3) $g = 0; else $g -= 3;
                if ($b < 3) $b = 0; else $b -= 3;

            }

            $output .= "</div>";
            $output .= "</div></div>";
            $output .= "<script type='text/javascript'>";

            // HTML output
            $output .= <<<OUTPUT

            var dashcontainer = $('#rssdashlet{$rand}').parent().parent();
            var innercontainer = $('#rssdashlet{$rand}').parent();

            // Set opacity to overwrite the dashlet opacity
            $(document).ready(function() {
                if (height == undefined) { var height = innercontainer.height() - 22; }
                if (width == undefined) { var width = innercontainer.width(); }
                innercontainer.css('opacity', '{$opacity}');
                $('#rssdashlet{$rand}').css('height', height+'px');
            });

            // Bind resize handlers to div and iframe
            dashcontainer.resize(function() {
                newHeight = $(this).height();
                newWidth = $(this).width(); 

                innercontainer.width(newWidth).height(newHeight);
                $('#rssdashlet{$rand}').width(newWidth).height(newHeight - 22);
            });

            // Headline rotation
            var interval = '';

            // Bind RSS rotation to hover event (so hovering won't rotate)
            $(document).ready(function() {
                $('#news-feed{$rand}').hover(function() {
                        clearTimeout(interval, 0);
                    },
                    function() {
                        interval = setTimeout('rotate()', 1000);
                    }
                );
            });

            // Sends top item to the bottom of the list
            function rotate() {
                var thisItem = $('.rss_dashlet_inboardclass_rssListItem:eq(0)');
                thisItem.appendTo($('#theList'));
                clearTimeout(interval, 0);
                interval = setTimeout('rotate()', 6000);
            }

            </script>
OUTPUT;
            break;

        case DASHLET_MODE_PREVIEW:
            $output = "<p><img src='/nagiosxi/includes/dashlets/rss_dashlet/rss.jpg' height='40' width='40' alt='RSS' /></p>";

            if(!is_neptune()) {
                $output = "<p><img src='/nagiosxi/includes/dashlets/rss_dashlet/rss.jpg' height='40' width='40' alt='RSS' /></p>";
			} else if (get_theme() == "neptunelight") {
                $output = "<img src='/nagiosxi/includes/dashlets/rss_dashlet/rss_neptune_light_preview.png' alt='RSS' />";
            } else {
                $output = "<img src='/nagiosxi/includes/dashlets/rss_dashlet/neptune_preview.png' alt='RSS' />";
            }
            
            break;
        
    }

    return $output;
}