<?php
//
// BPI Config Wizard
// Copyright (c) 2008-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

bpiwizard_configwizard_init();

function bpiwizard_configwizard_init()
{
    $name = "bpiwizard";

    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Create service checks for your Nagios BPI groups."),
        CONFIGWIZARD_DISPLAYTITLE => _("BPI Wizard"),
        CONFIGWIZARD_FUNCTION => "bpiwizard_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "bpiwizard.png",
        CONFIGWIZARD_VERSION => "2.0.1",
        CONFIGWIZARD_COPYRIGHT => "Copyright &copy; 2008-2024 Nagios Enterprises, LLC.",
        CONFIGWIZARD_AUTHOR => "Nagios Enterprises, LLC",
        CONFIGWIZARD_FILTER_GROUPS => array('nagios'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100,
    );

    register_configwizard($name, $args);
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function bpiwizard_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    global $cfg;

    // Grab core wizard info
    $wizard_name = grab_array_var($inargs, "wizard");
    $wiz_args = get_configwizard_by_name($wizard_name);
    $wizard_title = isset($wiz_args[CONFIGWIZARD_DISPLAYTITLE]) ? $wiz_args[CONFIGWIZARD_DISPLAYTITLE] : '';

    // Gotta have a sesh
    if (!isset($_SESSION[$wizard_name])) {
        $_SESSION[$wizard_name] = [];
    }

    // Give session a simple name for convenience
    $wz_sessn =& $_SESSION[$wizard_name];

    // initialize return code and output
    $result = 0;
    $output = "";

    // initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {

        case CONFIGWIZARD_MODE_GETSTAGE1HTML:
            //clear any previous wizard data
            if (!isset($_POST["backButton"])) {
                unset($_SESSION[$wizard_name]);
                $_SESSION[$wizard_name] = [];
                $wz_sessn =& $_SESSION[$wizard_name];

                $base = get_root_dir();
                //if we're using the new version of BPI

                if (file_exists($base . '/html/includes/components/nagiosbpi/bpi_display.php')) {
                    //echo "NEWVERSION<br />";
                    $configfile = is_null(get_option('bpi_configfile')) ? get_root_dir() . '/etc/components/bpi.conf' : get_option('bpi_configfile');

                    if (!file_exists($configfile)) {
                        $output = _('Unable to open BPI Config file, Please setup BPI before using the wizard');
                        break;
                    }

                    define('CLI', false);
                    $includefile = $cfg['root_dir'] . '/html/includes/components/nagiosbpi/inc.inc.php';
                    require($includefile);
                    $groupsArray = parse_bpi_conf(true);

                } else { //old version of BPI
                    //echo "OLD VERSION!<br />";
                    define('CONFIGFILE', get_base_dir() . '/includes/components/nagiosbpi/bpi.conf');
                    $configfile = get_base_dir() . '/includes/components/nagiosbpi/bpi.conf';
                    $includefile = get_base_dir() . '/includes/components/nagiosbpi/functions/read_conf.php';
                    require($includefile);
                    $groupsArray = get_info($configfile);
                }

                //make sure we have the necessary include
                if (file_exists($includefile)) {
                    $wz_sessn['data'] = $groupsArray;
                } else {
                    $outargs[CONFIGWIZARD_ERROR_MESSAGES] = _('Nagios BPI is not installed as a component') . '
                    <a href="http://exchange.nagios.org/directory/Addons/Components/Nagios-Business-Process-Intelligence-%28BPI%29/details" title="Download BPI" target="_blank">' . _('Download Nagios BPI') . '</a>';
                }
            }

            $hostname = grab_in_var($inargs, "hostname", "", $wz_sessn);
            $groups = grab_in_var($inargs, "groups", [], $wz_sessn);

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:
            // get variables that were passed to us
            $hostname = grab_in_var($inargs, "hostname", "", $wz_sessn);
            $hostname = nagiosccm_replace_user_macros($hostname);

            if (!isset($_POST['backButton'])) { //skip validation if back button pressed
                // check for errors
                $errors = [];

                $configfile = is_null(get_option('bpi_configfile')) ? get_root_dir() . '/etc/components/bpi.conf' : get_option('bpi_configfile');

                if (!file_exists($configfile)) {
                    $errmsg[$errors++] = _('Unable to open BPI Config file, Please setup BPI before using the wizard');
                }

                if (!isset($hostname) || $hostname == '')
                    $errors[] = "No name specified.";

                //check if BPI is installed
                if (!file_exists(get_base_dir() . '/includes/components/nagiosbpi/')) {
                    $errors[] = _('Nagios BPI is not installed as a component') . '.
                    <a href="http://exchange.nagios.org/directory/Addons/Components/Nagios-Business-Process-Intelligence/details" title="Download BPI" target="_blank">' . _('Download Nagios BPI') . '</a>';
                }

                if (count($errors)) {
                    $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errors;
                    $result = 1;
                }
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            $hostname = grab_in_var($inargs, "hostname", "", $wz_sessn);
            $groupsArray = $wz_sessn['data'];
            // $groups = isset($wz_sessn['groups']) ? $wz_sessn['groups'] : [];
            $groups = grab_in_var($inargs, "groups", [], $wz_sessn);

            // Build groups multi-select HTML
            $groups_by_name = [];
            foreach($groupsArray as $group) {
                if ($group['title']) {
                    $groups_by_name[$group['title']] = $group;
                }
            }

            $group_list = '';
            $groups_array = [];
            foreach ($groupsArray as $name => $group) {
                $name = encode_form_val($name);
                $safe_name = clean_str_for_dom($name);
                $groups_array[$name] = $group;  // Same as groupsArray - change later
                $selected = '';
                if (isset($groups[$name]) && $groups[$name] != ''){ $selected = ' selected'; }
                $group_list .= "<option value='$safe_name' $selected>$name</option>";
            }

            $group_select_html = '<select name="groups-select[]" id="groups-select" multiple  class="form form-control metrics-select multiselect form-select">';
            $group_select_html .= $group_list;
            $group_select_html .= '</select>';

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            if (!isset($_POST['backButton'])) {
                $groups = grab_in_var($inargs, "groups", [], $wz_sessn);
                $errors = [];

                if (! isset($inargs['groups'])) {
                    $errors[] = _("You must select at least one group.");
                    $wz_sessn['groups'] = [];
                }

                if (count($errors)) {
                    $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errors;
                    $result = 1;
                }
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            $groups = grab_in_var($inargs, "groups", [], $wz_sessn);
            $prepend = grab_in_var($inargs, "prepend", "", $wz_sessn);
            $wz_sessn['prepend'] = $prepend;
            $wz_sessn['groups'] = $groups;

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:
            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:
            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:
            $hostname = $wz_sessn['hostname'];
            $groups = $wz_sessn['groups'];
            $data = $wz_sessn['data'];
            $prepend = $wz_sessn['prepend'];
            $objs = array();

            // TODO: add a host template for the dummy host
            if (!host_exists($hostname)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_bpi_host",
                    "host_name" => $hostname,
                    "address" => '127.0.0.1',
                    "_xiwizard" => $wizard_name,
                );
            }

            // See which services we should monitor
            foreach ($groups as $group => $value) {

                $objs[] = array(
                    "type" => OBJECTTYPE_SERVICE,
                    "host_name" => $hostname,
                    "service_description" => $prepend . $data[$group]['title'],
                    "use" => "xiwizard_bpi_service",
                    "check_command" => "check_bpi!".escapeshellarg($group),
                    "_xiwizard" => $wizard_name
                );
            }

            // Return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;
            unset($wz_sessn);
            break;

        default:
            break;
    }

    return $output;
}
