<?php
//
// Microsoft/Office 365 Config Wizard
// Copyright (c) 2020-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__).'/../configwizardhelper.inc.php');

microsoft_365_configwizard_init();

function microsoft_365_configwizard_init() {
    $name = "microsoft_365";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.0.0",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor Microsoft 365 Subscription Services"),
        CONFIGWIZARD_DISPLAYTITLE => _("Microsoft 365"),
        CONFIGWIZARD_FUNCTION => "microsoft_365_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "microsoft_365.png",
        CONFIGWIZARD_FILTER_GROUPS => array('windows','database', 'server'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100  # 2024R1.1 - neptune & material-symbols-outlined
    );
    register_configwizard($name, $args);
}

const CONCEALED_INFO = '<div style="display: inline-block; padding-left: 5px;"><i class="credtooltip fa fa-question-circle fa-14" data-placement="right" data-content="This dropdown is disabled because your user data is concealed/obfuscated."></i></div>';

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function microsoft_365_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null) {

    $wizard_name = "microsoft_365";

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            $tenant = grab_array_var($inargs, "tenant", "");
            $appid = grab_array_var($inargs, "appid", "");
            $secret = grab_array_var($inargs, "secret", "");

            $selectedhostconfig = grab_array_var($inargs, "selectedhostconfig", "");
            $config_serial = grab_array_var($inargs, "selectedhostconfig", "");
            $operation = grab_array_var($inargs, "operation", "");

#            $tenant = decrypt_data($tenant);
#            $appid = decrypt_data($appid);
#            $secret = decrypt_data($secret);

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:
            // Get variables that were passed to us
            $appid = grab_array_var($inargs, "appid", "");
            $tenant = grab_array_var($inargs, "tenant", "");
            $secret = grab_array_var($inargs, "secret", "");

            $appid = nagiosccm_replace_user_macros($appid);
            $tenant = nagiosccm_replace_user_macros($tenant);
            $secret = nagiosccm_replace_user_macros($secret);

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (have_value($appid) == false)    $errmsg[$errors++] = "No Application ID specified.";
            if (have_value($tenant) == false)   $errmsg[$errors++] = "No Directory ID specified.";
            if (have_value($secret) == false)   $errmsg[$errors++] = "No Client Secret specified.";

            if ($errors == 0) {
                $authCheck = msip_auth_check($appid, $tenant, $secret);

                if ($authCheck) {
                    $errmsg[$errors++] = nl2br($authCheck);
                }
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us
            $appid = grab_array_var($inargs, "appid", "");
            $tenant = grab_array_var($inargs, "tenant", "");
            $secret = grab_array_var($inargs, "secret", "");
            $hostname = grab_array_var($inargs, "hostname", "");

            $selectedhostconfig = grab_array_var($inargs, "selectedhostconfig", "");
            $config_serial = grab_array_var($inargs, "selectedhostconfig", "");
            $operation = grab_array_var($inargs, "operation", "");

            # This should use ms graph batching, at some point.
            # Don't pass these along (like modelist and services).
            $usersList = plugin_data($appid, $tenant, $secret, 'userslist');
            $groupsList = plugin_data($appid, $tenant, $secret, 'groupslist');
            $productsList = plugin_data($appid, $tenant, $secret, 'productslist');
            $organization = plugin_data($appid, $tenant, $secret, 'organization');

            # Review, in case the < Back button was used.
            $services_serial = grab_array_var($inargs, "services_serial", "");
            $services = json_decode(base64_decode($services_serial), true);
            $modelist_serial = grab_array_var($inargs, "modelist_serial", "");
            $modelist = json_decode(base64_decode($modelist_serial), true);

            # This should use ms graph batching, at some point.
            $modelist = (empty($modelist)) ? mode_list($appid, $tenant, $secret) : $modelist;

            # Check for concealed user data (affects user, group and site names in all reports.
            $isDataConcealed = is_data_concealed($appid, $tenant, $secret);

            $ms365_base_url = get_base_url().'includes/configwizards/microsoft_365/';
            $select2_css_url = $ms365_base_url.'css/select2.css';
            $select2_bootstrap_css_url = $ms365_base_url.'css/select2-bootstrap-5-theme.min.css';
            $additional_theme_css_url = $ms365_base_url.'css/select2-bootstrap-5-theme';

            // Check if we need dark or neptune.
            if (get_theme() === 'xi5dark') {
                $additional_theme_css_url .= '-dark.css';
            } else if (is_neptune()) {
                $additional_theme_css_url .= '-neptune.css';
            } else {
                $additional_theme_css_url = '';
            }

            $select2_js_url = $ms365_base_url.'js/select2.full.min.js';

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // get variables that were passed to us
            $appid = grab_array_var($inargs, "appid");
            $tenant = grab_array_var($inargs, "tenant");
            $secret = grab_array_var($inargs, "secret");
            $hostname = grab_array_var($inargs, "hostname");

            $appid = nagiosccm_replace_user_macros($appid);
            $tenant = nagiosccm_replace_user_macros($tenant);
            $secret = nagiosccm_replace_user_macros($secret);

            #$modelist_serial = grab_array_var($inargs, "modelist", array());
            $modelist_serial = grab_array_var($inargs, "modelist_serial");
            $modelist = json_decode(base64_decode($modelist_serial), true);

            $services = grab_array_var($inargs, "services", array());

            // check for errors
            $errors = 0;
            $errmsg = array();

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;


        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

       // get variables that were passed to us
            $tenant = grab_array_var($inargs, "tenant");
            $appid = grab_array_var($inargs, "appid");
            $secret = grab_array_var($inargs, "secret");
            $hostname = grab_array_var($inargs, "hostname");

#            $modelist = grab_array_var($inargs, "modelist");
            $modelist_serial = grab_array_var($inargs, "modelist_serial");
            $modelist = json_decode(base64_decode($modelist_serial), true);

            $services = grab_array_var($inargs, "services");

            $output = '

        <input type="hidden" name="tenant" value="'.encode_form_val($tenant).'" />
        <input type="hidden" name="appid" value="'.encode_form_val($appid).'" />
        <input type="hidden" name="secret" value="'.encode_form_val($secret).'" />
        <input type="hidden" name="hostname" value="'.encode_form_val($hostname).'" />

        <input type="hidden" name="modelist_serial" value="'.$modelist_serial.'" />
        <input type="hidden" name="services_serial" value="'.base64_encode(json_encode($services)).'" />
            ';
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:

            $output = '
            ';
            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            $appid = grab_array_var($inargs, "appid", "");
            $tenant = grab_array_var($inargs, "tenant", "");
            $secret = grab_array_var($inargs, "secret", "");
            $hostname = grab_array_var($inargs, "hostname", "");

            $modelist_serial = grab_array_var($inargs, "modelist_serial");
            $services_serial = grab_array_var($inargs, "services_serial", "");

            $modelist = json_decode(base64_decode($modelist_serial), true);
            $services = json_decode(base64_decode($services_serial), true);

            // save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["appid"] = $appid;
            $meta_arr["tenant"] = $tenant;
            $meta_arr["secret"] = $secret;
            $meta_arr["hostname"] = $hostname;
            $meta_arr["modelist"] = $modelist;
            $meta_arr["services"] = $services;

            save_configwizard_object_meta($wizard_name, $appid, "", $meta_arr);

            $objs = array();

            if (!host_exists($appid)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_microsoft_365_host",
                    "host_name" => $hostname,
                    "icon_image" => "microsoft_365.png",
                    "statusmap_image" => "microsoft_365.png",
                    "_xiwizard" => $wizard_name,
                );
            }

            // common plugin opts
            $commonopts = "--tenant '$tenant' --appid '$appid' --secret '$secret' ";

            foreach ((array) $services as $service => $args) {

                $pluginopts = "".$commonopts;

                switch ($service) {

                    case "connectiontime":

                        $pluginopts .= "--mode time2connect --warning '".$services["warning"]."' --critical '".$service["critical"]."'";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => " Graph Connection Time",
                            "use" => "xiwizard_microsoft_365_service",
                            "check_command" => "check_xi_microsoft_365!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "process":

                        foreach ((array) $args as $counter_name => $services) {
                            if (empty($counter_name)) {
                                continue;
                            }

                            $filter = (array_key_exists('filter', $services) ? $services['filter'] : false);

                            foreach ((array) $services as $i => $service) {
                                # Skips if not checked (monitor)
                                # Skips higher level data, like the filter entry (!numeric).
                                if (!is_numeric($i) || !array_key_exists('monitor', $service)) {
                                    continue;
                                }


                                # Make sure the service_description(s) are unique, legal for performance data and short.
                                # NOTE: @ characters break the performance data, when in the service_description.
                                $svcDscExtension = "";

                                if (array_key_exists('displayName', $service) || array_key_exists('product', $service)) {

                                    # Since @ are break performance data, and only a complete email address is guaranteed to be
                                    # unique (hopefully), add an encoded version of the email address to the service_description,
                                    # after the displayName.
                                    if (array_key_exists('user', $service)) {
                                        $offset = strlen($service['user']." - ");
                                        $displayName = substr($service['displayName'], $offset);

                                        $svcDscExtension .= " - ".$displayName;
                                        $svcDscExtension .= " - ".substr(md5($service['user']), -6); # Last 6 characters
                                    }

                                    if (array_key_exists('group', $service)) {
                                        $length = strlen($service['displayName']) - strlen(" - ".$service['group']);
                                        $displayName = substr($service['displayName'], 0, $length);

                                        $svcDscExtension .= " - ".$displayName;
                                        $svcDscExtension .= " - ".substr($service['group'], -6); # Last 6 characters
                                    }

                                    if (array_key_exists('product', $service)) {
                                        # Make an acronym.
                                        $expr = '/(?<=\s|^)[a-z]|[\d]/i';
                                        preg_match_all($expr, $service['product'], $matches);
                                        $acronym = implode('', $matches[0]);

                                        $svcDscExtension .= " - ".$acronym;
                                    }
                                }

                                $pluginCustomOpts  = "--mode '".$counter_name."' --warning '".$service['warning']."' --critical '".$service['critical']."' ";

                                #if (array_key_exists('filterIdx', $modelist)) {
                                if ($filter) {
                                    $pluginCustomOpts .= " --filter '".$service[$filter]."'";
                                }

                                $serviceDescription = $modelist[$counter_name]['name'].$svcDscExtension;

                                $objs[] = array(
                                    "type" => OBJECTTYPE_SERVICE,
                                    "host_name" => $hostname,
                                    "service_description" => $serviceDescription,
                                    "use" => "xiwizard_microsoft_365_service",
                                    "_xiwizard" => $wizard_name,
                                    "check_command" => "check_xi_microsoft_365!".$pluginopts.$pluginCustomOpts,
                                );
                            }
                        }
                        break;
                }
            }

#            echo "OBJECTS:<BR>";
#            print_r($objs);
#            exit();

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;


        default:
            break;
    }

    return $output;
}

#############################################################################################################################
# Wizard data requests using check_microsoft_365.php plugin.
#
# time2token:   The time it takes to get the security token from MicroSoft Identity Platform (MSIP)
#       --appid=749628a7-XXXX-XXXX-XXXX-480485c2cfc3 --tenant=b45d6a6c-XXXX-XXXX-XXXX-b155bbbd594a
#       --secret=R98Hx1X~FK~XXXXXXX~3.31K5gti-bH5nR
#       -w 5 -c 10 --mode time2token
#
# modelist:     Modes/services available to users.
#       --modelist
#
#
#
const RETURN_JSON = "json";
const RETURN_CODE = "return_code";
const RETURN_DATA = "return_data";
const RETURN_VALUE = "return_value";
const RETURN_OK_OR_DATA = "return_ok_or_data";

# Verify connectivity to Microsoft Graph with the appid, tenant and secret, provided.
# NOTE: time2token is probably sufficient.  If not, hit the MS Graph endpoint (time2connect).  If we are using more
#       endpoints, than just MS Graph, figure something out.  Maybe check each endpoint as it it chosen.
#       Checking each mode/service may be necessary, if there are permission issues.
function msip_auth_check($appid, $tenant, $secret) {
    return plugin_data_request(" --appid ".escapeshellarg($appid)." --tenant ".escapeshellarg($tenant)." --secret ".escapeshellarg($secret)." -w 5 -c 10 --mode time2token", RETURN_OK_OR_DATA);
}

function is_data_concealed($appid, $tenant, $secret) {
    #$results = plugin_report_data($appid, $tenant, $secret, 'reportdatauserslist');
    $results = plugin_report_data($appid, $tenant, $secret, 'testuserslist');

    if (preg_match('/\@/', $results) > 0) {
        return false;
    }

    return true;
}

function plugin_data($appid, $tenant, $secret, $mode) {
    return plugin_data_request(" --appid ".escapeshellarg($appid)." --tenant ".escapeshellarg($tenant)." --secret ".escapeshellarg($secret)." -w 0 -c 0 --mode ".escapeshellarg($mode), RETURN_VALUE);
}

function plugin_report_data($appid, $tenant, $secret, $mode) {
    return plugin_data_request(" --appid ".escapeshellarg($appid)." --tenant ".escapeshellarg($tenant)." --secret ".escapeshellarg($secret)." -w 0 -c 0 --mode ".escapeshellarg($mode), RETURN_JSON);
}

function mode_list($appid, $tenant, $secret) {
    return plugin_data_request(" --appid ".escapeshellarg($appid)." --tenant ".escapeshellarg($tenant)." --secret ".escapeshellarg($secret)." -w 0 -c 0 --mode modelist", RETURN_DATA);
}

# Data for drop downs.
function plugin_data_request($cmdString, $type=RETURN_JSON) {
    $returnCode = 0;

    $cmd = "/usr/local/nagios/libexec/check_microsoft_365.php ".$cmdString;

    # If this is the concealed data check, use --refresh to make sure we are getting the latest data (not cached).
    #if (strpos($cmd, "reportdatauserslist") !== false) {
    if (strpos($cmd, "testuserslist") !== false) {
        $cmd .= " --refresh";
    }

    exec($cmd, $data, $returnCode);

    if ($type == RETURN_DATA && !$returnCode) {
        $dataArray = json_decode($data[0], true);

        if (json_last_error() == JSON_ERROR_NONE) {
            return $dataArray;
        }

        return json_last_error();
    }

    if ($type == RETURN_VALUE) {
        if (is_array($data)) {
            $dataArray = json_decode($data['0'], true);

            # How do errors like this need to be handled??
            # TODO: Handle errors
            if (!is_array($dataArray)) {
                # TODO: Development accounts, which no longer support reports, will have errors.
#                echo("ERROR! [".$data['0']."]");
            }

            if (is_array($dataArray)) {
                $dataArray = current($dataArray);
            }

            return $dataArray;
        }
    }

    $data = implode(PHP_EOL, $data);

    if ($type == RETURN_OK_OR_DATA) {
        if ($returnCode) {
            return $data;
        }

        return $returnCode;
    }

    if ($type == RETURN_CODE || $returnCode) {
        return $returnCode;
    }

    if ($type == RETURN_JSON) {
        if (!$data) {
            return 2; // Not JSON data
        }
        # If we have data, return $data
    }

    return $data;
}

?>
