<?php
//
// MySQL Server Config Wizard
// Copyright (c) 2010-2022 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

mysqlserver_configwizard_init();

function mysqlserver_configwizard_init()
{
    $name = "mysqlserver";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.0.1",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor a MySQL server."),
        CONFIGWIZARD_DISPLAYTITLE => _("MySQL Server"),
        CONFIGWIZARD_FUNCTION => "mysqlserver_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "mysql.png",
        CONFIGWIZARD_FILTER_GROUPS => array('database', 'server'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );
    register_configwizard($name, $args);
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function mysqlserver_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    $wizard_name = "mysqlserver";

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            $address = grab_array_var($inargs, "ip_address", "");
            $port = grab_array_var($inargs, "port", "3306");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");
            $database = grab_array_var($inargs, "database", "information_schema");

            # These will only be set by going back from step 2.
            $services = grab_array_var($inargs, "services");
            $serviceargs = grab_array_var($inargs, "serviceargs");

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address", "");
            $port = grab_array_var($inargs, "port", "");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");
            $database = grab_array_var($inargs, "database", "");

            $address = nagiosccm_replace_user_macros($address);
            $port = nagiosccm_replace_user_macros($port);
            $username = nagiosccm_replace_user_macros($username);
            $password = nagiosccm_replace_user_macros($password);

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (have_value($address) == false)
                $errmsg[$errors++] = _("No address specified.");
            if (have_value($port) == false)
                $errmsg[$errors++] = _("No port number specified.");
            if (have_value($username) == false)
                $errmsg[$errors++] = _("No username specified.");
            if (have_value($password) == false)
                $errmsg[$errors++] = _("No password specified.");
            if (have_value($database) == false)
                $errmsg[$errors++] = _("No database specified.");

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $port = grab_array_var($inargs, "port", "");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");
            $database = grab_array_var($inargs, "database", "");

            $ha = @gethostbyaddr($address);

            if ($ha == "") {
                $ha = $address;
            }

            $hostname = grab_array_var($inargs, "hostname", $ha);
            $services = grab_array_var($inargs, "services", array());
            $serviceargs = grab_array_var($inargs, "serviceargs", array());

            $services_default = array(
                "connection_time" => "on",
                "uptime" => "on",
                "threads_connected" => "on",
                "threadcache_hitrate" => "on",
                "qcache_hitrate" => "on",
                "keycache_hitrate" => "on",
                "bufferpool_hitrate" => "on",
                "log_waits" => "on",
                "tablecache_hitrate" => "",
                "index_usage" => "on",
                "slow_queries" => "on",
                "long_running_procs" => "on",

                "slave_io_running" => "",
                "slave_sql_running" => "",
                "slave_lag" => "",
            );

            $services_serial = grab_array_var($inargs, "services_serial");

            if ($services_serial != "") {
                $services = json_decode(base64_decode($services_serial), true);
            }

            if (!is_array($services)) {
                $services = grab_array_var($inargs, "services", $services_default);
            }

            $serviceargs = "";
            $serviceargs_default = array(
                "connection_time_warning" => "1",
                "connection_time_critical" => "5",
                "uptime_warning" => array("10", ""),
                "uptime_critical" => array("5", ""),
                "threads_connected_warning" => "10",
                "threads_connected_critical" => "20",
                "threadcache_hitrate_warning" => array("90", ""),
                "threadcache_hitrate_critical" => array("80", ""),
                "qcache_hitrate_warning" => array("90", ""),
                "qcache_hitrate_critical" => array("80", ""),
                "keycache_hitrate_warning" => array("99", ""),
                "keycache_hitrate_critical" => array("95", ""),
                "bufferpool_hitrate_warning" => array("99", ""),
                "bufferpool_hitrate_critical" => array("95", ""),
                "log_waits_warning" => "1",
                "log_waits_critical" => "10",
                "tablecache_hitrate_warning" => array("99", ""),
                "tablecache_hitrate_critical" => array("95", ""),
                "index_usage_warning" => array("90", ""),
                "index_usage_critical" => array("80", ""),
                "slow_queries_warning" => "0.1",
                "slow_queries_critical" => "1",
                "long_running_procs_warning" => "10",
                "long_running_procs_critical" => "20",

                "slave_lag_warning" => "15",
                "slave_lag_critical" => "30",
            );

            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial");

            if ($serviceargs_serial != "") {
                $serviceargs = json_decode(base64_decode($serviceargs_serial), true);
            }

            if (!is_array($serviceargs)) {
                $serviceargs = grab_array_var($inargs, "serviceargs", $serviceargs_default);
            }

            $select_options = [
                "Connection Time" => "connection_time",
                "Uptime" => "uptime",
                "Open Connections" => "threads_connected",
                "Thread Cache Hitrate" => "threadcache_hitrate",
                "Query Cache Hitrate" => "qcache_hitrate",
                "MyISAM Key Cache Hitrate" => "keycache_hitrate",
                "InnoDB Buffer Pool Hitrate" => "bufferpool_hitrate",
                "Log Waits" => "log_waits",
                "Tablecache Hitrate" => "tablecache_hitrate",
                "Index Usage" => "index_usage",
                "Slow Queries" => "slow_queries",
                "Long Running Processes" => "long_running_procs",
                "Slave I/O" => "slave_io_running",
                "Slave SQL" => "slave_sql_running",
                "Slave Lag" => "slave_lag",
            ];
            $service_tooltips = [ // With units
                "connection_time" => ["Monitor the time it takes to connect to the server.", "sec"],
                "uptime" => ["Monitor the time the MySQL server has been running.Lower numbers are worse and are indicative of the server having been restarted.", "min"],
                "threads_connected" => ["Monitor the number of currently open connections.", ""],
                "threadcache_hitrate" => ["Monitor the thread cache hit rate.", "%"],
                "qcache_hitrate" => ["Monitor the query cache hit rate.", "%"],
                "keycache_hitrate" => ["Monitor the MyISAM key cache hit rate.", "%"],
                "bufferpool_hitrate" => ["Monitor the InnoDB buffer pool hit rate.", "%"],
                "log_waits" => ["Monitor the InnoDB log waits due to small log buffers.", "/sec"],
                "tablecache_hitrate" => ["Monitor the table cache hit rate.", "%"],
                "index_usage" => ["Monitor the usage of indexes.", "%"],
                "slow_queries" => ["Monitor the number of slow queries.", "/sec"],
                "long_running_procs" => ["Monitor the number of long running processes.", ""],
                "slave_io_running" => ["Checks to make sure the MySQL slave I/O is running.", ""],
                "slave_sql_running" => ["Checks to make sure the MySQL slave SQL is running.", ""],
                "slave_lag" => ["Monitor the time the slave is behind the master.", "sec"],
            ];

            // Build Services Multi select HTML
            $service_list = '';
            $services_array = [];
            foreach ($select_options as $name => $service) {
                $name = encode_form_val($name);
                $safe_name = $service;
                $services_array[$name] = $service;
                $selected = '';
                if (isset($services[$service]) && $services[$service] == 'on'){ $selected = ' selected'; }
                $service_list .= "<option value='$safe_name' $selected>$name</option>";
            }

            $service_select_html = '<select name="services-select[]" id="services-select" multiple  class="form form-control metrics-select multiselect form-select">';
            $service_select_html .= $service_list;
            $service_select_html .= '</select>';


            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $hostname = nagiosccm_replace_user_macros($hostname);
            $port = grab_array_var($inargs, "port", "");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");
            $database = grab_array_var($inargs, "database", "");

            $services = grab_array_var($inargs, "services", array());
            $serviceargs = grab_array_var($inargs, "serviceargs", array());

            // check for errors
            $errors = 0;
            $errmsg = array();

            if (is_valid_host_name($hostname) == false) {
                $errmsg[$errors++] = _("Invalid host name.");
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $port = grab_array_var($inargs, "port", "");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");
            $database = grab_array_var($inargs, "database", "");

            $services_serial = grab_array_var($inargs, "services_serial");
            if ($services_serial != "")
                $services = json_decode(base64_decode($services_serial), true);
            else
                $services = grab_array_var($inargs, "services");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial");
            if ($serviceargs_serial != "")
                $serviceargs = json_decode(base64_decode($serviceargs_serial), true);
            else
                $serviceargs = grab_array_var($inargs, "serviceargs");

            $output = '

        <input type="hidden" name="ip_address" value="' . encode_form_val($address) . '">
        <input type="hidden" name="hostname" value="' . encode_form_val($hostname) . '">
        <input type="hidden" name="port" value="' . encode_form_val($port) . '">
        <input type="hidden" name="username" value="' . encode_form_val($username) . '">
        <input type="hidden" name="password" value="' . encode_form_val($password) . '">
        <input type="hidden" name="database" value="' . encode_form_val($database) . '">
        <input type="hidden" name="services_serial" value="' . base64_encode(json_encode($services)) . '">
        <input type="hidden" name="serviceargs_serial" value="' . base64_encode(json_encode($serviceargs)) . '">
';
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:


            $output = '
            ';
            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            $hostname = grab_array_var($inargs, "hostname", "");
            $address = grab_array_var($inargs, "ip_address", "");
            $hostaddress = $address;
            $port = grab_array_var($inargs, "port", "");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");
            $database = grab_array_var($inargs, "database", "");

            $services_serial = grab_array_var($inargs, "services_serial", "");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            $services = json_decode(base64_decode($services_serial), true);
            $serviceargs = json_decode(base64_decode($serviceargs_serial), true);

            // save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["hostname"] = $hostname;
            $meta_arr["ip_address"] = $address;
            $meta_arr["port"] = $port;
            $meta_arr["username"] = $username;
            $meta_arr["password"] = $password;
            $meta_arr["database"] = $database;
            $meta_arr["services"] = $services;
            $meta_arr["serivceargs"] = $serviceargs;
            save_configwizard_object_meta($wizard_name, $hostname, "", $meta_arr);

            $objs = array();

            if (!host_exists($hostname)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_mysqlserver_host",
                    "host_name" => $hostname,
                    "address" => $hostaddress,
                    "icon_image" => "mysql.png",
                    "statusmap_image" => "mysql.png",
                    "_xiwizard" => $wizard_name,
                );
            }

            // common plugin opts
            $commonopts = "--hostname=" . $address . " --port=" . $port . " --username=" . $username . " --password=\"" . $password . "\" --database='" . $database . "' ";

            foreach ($services as $svcvar => $svcval) {

                $pluginopts = "";
                $pluginopts .= $commonopts;

                switch ($svcvar) {

                    case "connection_time":

                        $pluginopts .= "--mode connection-time --warning " . $serviceargs["connection_time_warning"] . " --critical " . $serviceargs["connection_time_critical"] . "";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MySQL Connection Time",
                            "use" => "xiwizard_mysqlserver_service",
                            "check_command" => "check_xi_mysql_health!" . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "uptime":

                        $pluginopts .= "--mode uptime --warning " . $serviceargs["uptime_warning"][0] . ":" . $serviceargs["uptime_warning"][1] . " --critical " . $serviceargs["uptime_critical"][0] . ":" . $serviceargs["uptime_critical"][1];

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MySQL Uptime",
                            "use" => "xiwizard_mysqlserver_service",
                            "check_command" => "check_xi_mysql_health!" . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "threads_connected":

                        $pluginopts .= "--mode threads-connected --warning " . $serviceargs["threads_connected_warning"] . " --critical " . $serviceargs["threads_connected_critical"] . "";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MySQL Open Connections",
                            "use" => "xiwizard_mysqlserver_service",
                            "check_command" => "check_xi_mysql_health!" . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "threadcache_hitrate":

                        $pluginopts .= "--mode threadcache-hitrate --warning " . $serviceargs["threadcache_hitrate_warning"][0] . ":" . $serviceargs["threadcache_hitrate_warning"][1] . " --critical " . $serviceargs["threadcache_hitrate_critical"][0] . ":" . $serviceargs["threadcache_hitrate_critical"][1];

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MySQL Thread Cache Hit Rate",
                            "use" => "xiwizard_mysqlserver_service",
                            "check_command" => "check_xi_mysql_health!" . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "qcache_hitrate":

                        $pluginopts .= "--mode qcache-hitrate --warning " . $serviceargs["qcache_hitrate_warning"][0] . ":" . $serviceargs["qcache_hitrate_warning"][1] . " --critical " . $serviceargs["qcache_hitrate_critical"][0] . ":" . $serviceargs["qcache_hitrate_critical"][1];

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MySQL Query Cache Hit Rate",
                            "use" => "xiwizard_mysqlserver_service",
                            "check_command" => "check_xi_mysql_health!" . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "keycache_hitrate":

                        $pluginopts .= "--mode keycache-hitrate --warning " . $serviceargs["keycache_hitrate_warning"][0] . ":" . $serviceargs["keycache_hitrate_warning"][1] . " --critical " . $serviceargs["keycache_hitrate_critical"][0] . ":" . $serviceargs["keycache_hitrate_critical"][1];

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MySQL MyISAM Key Cache Hit Rate",
                            "use" => "xiwizard_mysqlserver_service",
                            "check_command" => "check_xi_mysql_health!" . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "bufferpool_hitrate":

                        $pluginopts .= "--mode bufferpool-hitrate --warning " . $serviceargs["bufferpool_hitrate_warning"][0] . ":" . $serviceargs["bufferpool_hitrate_warning"][1] . " --critical " . $serviceargs["bufferpool_hitrate_critical"][0] . ":" . $serviceargs["bufferpool_hitrate_critical"][1];

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MySQL InnoDB Buffer Pool Hit Rate",
                            "use" => "xiwizard_mysqlserver_service",
                            "check_command" => "check_xi_mysql_health!" . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "log_waits":

                        $pluginopts .= "--mode log-waits --warning " . $serviceargs["log_waits_warning"] . " --critical " . $serviceargs["log_waits_critical"] . "";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MySQL InnoDB Log Waits",
                            "use" => "xiwizard_mysqlserver_service",
                            "check_command" => "check_xi_mysql_health!" . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "tablecache_hitrate":

                        $pluginopts .= "--mode tablecache-hitrate --warning " . $serviceargs["tablecache_hitrate_warning"][0] . ":" . $serviceargs["tablecache_hitrate_warning"][1] . " --critical " . $serviceargs["tablecache_hitrate_critical"][0] . ":" . $serviceargs["tablecache_hitrate_critical"][1];

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MySQL Table Cache Hit Rate",
                            "use" => "xiwizard_mysqlserver_service",
                            "check_command" => "check_xi_mysql_health!" . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "index_usage":

                        $pluginopts .= "--mode index-usage --warning " . $serviceargs["index_usage_warning"][0] . ":" . $serviceargs["index_usage_warning"][1] . " --critical " . $serviceargs["index_usage_critical"][0] . ":" . $serviceargs["index_usage_critical"][1];

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MySQL Index Usage",
                            "use" => "xiwizard_mysqlserver_service",
                            "check_command" => "check_xi_mysql_health!" . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "slow_queries":

                        $pluginopts .= "--mode slow-queries --warning " . $serviceargs["slow_queries_warning"] . " --critical " . $serviceargs["slow_queries_critical"] . "";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MySQL Slow Queries",
                            "use" => "xiwizard_mysqlserver_service",
                            "check_command" => "check_xi_mysql_health!" . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "long_running_procs":

                        $pluginopts .= "--mode long-running-procs --warning " . $serviceargs["long_running_procs_warning"] . " --critical " . $serviceargs["long_running_procs_critical"] . "";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MySQL Long Running Processes",
                            "use" => "xiwizard_mysqlserver_service",
                            "check_command" => "check_xi_mysql_health!" . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "slave_lag":

                        $pluginopts .= "--mode slave-lag --warning " . $serviceargs["slave_lag_warning"] . " --critical " . $serviceargs["slave_lag_critical"] . "";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MySQL Slave Lag",
                            "use" => "xiwizard_mysqlserver_service",
                            "check_command" => "check_xi_mysql_health!" . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "slave_io_running":

                        $pluginopts .= "--mode slave-io-running";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MySQL Slave I/O",
                            "use" => "xiwizard_mysqlserver_service",
                            "check_command" => "check_xi_mysql_health!" . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "slave_sql_running":

                        $pluginopts .= "--mode slave-sql-running";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MySQL Slave SQL",
                            "use" => "xiwizard_mysqlserver_service",
                            "check_command" => "check_xi_mysql_health!" . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;


                    default:
                        break;
                }
            }

            //echo "OBJECTS:<BR>";
            //print_r($objs);
            //exit();

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:
            break;
    }

    return $output;
}


?>
