#!/usr/bin/env php
<?php
//
// XML Config Wizard
// Copyright (c) 2020-2021 Nagios Enterprises, LLC. All rights reserved.
//

$longopts = array ("help");
$options = getopt("w:c:k:u:h:s:r:n:R:v::", $longopts);

$usage = "    === Usage ===
    String Matching:
        php check_xml.php -u <url> -k <key> -s <string> -r <status_code> -n <status_code> [-h <header>] [-R <replacement>] [--help]
    
    Threshold Checking:
        php check_xml.php -u <url> -k <key> -c <critical_theshold> -w <warning_threshold> [-h <header>] [-R <replacement>] [--help]

    === Arguments ===
    Required:
        -u <url>            URL to check. This must be a valid URL and return XML data.
        -k <key>            XML variable key to check (e.g. node.subnode.value)
    
    String Matching:
        -s <string>         String to match against the key value
        -r <status_code>    (optional) Status code if <string> matches <key> value (defaults to 0 when only -s used)
        -n <status_code>    (optional) Status code if <string> does not match <key> value (defaults to 2 when only -s is used)

    Threshold Checking:
        -c <critical>       Critical threshold to check against the key value (numeric)
        -w <warning>        Warning threshold to check against the key value (numeric)

    Optional Arguments:
        -h <header>         Additional 'Accept: application/xml' header is added to the request
        -R <replacement>    Replaces colons (:) in key names with the provided string (we recommend using _)
        -v <verbose>        Show verbose output for debugging purposes
        --help              This help screen\n";

if (isset($options["help"])) {
    echo "$usage";
    exit(0);
}

// Validate required parameters first
if (!isset($options["u"])) {
    echo "UNKNOWN - Missing a required parameter (-u is required)\n";
    exit(3);
}

if (!isset($options["k"])) {
    echo "UNKNOWN - Missing a required parameter (-k is required)\n";
    exit(3);
}

// Validate a check
if (isset($options["c"]) && !isset($options["w"])) {
    echo "UNKNOWN - Missing required parameters (-w is required if -c is used)\n";
    exit(3);
}

$verbose = isset($options["v"]);

// Set up headers
$headers = array('Accept: application/xml');
if (isset($options["h"])) {
    $headers[] = $options["h"];
}

// Check for replacement
if (isset($options["R"])) {
    $replace = $options["R"];
}

$ch = curl_init();
curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
curl_setopt($ch, CURLOPT_URL, $options["u"]);
curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);

$chData = curl_exec($ch);

curl_close($ch);

// Replace all : with _ to temporarily fix issues with child notes
if (isset($replace)) {
    if ($verbose) {
        echo "Replacing colons (:) with $replace\n";
        echo "=== Original XML ===\n";
        echo $chData;
    }

    $chData = preg_replace('/(?<!:)(?<!http)(?<!https):/', $replace, $chData);

    if ($verbose) {
        echo "\n\n=== Replaced XML ===\n";
        echo $chData;
    }
}

$chDataArray = simplexml_load_string($chData);

// Error parsing XML data
if (!$chDataArray) {
    echo "UNKNOWN - Could not parse XML data\n";
    exit(3);
}

$xml_data_array = xmlobj_to_array($chDataArray);

// Find the key
if (isset($options["k"])) {
    // Break up variable name so it displays properly
    $k = str_replace(array('[', ']', '"', "'"), array('.', '', '', ''), $options["k"]);
    if (isset($replace)) {
        $k = str_replace(':', $replace, $k);
    }
    $associative_keys = explode(".", $k);

    if ($verbose) {
        echo "\n\nSearching for keys:\n";
        print_r($associative_keys);
        echo "\n";
        echo "In data:\n";
        print_r($xml_data_array);
    }

    // Check each key and continue
    foreach ($associative_keys as $key) {
        if (isset($xml_data_array[$key])) {
            $xml_data_array = $xml_data_array[$key];
            if ($verbose) {
                echo "Found key: $key\n";
                echo "Value: " . json_encode($xml_data_array) . "\n";
            }
        } else {
            if ($verbose) {
                echo "Could not find key: $key\n";
            }
            echo "UNKNOWN - Could not find the specified key\n";
            exit(3);
        }
    }
}

if (isset($options["s"])) {
    if (is_numeric($xml_data_array)) {
        echo "UNKNOWN - The key's value is numeric for a string match check\n";
        exit(3);
    }

    $match_status_code = grab_array_var($options, "r", 0);
    $no_match_status_code = grab_array_var($options, "n", 2);

    $string = $options["s"];

    if ($string == $xml_data_array) {
        $status_code = intval($match_status_code);
        echo get_status_msg($status_code) . " - Value '" . $xml_data_array . "' matched '" . $string . "'\n";
    } else {
        $status_code = intval($no_match_status_code);
        echo get_status_msg($status_code) . " - Value '" . $xml_data_array . "' did not match '" . $string . "'\n";
    }

    exit($status_code);
    
} else {
    if (!is_numeric($xml_data_array)) {
        echo "UNKNOWN - The key's value is not numeric for a numeric check\n";
        exit(3);
    }

    if (isset($options["w"])) {
        $warning = new Nagios_Plugin_Range($options["w"]);
    }
    
    if (isset($options["c"])) {
        $critical = new Nagios_Plugin_Range($options["c"]);
    }

    if ($critical->check_range($xml_data_array)) {
        echo "CRITICAL - " . $options["k"] . " = " . $xml_data_array . "|" . $options["k"] . "=" . $xml_data_array . "\n";
        exit(2);
    }

    if ($warning->check_range($xml_data_array)) {
        echo "WARNING - " . $options["k"] . " = " . $xml_data_array . "|" . $options["k"] . "=" . $xml_data_array . "\n";
        exit(1);
    }

    echo "OK - " . $options["k"] . " = " . $xml_data_array . "|" . $options["k"] . "=" . $xml_data_array . "\n";
    exit(0);
}


function get_status_msg($status_code)
{
    $msg = "UNKNOWN";

    switch ($status_code)
    {
        case 0:
            $msg = "OK";
            break;
        case 1:
            $msg = "WARNING";
            break;
        case 2:
            $msg = "CRITICAL";
            break;
    }

    return $msg;
}


// Gets a value from an array, returning a default value if not found
function grab_array_var($a, $k, $d=null)
{
    return (is_array($a) && array_key_exists($k, $a)) ? $a[$k] : $d;
}


// Convert XML object to array
function xmlobj_to_array($xmlobj)
{
    return @json_decode(@json_encode($xmlobj), true);
}


/*
 * Created on Sep 14, (c) 2006 by Marcel K�hn
 * GPL licensed, marcel@kuehns.org
 *
 * Ported nagios range checking from Nagios::Plugin::Range
 * by Doug Warner <doug@warner.fm>
 *
 * Cleaned up and improved Oct 12, (c) 2018 by Joe Momma
 */

define('STATE_OK',0);
define('STATE_WARNING',1);
define('STATE_CRITICAL',2);
define('STATE_UNKNOWN',3);

class Nagios_Plugin_Range {

    const INSIDE = 1;
    const OUTSIDE = 0;
    protected $range;
    protected $start = 0;
    protected $end = 0;
    protected $start_infinity = false;
    protected $end_infinity = false;
    protected $alert_on;

    public function __construct($range) {
        $this->range = $range;
        $this->parse_range_string($range);
    }

    public function valid() {
        return isset($this->alert_on);
    }

    public function check_range($value) {
        $false = false;
        $true = true;
        if ($this->alert_on == self::INSIDE) {
            $false = true;
            $true = false;
        }

        if (!$this->end_infinity && !$this->start_infinity ) {
            if ($this->start <= $value && $value <= $this->end) {
                return $false;
            } else {
                return $true;
            }
        }
        elseif (!$this->start_infinity && $this->end_infinity) {
            if ( $value >= $this->start ) {
                return $false;
            } else {
                return $true;
            }
        }
        elseif ($this->start_infinity && !$this->end_infinity) {
            if ($value <= $this->end) {
                return $false;
            } else {
                return $true;
            }
        }
        else {
            return $false;
        }
    }

    # Returns a N::P::Range object if the string is a conforms to a Nagios Plugin range string, otherwise null
    protected function parse_range_string($string) {
        $valid = 0;
        $alert_on = self::OUTSIDE;

        $string = preg_replace('/\s/', '', $string);

        $value = "[-+]?[\d\.]+";
        $value_re = "{$value}(?:e{$value})?";

        # check for valid range definition
        if ( !preg_match('/[\d~]/', $string)
            || !preg_match("/^\@?({$value_re}|~)?(:({$value_re})?)?$/", $string)) {
            echo "invalid range definition '$string'";
            exit(STATE_UNKNOWN);
        }

        if (preg_match('/^\@/', $string)) {
            $string = preg_replace('/^\@/', '', $string);
            $alert_on = self::INSIDE;
        }

        # '~:x'
        if (preg_match('/^~/', $string)) {
            $string = preg_replace('/^~/', '', $string);
            $this->start_infinity = true;
        }

        # '10:'
        if (preg_match("/^({$value_re})?:/", $string, $matches)) {
            if (!empty($matches[1])) {
                $this->set_range_start($matches[1]);
            }
            $this->end_infinity = true;  # overridden below if there's an end specified
            $string = preg_replace("/^({$value_re})?:/", '', $string);
            $valid++;
        }

        # 'x:10' or '10'
        if (preg_match("/^({$value_re})$/", $string)) {
            $this->set_range_end($string);
            $valid++;
        }

        if ($valid
            && ($this->start_infinity || $this->end_infinity || $this->start <= $this->end)) {
            $this->alert_on = $alert_on;
        }
    }

    protected function set_range_start($value) {
        $this->start = (float) $value;
        if (empty($this->start)) {
            $this->start = 0;
        }
        $this->start_infinity = false;
    }

    protected function set_range_end($value) {
        $this->end = (float) $value;
        if (empty($this->end)) {
            $this->end = 0;
        }
        $this->end_infinity = false;
    }

}
