<?php
//
// Postgres DB Config Wizard
// Copyright (c) 2010-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

postgresdb_configwizard_init();

function postgresdb_configwizard_init()
{
    $name = "postgresdb";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.0.1",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor a Postgres database."),
        CONFIGWIZARD_DISPLAYTITLE => _("Postgres Database"),
        CONFIGWIZARD_FUNCTION => "postgresdb_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "postgres.png",
        CONFIGWIZARD_FILTER_GROUPS => array('database'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );
    register_configwizard($name, $args);
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function postgresdb_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    $wizard_name = "postgresdb";

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            $address = grab_array_var($inargs, "ip_address", "");
            $port = grab_array_var($inargs, "port", "5432");
            $username = grab_array_var($inargs, "username", "postgres");
            $password = grab_array_var($inargs, "password", "");
            $database = grab_array_var($inargs, "database", "postgres");

            # These will only be set by going back from step 2.
            $services = grab_array_var($inargs, "services");
            $serviceargs = grab_array_var($inargs, "serviceargs");

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address", "");
            $port = grab_array_var($inargs, "port", "");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");
            $database = grab_array_var($inargs, "database", "");

            $address = nagiosccm_replace_user_macros($address);
            $port = nagiosccm_replace_user_macros($port);
            $username = nagiosccm_replace_user_macros($username);
            $password = nagiosccm_replace_user_macros($password);

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (have_value($address) == false)
                $errmsg[$errors++] = _("No address specified.");
            if (have_value($port) == false)
                $errmsg[$errors++] = _("No port number specified.");
            if (have_value($username) == false)
                $errmsg[$errors++] = _("No username specified.");
            if (have_value($database) == false)
                $errmsg[$errors++] = _("No database specified.");

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $port = grab_array_var($inargs, "port", "");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");
            $database = grab_array_var($inargs, "database", "");

            $ha = @gethostbyaddr($address);

            if ($ha == "") {
                $ha = $address;
            }

            $hostname = grab_array_var($inargs, "hostname", $ha);
            $services = "";

            $services_default = array(
                "connection" => "on",
                "database_size" => "on",
                "table_sizes" => "on",
                "relation_sizes" => "on",
                "sequences" => "on",
            );

            $services_serial = grab_array_var($inargs, "services_serial");

            if ($services_serial != "") {
                $services = json_decode(base64_decode($services_serial), true);
            }

            if (!is_array($services)) {
                $services = grab_array_var($inargs, "services", $services_default);
            }

            $serviceargs = "";

            $serviceargs_default = array(
                "database_size_warning" => "500MB",
                "database_size_critical" => "1GB",
                "table_sizes_warning" => "200MB",
                "table_sizes_critical" => "400MB",
                "relation_sizes_warning" => "50MB",
                "relation_sizes_critical" => "100MB",
                "sequences_warning" => "30%",
                "sequences_critical" => "10%",
            );

            $select_options =[
                "Connection Status" => "connection",
                "Database Size" => "database_size",
                "Table Sizes" => "table_sizes",
                "Relation Sizes" => "relation_sizes",
                "Sequences" => "sequences",
            ];

            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial");

            if ($serviceargs_serial != "") {
                $serviceargs = json_decode(base64_decode($serviceargs_serial), true);
            }

            if (!is_array($serviceargs)) {
                $serviceargs = grab_array_var($inargs, "serviceargs", $serviceargs_default);
            }

            // Build Multi select HTML
            $select_options_html = '';
            foreach ($select_options as $opt => $val) {
                $selected = '';
                if ($services[$val] == 'on') { $selected = ' selected'; }
                $select_options_html .= '<option value="'.$val.'"'.$selected.'>'.$opt.'</option>';
            }

            $select_html = '<select name="metrics[]" id="metrics" multiple  class="form form-control metrics-select multiselect form-select">';
            $select_html .= $select_options_html;
            $select_html .= '</select>';

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $hostname = nagiosccm_replace_user_macros($hostname);
            $port = grab_array_var($inargs, "port", "");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");
            $database = grab_array_var($inargs, "database", "");

            $services = grab_array_var($inargs, "services", array());
            $serviceargs = grab_array_var($inargs, "serviceargs", array());
            $services_serial = (!empty($services)) ? base64_encode(json_encode($services)) : "";
            $serviceargs_serial = (!empty($serviceargs)) ? base64_encode(json_encode($serviceargs)) : "";

            // Check for errors
            $errors = [];

            if (!count($services)) {
            $errors[] = _("You must select at least one service for monitoring.");
}

            if (is_valid_host_name($hostname) == false) {
                $errors[] = _("Invalid host name.");
            }

            if (count($errors)) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errors;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $port = grab_array_var($inargs, "port", "");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");
            $database = grab_array_var($inargs, "database", "");

                            $services = grab_array_var($inargs, "services");
                            $serviceargs = grab_array_var($inargs, "serviceargs");

            $services_serial = (!empty($services)) ? base64_encode(json_encode($services)) : grab_array_var($inargs, "services_serial", "");
            $serviceargs_serial = (!empty($serviceargs)) ? base64_encode(json_encode($serviceargs)) : grab_array_var($inargs, "serviceargs_serial", "");

            $output = '

        <input type="hidden" name="ip_address" value="' . encode_form_val($address) . '">
        <input type="hidden" name="hostname" value="' . encode_form_val($hostname) . '">
        <input type="hidden" name="port" value="' . encode_form_val($port) . '">
        <input type="hidden" name="username" value="' . encode_form_val($username) . '">
        <input type="hidden" name="password" value="' . encode_form_val($password) . '">
        <input type="hidden" name="database" value="' . encode_form_val($database) . '">
        <input type="hidden" name="services_serial" value="' . $services_serial . '">
        <input type="hidden" name="serviceargs_serial" value="' . $serviceargs_serial . '">
';
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:


            $output = '
            ';
            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            $hostname = grab_array_var($inargs, "hostname", "");
            $address = grab_array_var($inargs, "ip_address", "");
            $hostaddress = $address;
            $port = grab_array_var($inargs, "port", "");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");
            $database = grab_array_var($inargs, "database", "");

            $services_serial = grab_array_var($inargs, "services_serial", "");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            $services = json_decode(base64_decode($services_serial), true);
            $serviceargs = json_decode(base64_decode($serviceargs_serial), true);

            // save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["hostname"] = $hostname;
            $meta_arr["ip_address"] = $address;
            $meta_arr["port"] = $port;
            $meta_arr["username"] = $username;
            $meta_arr["password"] = $password;
            $meta_arr["database"] = $database;
            $meta_arr["services"] = $services;
            $meta_arr["serivceargs"] = $serviceargs;
            save_configwizard_object_meta($wizard_name, $hostname, "", $meta_arr);

            $objs = array();

            if (!host_exists($hostname)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_postgresdb_host",
                    "host_name" => $hostname,
                    "address" => $hostaddress,
                    "icon_image" => "postgres.png",
                    "statusmap_image" => "postgres.png",
                    "_xiwizard" => $wizard_name,
                );
            }

            $commonopts = "-H " . $address . " --port=" . $port . " --dbuser=" . $username . " --dbname=" . $database . " ";

            foreach ($services as $svcvar => $svcval) {

                $pluginopts = "";
                $pluginopts .= $commonopts;

                if ($password != "")
                    $pluginopts .= " --dbpass=\"" . $password . "\"";

                switch ($svcvar) {

                    case "connection":

                        $pluginopts .= " --action=connection";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Postgres Database Connection - " . $database,
                            "use" => "xiwizard_postgresdb_service",
                            "check_command" => "check_xi_postgres_db!" . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "database_size":

                        $pluginopts .= " --action=database_size";
                        if ($serviceargs["database_size_warning"] != "")
                            $pluginopts .= " --warning=" . $serviceargs["database_size_warning"] . "";
                        if ($serviceargs["database_size_critical"] != "")
                            $pluginopts .= " --critical=" . $serviceargs["database_size_critical"] . "";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Postgres Database Size - " . $database,
                            "use" => "xiwizard_postgresdb_service",
                            "check_command" => "check_xi_postgres_db!" . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "table_sizes":

                        $pluginopts .= " --action=table_size";
                        if ($serviceargs["table_sizes_warning"] != "")
                            $pluginopts .= " --warning=" . $serviceargs["table_sizes_warning"] . "";
                        if ($serviceargs["table_sizes_critical"] != "")
                            $pluginopts .= " --critical=" . $serviceargs["table_sizes_critical"] . "";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Postgres Table Sizes - " . $database,
                            "use" => "xiwizard_postgresdb_service",
                            "check_command" => "check_xi_postgres_db!" . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "relation_sizes":

                        $pluginopts .= " --action=relation_size";
                        if ($serviceargs["relation_sizes_warning"] != "")
                            $pluginopts .= " --warning=" . $serviceargs["relation_sizes_warning"] . "";
                        if ($serviceargs["relation_sizes_critical"] != "")
                            $pluginopts .= " --critical=" . $serviceargs["relation_sizes_critical"] . "";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Postgres Relation Sizes - " . $database,
                            "use" => "xiwizard_postgresdb_service",
                            "check_command" => "check_xi_postgres_db!" . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "sequences":

                        $pluginopts .= " --action=sequence";
                        if ($serviceargs["sequences_warning"] != "")
                            $pluginopts .= " --warning=" . $serviceargs["sequences_warning"] . "";
                        if ($serviceargs["sequences_critical"] != "")
                            $pluginopts .= " --critical=" . $serviceargs["sequences_critical"] . "";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Postgres Database Sequences - " . $database,
                            "use" => "xiwizard_postgresdb_service",
                            "check_command" => "check_xi_postgres_db!" . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;


                    default:
                        break;
                }
            }

            //echo "OBJECTS:<BR>";
            //print_r($objs);
            //exit();

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:
            break;
    }

    return $output;
}


?>
