#!/usr/bin/env python3
import winrm
import argparse
import getpass
import os
from sys import argv

try:
    # Parse command-line arguments
    parser = argparse.ArgumentParser(description='winrm_check_cpu.py uses WinRM to obtain CPU statistics on a remote, Windows-based system.')
    parser.add_argument('-H', '--host', required=True, type=str, help='IP address or host name of the Windows system.')
    parser.add_argument('-u', '--user', required=True, type=str, help='Username for connecting to the Windows system.')
    parser.add_argument('-a', '--auth', required=True, type=str, choices=['basic-http', 'basic-https'], help='Authentication mechanism for the Windows system. Strongly recommended you avoid basic-http.')
    parser.add_argument('-p', '--password', required=True, type=str, help='Password for connecting to the Windows system.')
    parser.add_argument('-C', '--certvalidation', required=False, action="count", default=0)

    parser.add_argument('-f', '--format', required=False, type=str, choices=['KB', 'MB', 'GB'], help='Format you want your metrics measured in. (KB, MB, GB)', default="MB")
    parser.add_argument('-w', '--warning', required=False, type=float, help='Percent usage at which a warning alert with be sent.', default=0)
    parser.add_argument('-c', '--critical', required=False, type=float, help='Percent usage at which a critical alert with be sent', default=0)

    args = parser.parse_args(argv[1:])

except argparse.ArgumentError as e:
    print(f"\nArgument error: {str(e)}")
    exit(1)
except Exception as e:
    print(f"\nAn error occurred during argument parsing: {str(e)}")
    exit(1)

try:
    authentication = None

    if args.auth == 'basic-http':
        # Use basic-http Authentication
        authentication = 'basic'

    elif args.auth == 'basic-https':
        # Use basic-https Authentication
        authentication = 'ssl'

    elif args.auth == 'cert':
        # Use Certificate Authentication (TLS transport with SSL enabled)
        authentication = winrm.transport.TlsTransport(ssl=True)


    # memory usage powershell script
    ps_memory = f"""
    $strComputer = $Host
    Clear

    @{{
        total=(WmiObject Win32_ComputerSystem).TotalPhysicalMemory;
        free=(WmiObject Win32_OperatingSystem).FreePhysicalMemory;
        available=(get-counter -counter '\Memory\\available bytes').countersamples.cookedvalue;
    }} | ConvertTo-Json -Compress
    """


    # Create a WinRM session with the provided host, user, password, and authentication method
    if args.certvalidation > 0:
        winrmsession = winrm.Session(args.host, auth=(args.user, args.password), transport=authentication)
    else:
        winrmsession = winrm.Session(args.host, auth=(args.user, args.password), transport=authentication, server_cert_validation='ignore')


    # run the ps script and decode the output
    r = winrmsession.run_ps(ps_memory)
    final_dictionary = eval(r.std_out.decode('utf-8'))
 
    byte_format = {
        "TB": 1099511627776,
        "GB": 1073741824,
        "MB": 1048576,
        "KB": 1024
    }
    # calculate the memory metrics
    total_memory = round(final_dictionary['total']  / byte_format[args.format], 2)
    free_memory = round((final_dictionary['free'] * 1024) / byte_format[args.format], 2) # returns as Kb so convert back down to bytes to make calcs easier
    available_memory = round(final_dictionary['available']  / byte_format[args.format], 2)
    
    used_memory = round(total_memory - available_memory, 2)
    percent_used_memory = round((used_memory/total_memory) * 100, 2)

    status_dict = {
        0: "OK",
        1: "WARNING",
        2: "CRITICAL",
        3: "UNKNOWN"
    }

    status_code = 0
    if args.warning and percent_used_memory >= args.warning:
        status_code = 1
    if args.critical and percent_used_memory >= args.critical:
        status_code = 2
    
    # build and return the status message/perf data
    memory_warning = args.warning if args.warning else ''
    memory_critical = args.critical if args.critical else '' 
    message = f"{str(status_dict[status_code])}: Memory usage is {str(percent_used_memory)}% (Available: {str(available_memory)} {args.format}, Total: {str(total_memory)} {args.format}, Free: {str(free_memory)} {args.format}, Used: {str(used_memory)} {args.format}) | 'used'={str(used_memory)}{args.format};{str(memory_warning)};{str(memory_critical)}; 'total'={str(total_memory)}{args.format}; 'available'={str(available_memory)}{args.format}; 'free'={str(free_memory)}{args.format};"
    
    print(message)
    exit(status_code)

except winrm.exceptions.WinRMTransportError as e:
    print(f"WinRM transport error: {str(e)}")
    exit(4)
except Exception as e:
    print(f"An error occurred during WinRM session setup or command execution: {str(e)}")
    exit(4)