#!/usr/bin/env python3

import requests
import argparse
import sys
import time
from decimal import Decimal
import re

STATE_OK = 0
STATE_WARNING = 1
STATE_CRITICAL = 2
STATE_UNKNOWN = 3


def evaluate_threshold(value, threshold):
    """
    Evaluate if a value should trigger an alert based on the threshold
    0 = OK 
    1 = WARN/CRIT   
    """
    if threshold is None or len(threshold) == 0:
        return 0
    value = float(value)

    alert_range = [None, None] # [lower_bound, upper_bound]
    inverted_range = False

    # Determine the type of range and parse boundaries per the guidelines (https://nagios-plugins.org/doc/guidelines.html#THRESHOLDFORMAT)
    if threshold.startswith('@'):
        inverted_range = True
        threshold = threshold[1:]

    if ':' not in threshold:
        # number - Lower bound is 0
        alert_range[0] = 0.0
        alert_range[1] = float(threshold)
    else:
        parts = threshold.split(':')
        if threshold.startswith('~:') or parts[0] == '':
            # ~:number or :number - Lower bound is negative infinity
            alert_range[0] = float('-inf')
            alert_range[1] = float(parts[1]) if parts[1] else float('inf')
        elif parts[1] == '':
            # number: - Upper bound is positive infinity
            alert_range[0] = float(parts[0])
            alert_range[1] = float('inf')
        else:
            # number:number - Both bounds are specified
            alert_range[0] = float(parts[0])
            alert_range[1] = float(parts[1])

    in_alert_range = alert_range[0] <= value <= alert_range[1]

    if inverted_range:
        in_alert_range = not in_alert_range
    
    return 0 if in_alert_range else 1

# Function to query Windows Exporter metrics
def get_metrics(prometheus_url):
    try:
        response = requests.get(prometheus_url)
        response.raise_for_status()
        return response.text
    except requests.exceptions.RequestException as e:
        print(f"CRITICAL - Failed to query Prometheus - {e}")
        sys.exit(2)

# Function to parse metrics
def parse_metrics(metrics_text):
    metrics = {}
    for line in metrics_text.splitlines():
        if line.startswith('#') or not line.strip():
            continue
        if "TYPE" in line or "HELP" in line:
            continue
        parts = line.split()
        if len(parts) > 2:
            parts = [" ".join(parts[:-1]), parts[-1]]
        if len(parts) == 2:
            metrics[parts[0]] = float(parts[1])
    return metrics

# Function to check CPU usage
def check_cpu(metrics, previous_metrics, warning, critical):
    cpu_idle = sum(value for key, value in metrics.items() if 'windows_cpu_time_total' in key and 'idle' in key)
    cpu_total = sum(value for key, value in metrics.items() if 'windows_cpu_time_total' in key)
    prev_cpu_idle = sum(value for key, value in previous_metrics.items() if 'windows_cpu_time_total' in key and 'idle' in key)
    prev_cpu_total = sum(value for key, value in previous_metrics.items() if 'windows_cpu_time_total' in key)
    if not cpu_idle or not cpu_total or not prev_cpu_idle or not prev_cpu_total:
        return "UNKNOWN - Required CPU metrics not found", STATE_UNKNOWN

    cpu_usage = 100 * (1 - ((cpu_idle - prev_cpu_idle) / (cpu_total - prev_cpu_total)))
    perf_data = f" | 'cpu_usage'={cpu_usage:.2f}%;{warning};{critical};0;100"

    if evaluate_threshold(cpu_usage, critical):
        return f"CRITICAL - CPU usage is {cpu_usage:.2f}%{perf_data}", STATE_CRITICAL
    elif evaluate_threshold(cpu_usage, warning):
        return f"WARNING - CPU usage is {cpu_usage:.2f}%{perf_data}", STATE_WARNING
    else:
        return f"OK - CPU usage is {cpu_usage:.2f}%{perf_data}", STATE_OK

# Function to check memory usage
def check_memory(metrics, warning, critical):
    mem_total = metrics.get('windows_memory_physical_total_bytes', 0)
    mem_available = metrics.get('windows_memory_available_bytes', 0)
    if not mem_total or not mem_available:
        return "UNKNOWN - Required memory metrics not found", STATE_UNKNOWN

    mem_usage = 100 * (1 - (mem_available / mem_total))
    perf_data = f" | 'memory_usage'={mem_usage:.2f}%;{warning};{critical};0;100"

    if evaluate_threshold(mem_usage, critical):
        return f"CRITICAL - Memory usage is {mem_usage:.2f}%{perf_data}", STATE_CRITICAL
    elif evaluate_threshold(mem_usage, warning):
        return f"WARNING - Memory usage is {mem_usage:.2f}%{perf_data}", STATE_WARNING
    else:
        return f"OK - Memory usage is {mem_usage:.2f}%{perf_data}", STATE_OK

# Function to check disk usage
def check_disk(metrics, warning, critical):
    disk_total = sum(value for key, value in metrics.items() if 'windows_logical_disk_size_bytes' in key)
    disk_free = sum(value for key, value in metrics.items() if 'windows_logical_disk_free_bytes' in key)
    if not disk_total or not disk_free:
        return "UNKNOWN - Required disk metrics not found", STATE_UNKNOWN

    disk_usage = 100 * (1 - (disk_free / disk_total))
    perf_data = f" | 'disk_usage'={disk_usage:.2f}%;{warning};{critical};0;100"

    if evaluate_threshold(disk_usage, critical):
        return f"CRITICAL - Disk usage is {disk_usage:.2f}%{perf_data}", STATE_CRITICAL
    elif evaluate_threshold(disk_usage, warning):
        return f"WARNING - Disk usage is {disk_usage:.2f}%{perf_data}", STATE_WARNING
    else:
        return f"OK - Disk usage is {disk_usage:.2f}%{perf_data}", STATE_OK

# Function to check custom metric
def check_custom_metric(metrics, custom_metric, warning, critical):
    value = metrics.get(custom_metric, None)
    if value is None:
        return f"UNKNOWN - Custom metric {custom_metric} not found", STATE_UNKNOWN

    # Ensure value is converted to a float and handle scientific notation
    if 'e' in str(value):
        value = Decimal(value)
    
    unit = ""
    perf_unit = ""
    if 'bytes' in custom_metric:
        unit = " bytes"
        perf_unit = "bytes"
    elif 'percent' in custom_metric:
        unit = "%"
        perf_unit = "%"
    elif 'seconds' in custom_metric:
        unit = " seconds"
        perf_unit = "seconds"

    value = round(Decimal(value))
    custom_metric_escaped = custom_metric.replace('"', "_").replace("{", "_").replace("}", "_").replace("=", "_").replace(",", "_")
    
    perf_data = f" | '{custom_metric_escaped}'={value}{perf_unit};{warning};{critical}"
    if evaluate_threshold(value, critical):
        return f"CRITICAL - {custom_metric} is {value:,}{unit}{perf_data}", STATE_CRITICAL
    elif evaluate_threshold(value, warning):
        return f"WARNING - {custom_metric} is {value:,}{unit}{perf_data}", STATE_WARNING
    else:
        return f"OK - {custom_metric} is {value:,}{unit}{perf_data}", STATE_OK

# Main function to check metrics
def check_metrics(args):
    prometheus_url = f"http://{args.ip}:{args.port}/metrics"
    
    # Get initial metrics
    metrics_text = get_metrics(prometheus_url)
    metrics = parse_metrics(metrics_text)
    
    # Wait for the interval to get the next set of metrics
    time.sleep(args.interval)
    
    # Get the next set of metrics
    metrics_text = get_metrics(prometheus_url)
    new_metrics = parse_metrics(metrics_text)

    status = 0
    results = []

    if args.cpu:
        result, code = check_cpu(new_metrics, metrics, args.cpu_warning, args.cpu_critical)
        results.append((result, code))
        status = max(status, code)

    if args.mem:
        result, code = check_memory(new_metrics, args.mem_warning, args.mem_critical)
        results.append((result, code))
        status = max(status, code)

    if args.disk:
        result, code = check_disk(new_metrics, args.disk_warning, args.disk_critical)
        results.append((result, code))
        status = max(status, code)

    if args.custom_metric:
        result, code = check_custom_metric(new_metrics, args.custom_metric, args.custom_warning, args.custom_critical)
        results.append((result, code))
        status = max(status, code)

    # Sort results by status code: CRITICAL (2), WARNING (1), UNKNOWN (3), OK (0)
    results.sort(key=lambda x: x[1], reverse=True)
    output = ", ".join(result for result, code in results)
    print(output)
    sys.exit(status)

# Argument parsing
def parse_arguments():
    parser = argparse.ArgumentParser(description="Check Windows Exporter metrics.")
    parser.add_argument("-H", "--ip", required=True, help="IP address of the Windows Exporter machine")
    parser.add_argument("-P", "--port", required=True, help="Port of the Windows Exporter machine")
    parser.add_argument("--cpu", action="store_true", help="Check CPU usage")
    parser.add_argument("--mem", action="store_true", help="Check memory usage")
    parser.add_argument("--disk", action="store_true", help="Check disk usage")
    parser.add_argument("--custom-metric", help="Check custom metric")
    parser.add_argument("--interval", type=int, default=5, help="Interval in seconds between metric checks")
    parser.add_argument("--cpu-warning", type=str, default="70", help="Warning threshold for CPU usage (Nagios range format)")
    parser.add_argument("--cpu-critical", type=str, default="90", help="Critical threshold for CPU usage (Nagios range format)")
    parser.add_argument("--mem-warning", type=str, default="70", help="Warning threshold for memory usage (Nagios range format)")
    parser.add_argument("--mem-critical", type=str, default="90", help="Critical threshold for memory usage (Nagios range format)")
    parser.add_argument("--disk-warning", type=str, default="70", help="Warning threshold for disk usage (Nagios range format)")
    parser.add_argument("--disk-critical", type=str, default="90", help="Critical threshold for disk usage (Nagios range format)")
    parser.add_argument("--custom-warning", type=str, default="70", help="Warning threshold for custom metric (Nagios range format)")
    parser.add_argument("--custom-critical", type=str, default="90", help="Critical threshold for custom metric (Nagios range format)")
    try:
        args = parser.parse_args()
        return args
    except Exception as e:
        print(f"UNKNOWN - Error parsing arguments: {e}")
        sys.exit(STATE_UNKNOWN)

if __name__ == "__main__":
    args = parse_arguments()
    check_metrics(args)
