<?php
//
// Prometheus Config Wizard
// Copyright (c) 2025 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

prometheus_configwizard_init();

function prometheus_configwizard_init()
{
    $name = "prometheus";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "1.0.0",
        CONFIGWIZARD_IS_PREMIUM => TRUE,
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor Linux machines using Prometheus Node Exporter or Windows machines using Windows Exporter."),
        CONFIGWIZARD_DISPLAYTITLE => _("Prometheus"),
        CONFIGWIZARD_FUNCTION => "prometheus_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "prometheus.png",
        CONFIGWIZARD_FILTER_GROUPS => array('linux', 'windows'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );
    register_configwizard($name, $args);
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function prometheus_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    $wizard_name = "prometheus";

    // initialize return code and output
    $result = 0;
    $output = "";

    // initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:
            // Remove any existing session variables if this is the first time on the page
            if (isset($_POST['backButton']) == false) {
                unset($_SESSION['prometheus_wizard_hosts']);
                unset($_SESSION['prometheus_wizard_port']);
                $_SESSION['prometheus_wizard_services'] = array(
                    'cpu' => 'on',
                    'disk' => 'on',
                    'memory' => 'on'
                );
                unset($_SESSION['prometheus_wizard_serviceargs']);
                unset($_SESSION['prometheus_wizard_custom_metrics']);
                unset($_SESSION['prometheus_wizard_parsed_metrics']);
                unset($_SESSION['prometheus_wizard_os_type']);
                unset($_SESSION['prometheus_wizard_parsed_metrics_count']);
            }

            $os_type = grab_array_var($inargs, "os_type", '');
            $hosts = grab_array_var($inargs, 'hosts', '');
            $port = grab_array_var($inargs, "port", 9100);

            if (array_key_exists('prometheus_wizard_os_type', $_SESSION) && $os_type == "") {
                $os_type = $_SESSION['prometheus_wizard_os_type'];
            }
            if (array_key_exists('prometheus_wizard_hosts', $_SESSION) && $hosts == "") {
                $hosts = $_SESSION['prometheus_wizard_hosts'];
            }
            if (array_key_exists('prometheus_wizard_port', $_SESSION)) {
                $port = $_SESSION['prometheus_wizard_port'];
            }

            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:
            // Get variables that were passed to us
            $os_type = grab_array_var($inargs, "os_type", "");
            $hosts = grab_array_var($inargs, "hosts", "");
            $port = grab_array_var($inargs, "port", "");

            if (array_key_exists('prometheus_wizard_os_type', $_SESSION) && $os_type == "") {
                $os_type = $_SESSION['prometheus_wizard_os_type'];
            }
            if (array_key_exists('prometheus_wizard_hosts', $_SESSION) && $hosts == "") {
                $hosts = $_SESSION['prometheus_wizard_hosts'];
            }
            if (array_key_exists('prometheus_wizard_port', $_SESSION)) {
                $port = $_SESSION['prometheus_wizard_port'];
            }

            // Check for errors
            $errors = 0;
            $errmsg = array();

            $host = filter_var(trim(explode("\n", $hosts)[0]), FILTER_UNSAFE_RAW); 
            $port = filter_var($port, FILTER_SANITIZE_NUMBER_INT);
            $url = "http://$host:$port/metrics";
            $context = stream_context_create([
                'http' => [
                    'timeout' => 5,
                    'ignore_errors' => true
                ]
            ]);
            $response = file_get_contents($url, false, $context);

            if ($response === FALSE) {
                $errmsg[$errors++] = _("Cannot scrape metrics from " . $url . ". Ensure Prometheus Node Exporter or Windows Exporter is running and accessible.");
            } else {
                $metrics_array = array_filter(explode("\n", $response)); 

                $parsed_metrics = [];
                $metrics_count = 0;
                $current_category = '';

                foreach ($metrics_array as $line) {
                    $line_content = explode(' ', $line);

                    // Check for HELP line and create a new category
                    if (isset($line_content[1]) && $line_content[1] === 'HELP') {
                        $current_category = $line_content[2];
                        $parsed_metrics[$current_category] = [
                            "tooltip" => implode(' ', array_slice($line_content, 3)),
                            "metric_type" => '',
                            "metrics" => []
                        ];
                        continue;
                    }

                    // Check for TYPE line and add the type to the current category
                    if (isset($line_content[1]) && $line_content[1] === 'TYPE' && $current_category) {
                        $parsed_metrics[$current_category]["metric_type"] = $line_content[3];
                        continue;
                    }

                    if ($current_category) {
                        $metrics_count++;
                        $metric_name = '';
                        if (count($line_content) > 2) {
                            $metric_name = implode(' ', array_slice($line_content, 0, -1));
                        } else {
                            $metric_name = $line_content[0];
                        }
                        $metric_value = is_numeric($line_content[1]) ? $line_content[1] : 0;
                        $parsed_metrics[$current_category]["metrics"][$metric_name] = $metric_value;
                    }
                }
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            $_SESSION['prometheus_wizard_os_type'] = $os_type;
            $_SESSION['prometheus_wizard_hosts'] = $hosts;
            $_SESSION['prometheus_wizard_port'] = $port;
            if ($parsed_metrics != '') $_SESSION['prometheus_wizard_parsed_metrics'] = $parsed_metrics;
            $_SESSION['prometheus_wizard_parsed_metrics_count'] = $metrics_count;

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:
            $os_type = grab_array_var($inargs, "os_type", '');
            $hosts = grab_array_var($inargs, "hosts", '');
            $port = grab_array_var($inargs, "port", 9100);
            $services = grab_array_var($inargs, "services", '');
            $serviceargs = grab_array_var($inargs, "serviceargs", '');
            $custom_metrics = grab_array_var($inargs, "custom_metrics", '');
            $parsed_metrics = grab_array_var($inargs, "parsed_metrics", '');
            $parsed_metrics_count = 0;
            
            if (array_key_exists('prometheus_wizard_os_type', $_SESSION) && $os_type == "") {
                $os_type = $_SESSION['prometheus_wizard_os_type'];
            }
            if (array_key_exists('prometheus_wizard_services', $_SESSION) && $services == "") {
                $services = $_SESSION['prometheus_wizard_services'];
            }
            if (array_key_exists('prometheus_wizard_serviceargs', $_SESSION) && $serviceargs == "") {
                $serviceargs = $_SESSION['prometheus_wizard_serviceargs'];
            }
            if (array_key_exists('prometheus_wizard_custom_metrics', $_SESSION) && $custom_metrics == "") {
                $custom_metrics = $_SESSION['prometheus_wizard_custom_metrics'];
            }
            if (array_key_exists('prometheus_wizard_parsed_metrics', $_SESSION)) {
                $parsed_metrics = $_SESSION['prometheus_wizard_parsed_metrics'];
            }
            if (array_key_exists('prometheus_wizard_parsed_metrics_count', $_SESSION)) {
                $parsed_metrics_count = $_SESSION['prometheus_wizard_parsed_metrics_count'];
            }

            // Function to escape special characters for HTML
            function escapeHtml($string) {
                return htmlspecialchars($string, ENT_QUOTES, 'UTF-8');
            }

            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:
            // Get variables that were passed to us
            $services = grab_array_var($inargs, "services", '');
            $serviceargs = grab_array_var($inargs, "serviceargs", '');
            $custom_metrics = grab_array_var($inargs, "custom_metrics", '');

            if (array_key_exists('prometheus_wizard_services', $_SESSION) && $services == "") {
                $services = $_SESSION['prometheus_wizard_services'];
            }
            if (array_key_exists('prometheus_wizard_serviceargs', $_SESSION) && $serviceargs == "") {
                $serviceargs = $_SESSION['prometheus_wizard_serviceargs'];
            }
            if (array_key_exists('prometheus_wizard_custom_metrics', $_SESSION) && $custom_metrics == "") {
                $custom_metrics = $_SESSION['prometheus_wizard_custom_metrics'];
            }

            // Debug output
            // echo "Services: <pre>" . print_r($services, true) . "</pre><br>\n";
            // echo "Service Args: <pre>" . print_r($serviceargs, true) . "</pre><br>\n";
            // echo "Custom Metrics: <pre>" . print_r($custom_metrics, true) . "</pre><br>\n";

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            $_SESSION['prometheus_wizard_services'] = $services;
            $_SESSION['prometheus_wizard_serviceargs'] = $serviceargs;
            $_SESSION['prometheus_wizard_custom_metrics'] = $custom_metrics;

            break;

        case CONFIGWIZARD_MODE_GETSTAGE3HTML:
            $output = '';
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:
            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:
            $output = '';
            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:
            // Get all input data
            $os_type = $_SESSION['prometheus_wizard_os_type'];
            $hosts = $_SESSION['prometheus_wizard_hosts'];
            $port = $_SESSION['prometheus_wizard_port'];
            $services = $_SESSION['prometheus_wizard_services'];
            $serviceargs = $_SESSION['prometheus_wizard_serviceargs'];
            $custom_metrics = $_SESSION['prometheus_wizard_custom_metrics'];

            // Debug output
            // print "Get Objects - Input data: <pre>" . print_r($inargs, true) . "</pre><br>\n";
            // print "Hosts: " . $hosts . "<br>\n";
            // print "Host Port: " . $port . "<br>\n";
            // print "Services: <pre>" . print_r($services, true) . "</pre><br>\n";
            // print "Service Args: <pre>" . print_r($serviceargs, true) . "</pre><br>\n";
            // print "Custom Metrics: <pre>" . print_r($custom_metrics, true) . "</pre><br>\n";

            $objs = array();

            // If no hosts are selected, break
            if (empty($hosts)) {
                break;
            }

            // Add hosts and their services
            $host_list = explode("\n", trim($hosts));
            foreach ($host_list as $host) {
                $host = trim($host);
                if (empty($host)) {
                    continue;
                }

                // Add host
                $host_name = "Prometheus " . $os_type . " Host " . $host;
                if (!host_exists($host)) {
                    $objs[] = array(
                        "type" => OBJECTTYPE_HOST,
                        "use" => "xiwizard_generic_host",
                        "host_name" => $host_name,
                        "address" => $host,
                        "icon_image" => "prometheus.png",
                        "statusmap_image" => "prometheus.png",
                        "_xiwizard" => $wizard_name,
                    );
                }

                // Build the check commands with all enabled metrics
                foreach ($services as $svc => $svcstate) {
                    if (empty($svcstate) || $svcstate !== "on") {
                        continue;
                    }

                    $check_command = "check_prometheus_" . strtolower($os_type) . "!-P " . $port;
                    $service_description = "";
                    
                    switch ($svc) {
                        case "cpu":
                            $check_command .= "!--cpu!--cpu-warning " . $serviceargs["cpu"]["warning"] . "!--cpu-critical " . $serviceargs["cpu"]["critical"] . " ";
                            $service_description = "CPU Usage";
                            break;
                        case "memory":
                            $check_command .= "!--mem!--mem-warning " . $serviceargs["memory"]["warning"] . "!--mem-critical " . $serviceargs["memory"]["critical"] . " ";
                            $service_description = "Memory Usage";
                            break;

                        case "disk":
                            $check_command .= "!--disk!--disk-warning " . $serviceargs["disk"]["warning"] . "!--disk-critical " . $serviceargs["disk"]["critical"] . " ";
                            $service_description = "Disk Usage";
                            break;
                    }

                    // Add the service check
                    $objs[] = array(
                        "type" => OBJECTTYPE_SERVICE,
                        "host_name" => $host_name,
                        "service_description" => $service_description,
                        "use" => "xiwizard_generic_service",
                        "check_command" => $check_command,
                        "check_interval" => 1,
                        "_xiwizard" => $wizard_name,
                    );
                }

                // Add services for custom metrics
                if (!empty($custom_metrics)) {
                    foreach ($custom_metrics as $metric) {
                        $check_command = "check_prometheus_" . strtolower($os_type) . "!-P " . $port;
                        $check_command .= "!--custom-metric '" . $metric['name'] . "'!--custom-warning '" . $metric['warning'] . "'!--custom-critical '" . $metric['critical'] . "' ";
                        
                        // Add the service check for custom metric
                        $metric_label = str_replace(',', '-', $metric['label']);
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $host_name,
                            "service_description" => $metric_label,
                            "use" => "xiwizard_generic_service", 
                            "check_command" => $check_command,
                            "check_interval" => 1,
                            "_xiwizard" => $wizard_name,
                        );
                    }
                }
            }

            // print "Created objects: <pre>" . print_r($objs, true) . "</pre><br>\n";

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:
            break;
    }

    return $output;
}
