<?php
include_once(dirname(__FILE__).'/../configwizardhelper.inc.php');

java_glassfish_configwizard_init();
java_jboss_configwizard_init();
java_jetty_configwizard_init();
java_tomcat_configwizard_init();
java_weblogic_configwizard_init();

function common_args() {
    return array(
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_VERSION => '2.0.3',
        CONFIGWIZARD_COPYRIGHT => _("Copyright &copy; 2018-2024 Nagios Enterprises, LLC."),
        CONFIGWIZARD_AUTHOR => _('Nagios Enterprises, LLC.'),
        CONFIGWIZARD_FILTER_GROUPS => array(''),
        CONFIGWIZARD_REQUIRES_VERSION => 60100,
    );
}

function java_glassfish_configwizard_init(){

    /* Name / ID for config wizard */
    $name = 'java_glassfish';
    $common_args = common_args();
    /* Relevant info for wizard */
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_DESCRIPTION => _('Monitor GlassFish JVMs through JMX (uses NCPA or requires Java)'),
        CONFIGWIZARD_DISPLAYTITLE => _('GlassFish'),
        CONFIGWIZARD_FUNCTION => 'java_as_configwizard_func',
        CONFIGWIZARD_PREVIEWIMAGE => 'java_glassfish.png',
        );
    $args = array_merge($args, $common_args);
    /* Register wizard with XI */
    register_configwizard($name,$args);
}

function java_jboss_configwizard_init(){

    /* Name / ID for config wizard */
    $name = 'java_jboss';
    $common_args = common_args();
    /* Relevant info for wizard */
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_DESCRIPTION => _('Monitor JBoss/WildFly JVMs through JMX (uses NCPA or requires Java)'),
        CONFIGWIZARD_DISPLAYTITLE => _('JBoss/WildFly'),
        CONFIGWIZARD_FUNCTION => 'java_as_configwizard_func',
        CONFIGWIZARD_PREVIEWIMAGE => 'java_jboss.png',
        );
    $args = array_merge($args, $common_args);
    /* Register wizard with XI */
    register_configwizard($name,$args);
}

function java_jetty_configwizard_init(){

    /* Name / ID for config wizard */
    $name = 'java_jetty';
    $common_args = common_args();
    /* Relevant info for wizard */
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_DESCRIPTION => _('Monitor Jetty JVMs through JMX (uses NCPA or requires Java)'),
        CONFIGWIZARD_DISPLAYTITLE => _('Jetty'),
        CONFIGWIZARD_FUNCTION => 'java_as_configwizard_func',
        CONFIGWIZARD_PREVIEWIMAGE => 'java_jetty.png',
        );
    $args = array_merge($args, $common_args);
    /* Register wizard with XI */
    register_configwizard($name,$args);
}

function java_tomcat_configwizard_init(){

    /* Name / ID for config wizard */
    $name = 'java_tomcat';
    $common_args = common_args();
    /* Relevant info for wizard */
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_DESCRIPTION => _('Monitor Apache Tomcat JVMs through JMX (uses NCPA or requires Java)'),
        CONFIGWIZARD_DISPLAYTITLE => _('Apache Tomcat'),
        CONFIGWIZARD_FUNCTION => 'java_as_configwizard_func',
        CONFIGWIZARD_PREVIEWIMAGE => 'java_tomcat.png',
        );
    $args = array_merge($args, $common_args);
    /* Register wizard with XI */
    register_configwizard($name,$args);
}

/* Not yet */

function java_weblogic_configwizard_init(){

    /* Name / ID for config wizard */
    $name = 'java_weblogic';
    $common_args = common_args();
    /* Relevant info for wizard */
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_DESCRIPTION => _('Monitor a WebLogic instance via JMX (uses NCPA or requires Java).'),
        CONFIGWIZARD_DISPLAYTITLE => _('WebLogic'),
        CONFIGWIZARD_FUNCTION => 'java_weblogic_configwizard_func',
        CONFIGWIZARD_PREVIEWIMAGE => 'java_weblogic.png',
        );
    $args = array_merge($args, $common_args);
    /* Register wizard with XI */
    register_configwizard($name,$args);
}

/* This function is automatically called by the XI wizard framework when the wizard is run */
function java_as_configwizard_func($mode = '', $inargs = null, &$outargs = null, &$result = null) {

    /* Define the wizard name */
    $plugin_name = 'check_jvm.jar';
    $wizard_name = $inargs['wizard'];
    switch ($wizard_name) {
        case 'java_tomcat':
            $noun_long = _("Tomcat Server");
            $noun_short = _('Tomcat');
            $doc_name = 'How-to-Monitor-Apache-Tomcat-With-Nagios-XI.pdf';
            $sample_service_url = 'service:jmx:rmi:///jndi/rmi://&lt;host&gt:&lt;port&gt;/jmxrmi';
            break;

        case 'java_jboss':
            $noun_long = _("JBoss Server");
            $noun_short = _('JBoss');
            $doc_name = 'How-to-Monitor-JBoss-With-Nagios-XI.pdf';
            $sample_service_url = 'service:jmx:remote+http://&lt;host&gt:&lt;port&gt;';
            break;

        case 'java_jetty':
            $noun_long = _("Jetty Server");
            $noun_short = _('Jetty');
            $doc_name = 'How-to-Monitor-Jetty-JMX-With-Nagios-XI.pdf';
            $sample_service_url = 'service:jmx:rmi:///jndi/rmi://&lt;host&gt:&lt;port&gt;/jmxrmi';
            break;

        case 'java_glassfish':
            $noun_long = _("Glassfish Server");
            $noun_short = _('Glassfish');
            $doc_name = 'How-to-Monitor-GlassFish-With-Nagios-XI.pdf';
            $sample_service_url = 'service:jmx:rmi://&lt;host&gt:&lt;port&gt;/jndi/rmi://&lt;host&gt:&lt;port&gt;/jmxrmi';
            break;

        default:
            $noun_long = '';
            $plugin_name = '';
            $doc_name = '';
            break;
    }

    /* Prerequisite software */
    $NCPA_download_url = "https://www.nagios.org/ncpa/#downloads";
    $NCPA_docs_url = "https://www.nagios.org/ncpa/#docs";
    $java_as_plugin_url = get_base_uri() . "includes/configwizards/java-as/plugins/$plugin_name";
    $java_as_docs_url = "https://assets.nagios.com/downloads/nagiosxi/docs/$doc_name";

    /* Initialize return code and output */
    $result = 0;
    $output = '';

    /* Initialize output args */
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:
            /* This defines $nextstep variable, used for determining if the user did not */
            /* provide correct information when the user was on CONFIGWIZARD_MODE_GETSTAGE1HTML */
            $nextstep = htmlentities(grab_array_var($_POST,'nextstep',false),ENT_QUOTES);

            /* Determine if this is the first wizard run */
            if ($nextstep == '') {
                /* Clear the session array to be sure it is blank */
                unset($_SESSION[$wizard_name]);
            }

            /* Fresh Wizard Run */
            /* OR */
            /* Continuing from CONFIGWIZARD_MODE_VALIDATESTAGE1DATA due to user error */

            if (!isset($_POST['address'])) {
                /* Fresh Wizard Run */

                $_SESSION[$wizard_name] = array();

                $remote = '';
                $port = '5693';
                $token = '';
                $service_url = '';
                $ncpa_address = '';
                $username = '';
                $password = '';

                $address = '';

            } else {
                /* General */
                $remote_session = grab_array_var($_SESSION[$wizard_name],'remote',''); // only used in the next line
                $remote = grab_array_var($inargs,'remote', $remote_session);
                $address = grab_array_var($_SESSION[$wizard_name], 'address', '');
                $address = grab_array_var($inargs, 'address', $hostname_session);

                /* NCPA-specific */
                $port_session = grab_array_var($_SESSION[$wizard_name], 'port', '5693');
                $port = grab_array_var($inargs, 'port', $port_session);
                $token_session = grab_array_var($_SESSION[$wizard_name], 'token', '');
                $token = grab_array_var($inargs, 'token', $token_session);

                /* Java-specific */
                $service_url_session = grab_array_var($_SESSION[$wizard_name], 'service_url', '');
                $service_url = grab_array_var($inargs, 'service_url', $service_url_session);
                $username_session = grab_array_var($_SESSION[$wizard_name], 'username', '');
                $username = grab_array_var($inargs, 'username', $username_session);
                $password_session = grab_array_var($_SESSION[$wizard_name], 'password', '');
                $password = grab_array_var($inargs, 'password', $password_session);
            }

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;


        /* Form validation for CONFIGWIZARD_MODE_GETSTAGE1HTML */
        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:
            $back = htmlentities(grab_array_var($_POST,'backButton',false),ENT_QUOTES);

            if ($back) break;

            /* General */
            $remote_session = grab_array_var($_SESSION[$wizard_name],'remote','');
            $remote = grab_array_var($inargs,'remote', $remote_session);
            $address_session = grab_array_var($_SESSION[$wizard_name], 'address', '');
            $address = grab_array_var($inargs, 'address', $address_session);
            $hostname_session = grab_array_var($_SESSION[$wizard_name], 'hostname', gethostbyaddr($address));
            $hostname = grab_array_var($inargs, 'hostname', $hostname_session);

            /* NCPA-specific */
            $port_session = grab_array_var($_SESSION[$wizard_name], 'port', '5693');
            $port = grab_array_var($inargs, 'port', $port_session);
            $port = intval($port);
            $token_session = grab_array_var($_SESSION[$wizard_name], 'token', '');
            $token = grab_array_var($inargs, 'token', $token_session);

            /* Java-specific */
            $service_url_session = grab_array_var($_SESSION[$wizard_name], 'service_url', '');
            $service_url = grab_array_var($inargs, 'service_url', $service_url_session);
            $username_session = grab_array_var($_SESSION[$wizard_name], 'username', '');
            $username = grab_array_var($inargs, 'username', $username_session);
            $password_session = grab_array_var($_SESSION[$wizard_name], 'password', '');
            $password = grab_array_var($inargs, 'password', $password_session);

            $errors = 0;
            $errmsg = array();

            if (have_value($service_url) == false) {
                $errmsg[$errors++] = _('Please specify a JMX service URL');
            }

            if ($remote === "NCPA") {
                if (!have_value($token)) {
                    $errmsg[$errors++] = _("Please specify a token when using NCPA");
                }

                if (!have_value($address)) {
                    $errmsg[$errors++] = _("Please specify the IP address of your Java Application Server");
                }

                // Check that the plugin is installed.
                $ncpa_full_url = "https://" . $address . ":" . $port . "/api/plugins?token=" . $token;
                $cmd = ''; // debug only
                $result = java_as_ncpa_plugin_check($ncpa_full_url, $plugin_name, $cmd); // defined at end of file

                if ($result > 0) {
                    if ($result === 1) {
                        $errmsg[$errors++] = _("cURL call to verify NCPA installation failed.");
                    } else if ($result === 2) {
                        $errmsg[$errors++] = _("cURL call to NCPA port did not return JSON data.");
                    } else if ($result === 3) {
                        $errmsg[$errors++] = _("cURL call to NCPA port returned invalid JSON.");
                    } else if ($result === 4) {
                        $errmsg[$errors++] = sprintf(_("%s is not installed on the remote machine."), $plugin_name);
                    } else {
                        $errmsg[$errors++] = _("cURL-related error: ") . htmlentities($result);
                    }
                }
            }

            if ($remote === "JMX") {
                if (java_as_java_not_installed()) {
                    $errmsg[$errors++] = _("Please install Java to use JMX directly");
                    $errmsg[$errors++] = _("Also make sure to edit the Configuration Manager definition for check_xi_java_as to include java's absolute path.");
                }
            }

            /* Check to see if the $errors array contains errors */
            if ($errors > 0){
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            } else {
                $_SESSION[$wizard_name]['remote'] = $remote;
                $_SESSION[$wizard_name]['address'] = $address;
                $_SESSION[$wizard_name]['port'] = $port;
                $_SESSION[$wizard_name]['token'] = $token;
                $_SESSION[$wizard_name]['service_url'] = $service_url;
                $_SESSION[$wizard_name]['username'] = $username;
                $_SESSION[$wizard_name]['password'] = $password;
            }

            /* The next line ends CONFIGWIZARD_MODE_VALIDATESTAGE1DATA */
            break;


        case CONFIGWIZARD_MODE_GETSTAGE2HTML:
            /* This defines $back variable, used for determining if the Back button */
            /* was clicked when the user was on CONFIGWIZARD_MODE_GETSTAGE3HTML */
            $back = htmlentities(grab_array_var($_POST,'backButton',false),ENT_QUOTES);

            if (isset($_POST['ncpa_address'])) {
                $service_description = sprintf(_("%s JVM Statistics"), $noun_short);

                $combine = 'on';

            } else {
                $sd_session = grab_array_var($_SESSION[$wizard_name], 'service_description', sprintf(_("%s JVM Statistics"), $noun_short));
                $service_description = grab_array_var($inargs, 'service_description', $sd_session);

                $combine = grab_array_var($_SESSION[$wizard_name], 'combine', "on");
                $combine = grab_array_var($inargs, 'combine', 'on');
            }
            // Old variables. These should not change for the rest of the wizard, so it shouldn't matter whether we're going forwards or backwards.

            $address = $_SESSION[$wizard_name]['address'];
            $ncpa_address = (array_key_exists('ncpa_address', $_SESSION[$wizard_name])) ? $_SESSION[$wizard_name]['ncpa_address'] : "";
            $hostname = grab_array_var($_SESSION[$wizard_name], 'hostname', gethostbyaddr($address));
            $remote = $_SESSION[$wizard_name]['remote'];
            $classpath = (array_key_exists('classpath', $_SESSION[$wizard_name])) ? $_SESSION[$wizard_name]['classpath'] : "";

            // New variable. This should also just kinda work.
            $serviceargs = grab_array_var($_SESSION[$wizard_name], 'serviceargs', array());

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        /* Form validation for CONFIGWIZARD_MODE_GETSTAGE2HTML */
        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:
            /* This defines $back variable, used for determining if the Back button */
            /* was clicked when the user was on CONFIGWIZARD_MODE_GETSTAGE4HTML */
            $back = array_key_exists("backButton", $_POST);

            /* If the user came back from CONFIGWIZARD_MODE_GETSTAGE3HTML then we don't need to revalidate and check for errors */
            if ($back) break;

            $hostname_session = grab_array_var($_SESSION, 'hostname', '');
            $hostname = grab_array_var($inargs, 'hostname', $hostname_session);

            $address_session = grab_array_var($_SESSION, 'address', '');
            $address = grab_array_var($inargs, 'address', $address_session);

            $remote = grab_array_var($_SESSION, 'remote', '');

            $serviceargs = grab_array_var($_SESSION[$wizard_name], 'serviceargs', array());
            $serviceargs = grab_array_var($inargs, 'serviceargs', $serviceargs);

            $service_description = grab_array_var($_SESSION[$wizard_name], 'service_description', '');
            $service_description = grab_array_var($inargs, 'service_description', $service_description);

            $classpath = grab_array_var($_SESSION[$wizard_name], 'classpath', '.');
            $classpath = grab_array_var($inargs, 'classpath', $classpath);

            $combine = grab_array_var($_SESSION[$wizard_name], 'combine', 'off');
            $combine = grab_array_var($inargs, 'combine', $combine);

            $_SESSION[$wizard_name]['classpath'] = $classpath;

            $valid_types = array(
                "MemorySimpleHeap",
                "MemoryEden",
                "MemorySurvivor",
                "MemoryOld",
                "MemorySimpleNonHeap",
                "MemoryCodeCache",
                "MemoryCompressedClass",
                "MemoryMetaspace",
                "requestproc",
                "ProcessCPUUsage",
                "SystemCPUUsage",
                "Uptime",
                "ClassCount",
                "ThreadCount",
            );

            foreach($serviceargs as $type => $values) {
                if (!in_array($type, $valid_types)) {
                    $errmsg[$errors++] = _("Check type ") . encode_form_val($type) . _(" is invalid.");
                }
            }

            if ($wizard_name === 'java_tomcat') {
                $valid_types = array(
                    "RequestsPerMinute",
                    "BytesPerMinute",
                    "BytesPerRequest",
                    "ErrorsPerMinute",
                    "ProcessingTimePerRequest",
                );

                foreach ($serviceargs['requestproc'] as $index => $values) {
                    if (!in_array($values['type'], $valid_types)) {
                        $errmsg[$errors++] = _("Check type ") . encode_form_val($values['type']) . _(" is invalid.");
                    }
                }
            }

            if (!have_value($combine))
                $combine = "off";

            if (!have_value($hostname)) {
                $errmsg[$errors++] = _("No host name specified.");
            } elseif (!is_valid_host_name($hostname)) {
                $errmsg[$errors++] = _("Invalid host name.");
            }

            $_SESSION[$wizard_name]['hostname'] = $hostname;

            if (!have_value($address)) {
                $errmsg[$errors++] = _("No IP Address specified");
            }

            $_SESSION[$wizard_name]['address'] = $address;

            if (!have_value($service_description)) {
                $errmsg[$errors++] = _("No service description specified.");
            }

            $_SESSION[$wizard_name]['service_description'] = $service_description;

            if (empty($serviceargs)) {
                $errmsg[$errors++] = _("serviceargs is empty. This error should never be reached. Please contact the config wizard maintainer.");
            }

            if ($errors>0){
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;

                if (!array_key_exists('serviceargs', $_SESSION[$wizard_name])) {
                    // If we've never assigned serviceargs to SESSION, it's fine to overwrite even if they're not valid. Same for combine.
                    $_SESSION[$wizard_name]['serviceargs'] = $serviceargs;
                    $_SESSION[$wizard_name]['combine'] = $combine;
                }
            } else {
                $_SESSION[$wizard_name]['serviceargs'] = $serviceargs;
                $_SESSION[$wizard_name]['combine'] = $combine;
            }

            /* The next line ends CONFIGWIZARD_MODE_VALIDATESTAGE2DATA */
            break;


        case CONFIGWIZARD_MODE_GETSTAGE3HTML:
            $back = array_key_exists('backButton', $_POST);
            $output = '';
            break;


        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            $back = array_key_exists("backButton", $_POST);
            if ($back) {
                break;
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:

            $output = '';
            break;


        case CONFIGWIZARD_MODE_GETOBJECTS:
            $remote = $_SESSION[$wizard_name]['remote'];
            $address = $_SESSION[$wizard_name]['address'];
            $hostname = $_SESSION[$wizard_name]['hostname'];
            $service_url = $_SESSION[$wizard_name]['service_url'];
            $username = $_SESSION[$wizard_name]['username'];
            $password = $_SESSION[$wizard_name]['password'];
            $service_description = $_SESSION[$wizard_name]['service_description'];
            $classpath = $_SESSION[$wizard_name]['classpath'];
            $combine = $_SESSION[$wizard_name]['combine'];

            if ($remote === "NCPA") {
                $token = $_SESSION[$wizard_name]['token'];
                $port = $_SESSION[$wizard_name]['port'];

                // Escape values for check_command line
                if (function_exists('nagiosccm_replace_command_line')) {
                    $token = nagiosccm_replace_command_line($token, '', '!');
                }
            }

            $serviceargs = $_SESSION[$wizard_name]['serviceargs'];

            $objs = array();

            if (!host_exists($hostname)) {

                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => 'xiwizard_java_as_host',
                    "host_name" => $hostname,
                    'icon_image' =>  $wizard_name . '.png',
                    "address" => $address,
                    "_xiwizard" => $wizard_name,
                );
            }

            $service_url_switch = "-s " . $service_url;

            if (have_value($username)) {
                $username_switch = "-u '$username'";
            }

            if (have_value($password)) {
                $password_switch = "-p '$password'";
            }

            $server_type_switch = "-t " . substr($wizard_name, 5); // i.e. 'java_tomcat' --> '-t tomcat'

            $first_one = true;

            $check_name = "";
            $warning = "";
            $critical = "";
            if ($combine === "off") {
                $check_name = array();
                $warning = array();
                $critical = array();
            }

            // Plugin wants bytes
            $heap_allocated = array('MemorySimpleHeap', 'MemoryEden', 'MemorySurvivor', 'MemoryOld');
            if (is_array($serviceargs['MemorySimpleHeap']) && isset($serviceargs['MemorySimpleHeap']['unit'])) {
                foreach ($heap_allocated as $statistic) {
                    $serviceargs[$statistic]['warning'] = java_as_scale_threshold($serviceargs[$statistic]['warning'], $serviceargs['MemorySimpleHeap']['unit']);
                    $serviceargs[$statistic]['critical'] = java_as_scale_threshold($serviceargs[$statistic]['critical'], $serviceargs['MemorySimpleHeap']['unit']);
                }
            }

            $non_heap_allocated = array('MemorySimpleNonHeap', 'MemoryCompressedClass', 'MemoryCodeCache', 'MemoryMetaspace');
            if (is_array($serviceargs['MemorySimpleNonHeap']) && isset($serviceargs['MemorySimpleNonHeap']['unit'])) {
                foreach ($non_heap_allocated as $statistic) {
                    $serviceargs[$statistic]['warning'] = java_as_scale_threshold($serviceargs[$statistic]['warning'], $serviceargs['MemorySimpleNonHeap']['unit']);
                    $serviceargs[$statistic]['critical'] = java_as_scale_threshold($serviceargs[$statistic]['critical'], $serviceargs['MemorySimpleNonHeap']['unit']);
                }
            }

            // Plugin wants ms
            if (is_array($serviceargs['Uptime'])) {
                $serviceargs['Uptime']['warning'] = java_as_scale_threshold($serviceargs['Uptime']['warning'], 1000);
                $serviceargs['Uptime']['critical'] = java_as_scale_threshold($serviceargs['Uptime']['critical'], 1000);
            }

            foreach ($serviceargs as $type => $value) {
                if ($type !== "requestproc" && $value['enabled'] === "on") {
                    if ($combine === "on") {
                        if ($first_one) {
                            $check_name .= $type;
                            $warning .= $value['warning'];
                            $critical .= $value['critical'];
                            $first_one = !$first_one;
                        }
                        else {
                            $check_name .= ",$type";
                            $warning .= "," . $value['warning'];
                            $critical .= "," . $value['critical'];
                        }
                    }
                    else {
                        $check_name[] = $type;
                        $warning[] = $value['warning'];
                        $critical[] = $value['critical'];
                    }
                }
                else if ($wizard_name === 'java_tomcat' && $type === "requestproc") {

                    // rp_filter just checks that all indices have values.
                    $serviceargs[$type] = array_filter($value, 'java_as_rp_filter', ARRAY_FILTER_USE_BOTH);

                    if (have_value($serviceargs[$type])) {
                        if ($combine === 'on') {
                            if ($first_one) {
                                $check_name .= implode(",", $serviceargs[$type]['on']);
                                $warning .= implode(",", $serviceargs[$type]['warning']);
                                $critical .= implode(",", $serviceargs[$type]['critical']);
                                $first_one = !$first_one;
                            }
                            else {
                                $check_name .= "," . implode(",", $serviceargs[$type]['on']);
                                $warning .= "," . implode(",", $serviceargs[$type]['warning']);
                                $critical .= "," . implode(",", $serviceargs[$type]['critical']);
                            }
                        }
                        else {
                            $check_name = array_merge($check_name, $serviceargs[$type]['on']);
                            $warning = array_merge($warning, $serviceargs[$type]['warning']);
                            $critical = array_merge($critical, $serviceargs[$type]['critical']);
                        }
                    }
                }
            }

            if ($combine === 'on') {
                $check_name = array($check_name);
                $warning = array($warning);
                $critical = array($critical);
            }
            foreach ($check_name as $index => $not_used) {
                $plugin_opts = array($service_url_switch, $username_switch, $password_switch, $server_type_switch, "-C '" . $check_name[$index] . "'", "-w '" . $warning[$index] . "'", "-c '" . $critical[$index] . "'");
                if ($remote === "NCPA") {
                    foreach ($plugin_opts as $key => $value) {
                        $tmp = trim($value);
                        if (!empty($tmp)) {
                            $plugin_opts[$key] = "args=" . $value;
                        }
                        else {
                            unset($plugin_opts[$key]);
                        }
                    }
                    $plugin_opts = implode(",", $plugin_opts);
                    $all_options = "-t " . escapeshellarg_nowrap($token) . " -P ". $port . " -M plugins/$plugin_name -q \"";
                    $all_options .= $plugin_opts . "\"";
                    $all_options = java_as_escape_exclamation_dollar($all_options);
                    $full_check_command = "check_xi_ncpa!$all_options";
                }
                else {
                    $plugin_opts = implode(" ", $plugin_opts);
                    $plugin_opts = java_as_escape_exclamation_dollar($plugin_opts);
                    $classpath = java_as_escape_exclamation_dollar($classpath);
                    $full_check_command = "check_xi_java_as!$plugin_opts!$classpath!";
                }

                $service_description_full = $service_description;
                if ($combine === "off") {
                    $service_description_full .= " " . $check_name[$index];
                }

                $objs[] = array(
                    'type'                  => OBJECTTYPE_SERVICE,
                    'host_name'             => $hostname,
                    'service_description'   => $service_description_full,
                    'icon_image'            => $wizard_name . '.png',
                    'check_command'         => $full_check_command,
                    '_xiwizard'             => $wizard_name,
                );
            }

            /* Return the object definitions to the wizard */
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            /* clear the session variables for this wizard run */
            unset($_SESSION[$wizard_name]);

            /* The next line ends CONFIGWIZARD_MODE_GETOBJECTS */
            break;

        default:
            break;
    };
    return $output;
}

function java_weblogic_configwizard_func($mode = '', $inargs = null, &$outargs = null, &$result = null) {

    /* Define the wizard name */
    $wizard_name = 'java_weblogic';

    /* Prerequisite software */
    $NCPA_download_url = "https://www.nagios.org/ncpa/#downloads";
    $NCPA_docs_url = "https://www.nagios.org/ncpa/#docs";
    $wlsagent_package = "/nagiosxi/includes/configwizards/java-as/plugins/wlsagent.tar.gz";
    $java_weblogic_plugin_url = "/nagiosxi/includes/configwizards/java-as/plugins/check_wlsagent.sh";
    $java_weblogic_docs_url = "https://assets.nagios.com/downloads/nagiosxi/docs/How-to-Monitor-WebLogic-With-Nagios-XI.pdf";
    /* rip in peace my sweet nrpe */
    // $agent_url = "https://assets.nagios.com/downloads/nagiosxi/agents/linux-nrpe-agent.tar.gz";
    // $agent_doc_url = "https://assets.nagios.com/downloads/nagiosxi/docs/Installing_The_XI_Linux_Agent.pdf";

    /* Initialize return code and output */
    $result = 0;
    $output = '';

    /* Initialize output args */
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:
            /* This defines $nextstep variable, used for determining if the user did not */
            /* provide correct information when the user was on CONFIGWIZARD_MODE_GETSTAGE1HTML */
            $nextstep = htmlentities(grab_array_var($_POST,'nextstep',false),ENT_QUOTES);

            /* Determine if this is the first wizard run */
            if ($nextstep == '') {
                /* Clear the session array to be sure it is blank */
                unset($_SESSION[$wizard_name]);
            }

            /* Fresh Wizard Run */
            /* OR */
            /* Continuing from CONFIGWIZARD_MODE_VALIDATESTAGE1DATA due to user error */

            if (!isset($_POST['address'])) {
                /* Fresh Wizard Run */

                $_SESSION[$wizard_name] = array();

                $remote = '';
                $hostname = '';
                $ip_address = '';
                $WLSAgent_port = '9090';
                $JMX_T3_port = '7001';
                $username = '';
                $password = '';

                $port = '';
                $token = '';

            } else {
                /* General */
                $remote_session = grab_array_var($_SESSION[$wizard_name], 'remote', '');
                $remote = grab_array_var($inargs, 'remote', $remote_session);
                $hostname_session = grab_array_var($_SESSION[$wizard_name], 'hostname', '');
                $hostname = grab_array_var($inargs, 'hostname', $hostname_session);
                $ip_address_session = grab_array_var($_SESSION[$wizard_name], 'ip_address', '');
                $ip_address = grab_array_var($inargs, 'ip_address', $hostname_session);
                $WLSAgent_port_session = grab_array_var($_SESSION[$wizard_name], 'WLSAgent_port', '9090');
                $WLSAgent_port = grab_array_var($inargs, 'WLSAgent_port', $port_session);
                $JMX_T3_port_session = grab_array_var($_SESSION[$wizard_name], 'JMX_T3_port', '7001');
                $JMX_T3_port = grab_array_var($inargs, 'JMX_T3_port', $port_session);

                /* Credentials */
                $username_session = grab_array_var($_SESSION[$wizard_name], 'username', '');
                $username = grab_array_var($inargs, 'username', $username_session);
                $password_session = grab_array_var($_SESSION[$wizard_name], 'password', '');
                $password = grab_array_var($inargs, 'password', $password_session);

                $port_session = grab_array_var($_SESSION[$wizard_name], 'port', '');
                $port = grab_array_var($inargs, 'port', $port_session);
                $token_session = grab_array_var($_SESSION[$wizard_name], 'token', '');
                $token = grab_array_var($inargs, 'token', $token_session);
            }

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
//            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.weblogic.php';
            $output = ob_get_clean();

            break;


        /* Form validation for CONFIGWIZARD_MODE_GETSTAGE1HTML */
        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:
            $back = htmlentities(grab_array_var($_POST,'backButton',false),ENT_QUOTES);

            if ($back) break;

            /* General */
            $remote_session = grab_array_var($_SESSION[$wizard_name], 'remote', '');
            $remote = grab_array_var($inargs, 'remote', $remote_session);
            $hostname_session = grab_array_var($_SESSION[$wizard_name], 'hostname', '');
            $hostname = grab_array_var($inargs, 'hostname', $hostname_session);
            $ip_address_session = grab_array_var($_SESSION[$wizard_name], 'ip_address', '');
            $ip_address = grab_array_var($inargs, 'ip_address', $ip_address_session);

            $WLSAgent_port_session = grab_array_var($_SESSION[$wizard_name], 'WLSAgent_port', '5693');
            $WLSAgent_port = grab_array_var($inargs, 'WLSAgent_port', $WLSAgent_port_session);
            $WLSAgent_port = intval($WLSAgent_port);
            $JMX_T3_port_session = grab_array_var($_SESSION[$wizard_name], 'JMX_T3_port', '5693');
            $JMX_T3_port = grab_array_var($inargs, 'JMX_T3_port', $JMX_T3_port_session);
            $JMX_T3_port = intval($JMX_T3_port);

            $username_session = grab_array_var($_SESSION[$wizard_name], 'username', '');
            $username = grab_array_var($inargs, 'username', $username_session);
            $password_session = grab_array_var($_SESSION[$wizard_name], 'password', '');
            $password = grab_array_var($inargs, 'password', $password_session);

            $port_session = grab_array_var($_SESSION[$wizard_name], 'port', '');
            $port = grab_array_var($inargs, 'port', $port_session);
            $token_session = grab_array_var($_SESSION[$wizard_name], 'token', '');
            $token = grab_array_var($inargs, 'token', $token_session);

            $errors = 0;
            $errmsg = array();

            if (!have_value($hostname))
                $errmsg[$errors++] = _('Please specify a nagios host name.');
            if (!have_value($ip_address))
                $errmsg[$errors++] = _('Please specify a DNS Hostname/IP Address');

            if (!have_value($WLSAgent_port)) // Shouldn't be possible to trigger this.
                $errmsg[$errors++] = _('Please specify the port of your WLS Agent.');
            if (!have_value($JMX_T3_port)) // Or this.
                $errmsg[$errors++] = _('Please specify the JMX T3 port of your WebLogic server');

            if ($remote === "NCPA" && !have_value($port))
                $errmsg[$errors++] = _('Please specify the NCPA Listener port');
            if ($remote === "NCPA" && !have_value($token))
                $errmsg[$errors++] = _('Please specify the NCPA API token.');

            /* TODO:
             * Include a thing to test the JMX connection?
             * Should be doable since the check is now just doing wget. */

            /* Check to see if the $errors array contains errors */
            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            } else {
                $_SESSION[$wizard_name]['remote'] = $remote;
                $_SESSION[$wizard_name]['hostname'] = $hostname;
                $_SESSION[$wizard_name]['ip_address'] = $ip_address;
                $_SESSION[$wizard_name]['WLSAgent_port'] = $WLSAgent_port;
                $_SESSION[$wizard_name]['JMX_T3_port'] = $JMX_T3_port;
                $_SESSION[$wizard_name]['username'] = $username;
                $_SESSION[$wizard_name]['password'] = $password;
                $_SESSION[$wizard_name]['port'] = $port;
                $_SESSION[$wizard_name]['token'] = $token;

            }

            /* The next line ends CONFIGWIZARD_MODE_VALIDATESTAGE1DATA */
            break;


        case CONFIGWIZARD_MODE_GETSTAGE2HTML:
            /* This defines $back variable, used for determining if the Back button */
            /* was clicked when the user was on CONFIGWIZARD_MODE_GETSTAGE3HTML */
            $back = htmlentities(grab_array_var($_POST,'backButton',false),ENT_QUOTES);

            $service_description = grab_array_var($_SESSION[$wizard_name], 'service_description', _("WebLogic Statistics"));
            $service_description = grab_array_var($inargs, 'service_description', $service_description);

            $combine = grab_array_var($_SESSION[$wizard_name], 'combine', 'on');
            $combine = grab_array_var($inargs, 'combine', $combine);

            $serviceargs = grab_array_var($_SESSION[$wizard_name], 'serviceargs', array());

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
//            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.weblogic.php';
            $output = ob_get_clean();

            break;


        /* Form validation for CONFIGWIZARD_MODE_GETSTAGE2HTML */
        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:
            /* This defines $back variable, used for determining if the Back button */
            /* was clicked when the user was on CONFIGWIZARD_MODE_GETSTAGE4HTML */
            $back = array_key_exists("backButton", $_POST);
            /* If the user came back from CONFIGWIZARD_MODE_GETSTAGE3HTML then we don't need to revalidate and check for errors */
            if ($back) break;

            $serviceargs = grab_array_var($_SESSION[$wizard_name], 'serviceargs', array());
            $serviceargs = grab_array_var($inargs, 'serviceargs', $serviceargs);

            $service_description = grab_array_var($_SESSION[$wizard_name], 'service_description', '');
            $service_description = grab_array_var($inargs, 'service_description', $service_description);

            $combine = grab_array_var($_SESSION[$wizard_name], 'combine', 'off');
            $combine = grab_array_var($inargs, 'combine', $combine);

            $simple_types = array(
                "jvm",
                "jta",
                "jms_runtime",
                "thread_pool",
            );

            $complex_types = array(
                "jdbc",
                "component",
                "jms_queue",
            );

            foreach($serviceargs as $type => $values) {
                if (!in_array($type, $simple_types)) {
                    if (!in_array($type, $complex_types)) {
                        $errmsg[$errors++] = _("Check type ") . encode_form_val($type) . _(" is invalid.");
                    } else {
                        foreach($serviceargs[$type] as $index => $subvalues) {
                            $serviceargs[$type][$index]['on'] = str_replace("'", " ", $serviceargs[$type][$index]['on']);
                            // Thresholds aren't handled by the plugin, so we can convert to integer.
                            $serviceargs[$type][$index]['warning'] = intval($subvalues['warning']);
                            $serviceargs[$type][$index]['critical'] = intval($subvalues['critical']);
                        }
                    }
                } else {
                    $serviceargs[$type]['warning'] = intval($values['warning']);
                    $serviceargs[$type]['critical'] = intval($values['critical']);
                }
            }


            if (!have_value($combine)) {
                $combine = "off";
            }

            if (!have_value($service_description)) {
                $errmsg[$errors++] = _("No service description specified.");
            }

            $_SESSION[$wizard_name]['service_description'] = $service_description;

            if (empty($serviceargs)) {
                $errmsg[$errors++] = _("serviceargs is empty. This error should never be reached. Please contact the config wizard maintainer.");
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;

                if (!array_key_exists('serviceargs', $_SESSION[$wizard_name])) {
                    // If we've never assigned serviceargs to SESSION, it's fine to overwrite even if they're not valid. Same for combine.
                    $_SESSION[$wizard_name]['serviceargs'] = $serviceargs;
                    $_SESSION[$wizard_name]['combine'] = $combine;
                }
            } else {
                $_SESSION[$wizard_name]['serviceargs'] = $serviceargs;
                $_SESSION[$wizard_name]['combine'] = $combine;
            }

            /* The next line ends CONFIGWIZARD_MODE_VALIDATESTAGE2DATA */
            break;


        case CONFIGWIZARD_MODE_GETSTAGE3HTML:
            $back = array_key_exists('backButton', $_POST);

            $output = '
            ';
            /* The next line ends CONFIGWIZARD_MODE_GETSTAGE3HTML */
            break;


        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            $back = array_key_exists("backButton", $_POST);
            if ($back) {
                break;
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;


        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:
            $output = '';
            break;


        case CONFIGWIZARD_MODE_GETOBJECTS:
            $remote = $_SESSION[$wizard_name]['remote'];
            $ip_address = $_SESSION[$wizard_name]['ip_address'];
            $hostname = $_SESSION[$wizard_name]['hostname'];
            $username = $_SESSION[$wizard_name]['username'];
            $password = $_SESSION[$wizard_name]['password'];
            $service_description = $_SESSION[$wizard_name]['service_description'];
            $combine = $_SESSION[$wizard_name]['combine'];
            $WLSAgent_port = $_SESSION[$wizard_name]['WLSAgent_port'];
            $JMX_T3_port = $_SESSION[$wizard_name]['JMX_T3_port'];
            $port = $_SESSION[$wizard_name]['port'];
            $token = $_SESSION[$wizard_name]['token'];

            $serviceargs = $_SESSION[$wizard_name]['serviceargs'];

            $objs = array();
            if (!host_exists($hostname)) {

                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_java_as_host",
                    "host_name" => $hostname,
                    'icon_image' => 'java_weblogic.png',
                    "address" => $ip_address,
                    "_xiwizard" => $wizard_name,
                );
            }

            $simple_types = array(
                "jvm",
                "jta",
                "jms_runtime",
                "thread_pool",
            );

            $complex_types = array(
                "jdbc",
                "component",
                "jms_queue",
            );

            $type_argname_map = array(
                "jvm" => "jvm=UsedMemory",
                "jta" => "jta=ActiveTransactions",
                "jms_runtime" => "jms-runtime=CurrentConnections",
                "thread_pool" => "thread-pool=ThreadStuckCount",
                "jdbc" => "jdbc=",
                "component" => "component=",
                "jms_queue" => "jms-queue=",
            );

            $args_array = array(); // Associative if combine is off, else indexed.

            $standard_options = array(
                "username=" . urlencode($username),
                "password=" . urlencode($password),
                "hostname=localhost",
                "port=" . $JMX_T3_port, // already an integer

            );

            if ($combine === "on") {
                $args_array = $standard_options;
            }

            foreach ($serviceargs as $type => $value) {
                if (in_array($type, $simple_types) && $value['enabled'] === "on") {
                    if ($combine === "on") {
                        $args_array[] = implode(",", array($type_argname_map[$type], $value['warning'], $value['critical']));
                    }
                    else {
                        $args_array[$type] = $standard_options;
                        $args_array[$type][] = implode(",", array($type_argname_map[$type], $value['warning'], $value['critical']));
                    }
                }
                else if (in_array($type, $complex_types)) {
                    foreach($serviceargs[$type] as $subvalues) {
                        if ($combine === "on") {
                            $args_array[] = implode(",", array($type_argname_map[$type] . $subvalues['on'], $subvalues['warning'], $subvalues['critical']));
                        }
                        else {
                            $args_array[$type] = $standard_options;
                            $args_array[$type][] = implode(",", array($type_argname_map[$type] . $subvalues['on'], $subvalues['warning'], $subvalues['critical']));
                        }
                    }
                }
            }

            if ($combine === "on") {
                $args = implode("&", $args_array);
                $args = str_replace("'", "", $args);
                if ($remote === "wget") {
                    $args = "$ip_address $WLSAgent_port '$args'";
                    $full_check_command = "check_xi_java_weblogic!" . $args . "!";
                }
                else {
                    $full_check_command = "check_xi_ncpa!-t " . escapeshellarg_nowrap($token) . ' -P ' . $port . ' -M plugins/check_wlsagent.sh -q \'';
                    $full_check_command .= "args=$ip_address,args=$WLSAgent_port,args=\"$args\"'";
                }

                $objs[] = array(
                    'type'                  => OBJECTTYPE_SERVICE,
                    'host_name'             => $hostname,
                    'service_description'   => $service_description,
                    'icon_image'            => 'java_weblogic.png',
                    'check_command'         => $full_check_command,
                    '_xiwizard'             => $wizard_name,
                );
            }
            else {
                foreach ($args_array as $type => $subarray) {
                    $args = implode("&", $subarray);
                    $args = str_replace("'", "", $args);
                    if ($remote === "wget") {
                        $args = "$ip_address $WLSAgent_port '$args'";
                        $full_check_command = "check_xi_java_weblogic!" . $args . "!";
                    }
                    else {
                        $full_check_command = "check_xi_ncpa!-t " . escapeshellarg_nowrap($token) . ' -P ' . $port . ' -M plugins/check_wlsagent.sh -q "';
                        $full_check_command .= "args=$ip_address,args=$WLSAgent_port,args='$args'\"";
                    }

                    $objs[] = array(
                        'type'                  => OBJECTTYPE_SERVICE,
                        'host_name'             => $hostname,
                        'service_description'   => $service_description . ": " . $type,
                        'icon_image'            => 'java_weblogic.png',
                        'check_command'         => $full_check_command,
                        '_xiwizard'             => $wizard_name,
                    );
                }
            }

            /* Return the object definitions to the wizard */
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            /* clear the session variables for this wizard run */
            unset($_SESSION[$wizard_name]);

            /* The next line ends CONFIGWIZARD_MODE_GETOBJECTS */
            break;

        default:
            break;
    };
    return $output;
}


function java_as_ncpa_plugin_check($url, $plugin_name, &$cmd) {
    $url = escapeshellarg($url);
    $cmd = "curl " . $url . " -g -f -k --connect-timeout 10";

    $ret = 0;
    exec($cmd, $data, $ret);
    $data = implode("", $data);

    if ($ret)
        return 1; // cURL-specific error

    $data = json_decode($data, true);

    if (!$data) {
        return 2; // Not JSON data
    }

    if (!array_key_exists('plugins', $data)) {
        return 3; // Not the correct json data
    }

    $missing = 4; // If not changed, plugin not installed.
    foreach($data['plugins'] as $plugin) {
        if ($plugin === $plugin_name) {
            $missing = 0; // Everything is good.
            break;
        }
    }

    return $missing;
}

function java_as_java_not_installed() {
    $cmd = "which java";
    $output = "";
    $ret = 0;
    exec($cmd, $output, $ret);
    return $ret;
}

function java_as_scale_threshold($threshold, $scalar) {
    if (!is_string($threshold))
        return $threshold;

    $flipped = false;
    if ($threshold[0] === '@') {
        $flipped = true;
        $threshold = trim($threshold, '@');
    }

    $threshold = explode(":", $threshold);
    foreach ($threshold as $index => $bound) {
        $threshold[$index] = floatval($scalar) * floatval($bound);
        if ($threshold[$index] == 0) {
            $threshold[$index] = "";
        }
    }
    $threshold = implode(":", $threshold);

    if ($flipped)
        $threshold = '@' . $threshold;
    return $threshold;
}

function java_as_rp_filter($in) {
    if (!is_array($in))
        return $in;
    return have_value($in['on']) && have_value($in['type']) && have_value($in['warning']) && have_value($in['critical']);
}

function java_as_escape_exclamation_dollar($input) {
    $search = array('!', '$');
    $replace = array('\!', '\$');
    return str_replace($search, $replace, $input);
}
?>
