<?php
//
// Global Event Handler Component
// Copyright (c) 2010-2024 Nagios Enterprises, LLC. All rights reserved.
//

require_once(dirname(__FILE__) . '/../componenthelper.inc.php');


$globaleventhandler_component_name = "globaleventhandler";
globaleventhandler_component_init();


////////////////////////////////////////////////////////////////////////
// COMPONENT INIT FUNCTIONS
////////////////////////////////////////////////////////////////////////


function globaleventhandler_component_init()
{
    global $globaleventhandler_component_name;

    $args = array(
        COMPONENT_NAME           => $globaleventhandler_component_name,
        COMPONENT_VERSION        => '1.3.1',
        COMPONENT_AUTHOR         => "Nagios Enterprises, LLC",
        COMPONENT_DESCRIPTION    => _("Provides the ability to execute external scripts on host and service notifications and state changes."),
        COMPONENT_TITLE          => _("Global Event Handlers"),
        COMPONENT_CONFIGFUNCTION => "globaleventhandler_component_config_func"
    );

    register_component($globaleventhandler_component_name, $args);
}


///////////////////////////////////////////////////////////////////////////////////////////
//CONFIG FUNCTIONS
///////////////////////////////////////////////////////////////////////////////////////////


function globaleventhandler_component_config_func($mode, $inargs, &$outargs, &$result)
{
    global $globaleventhandler_component_name;

    $result = 0;
    $output = "";

    switch ($mode) {


    case COMPONENT_CONFIGMODE_GETSETTINGSHTML:

        $settings_raw = get_option("globaleventhandler_component_options");
        $settings     = "";

        if (!empty($settings_raw)) {
            $settings = unserialize($settings_raw);
        }

        if (empty($settings)) {
            $settings = array();
        }
        
        // Initial values
        $host_event_handler_commands           = array();
        $service_event_handler_commands        = array();
        $host_notification_handler_commands    = array();
        $service_notification_handler_commands = array();

        for ($x = 0; $x <= 2; $x++) {

            $host_event_handler_commands[$x] = array(
                "enabled"  => "",
                "command"  => "",
                "downtime" => 0,
            );

            $service_event_handler_commands[$x] = array(
                "enabled"  => "",
                "command"  => "",
                "downtime" => 0,
            );

            $host_notification_handler_commands[$x] = array(
                "enabled" => "",
                "command" => "",
            );

            $service_notification_handler_commands[$x] = array(
                "enabled" => "",
                "command" => "",
            );
        }

        // Sample definitions
        $host_event_handler_commands[0]["command"]           = '/tmp/host_change_handler.sh "%host%" %hoststate% %hoststateid% %lasthoststate% %lasthoststateid% %hoststatetype% %currentattempt% %maxattempts% %hosteventid% %hostproblemid% "%hostoutput%" %hostdowntime%';

        $service_event_handler_commands[0]["command"]        = '/tmp/service_change_handler.sh "%host%" "%service%" %hoststate% %servicestate% %servicestateid% %lastservicestate%  %lastservicestateid% %servicestatetype% %currentattempt% %maxattempts% %serviceeventid% %serviceproblemid% "%serviceoutput%" %servicedowntime%';

        $host_notification_handler_commands[0]["command"]    = '/tmp/host_notification_handler.sh "%contact%" "%type%" "%author%" "%comments%" "%host%" "%hostaddress%" %hoststate% %hoststateid% %lasthoststate% %lasthoststateid% %hoststatetype% %currentattempt% %maxattempts% %hosteventid% %hostproblemid% "%hostoutput%"';

        $service_notification_handler_commands[0]["command"] = '/tmp/service_notification_handler.sh "%contact%" "%type%" "%author%" "%comments%" "%host%" "%hostaddress%" %hoststate% %hoststateid% %servicestate% %servicestateid% %lastservicestate% %lastservicestateid% %servicestatetype% %currentattempt% %maxattempts% %serviceeventid% %serviceproblemid% "%serviceoutput%"';

        // Saved values
        $host_event_handler_commands           = grab_array_var($settings, "host_event_handler_commands", $host_event_handler_commands);
        $service_event_handler_commands        = grab_array_var($settings, "service_event_handler_commands", $service_event_handler_commands);
        $host_notification_handler_commands    = grab_array_var($settings, "host_notification_handler_commands", $host_notification_handler_commands);
        $service_notification_handler_commands = grab_array_var($settings, "service_notification_handler_commands", $service_notification_handler_commands);

        // Values passed to us
        $host_event_handler_commands           = grab_array_var($inargs, "host_event_handler_commands", $host_event_handler_commands);
        $service_event_handler_commands        = grab_array_var($inargs, "service_event_handler_commands", $service_event_handler_commands);
        $host_notification_handler_commands    = grab_array_var($inargs, "host_notification_handler_commands", $host_notification_handler_commands);
        $service_notification_handler_commands = grab_array_var($inargs, "service_notification_handler_commands", $service_notification_handler_commands);

        // Fix missing values
        for ($x = 0; $x <= 2; $x++) {

            if (!array_key_exists("enabled", $host_event_handler_commands[$x])) {
                $host_event_handler_commands[$x]["enabled"] = "";
            }
            if (!array_key_exists("enabled", $service_event_handler_commands[$x])) {
                $service_event_handler_commands[$x]["enabled"] = "";
            }
            if (!array_key_exists("enabled", $host_notification_handler_commands[$x])) {
                $host_notification_handler_commands[$x]["enabled"] = "";
            }
            if (!array_key_exists("enabled", $service_notification_handler_commands[$x])) {
                $service_notification_handler_commands[$x]["enabled"] = "";
            }

            if (!array_key_exists("downtime", $host_event_handler_commands[$x])) {
                $host_event_handler_commands[$x]["downtime"] = "";
            }
            if (!array_key_exists("downtime", $service_event_handler_commands[$x])) {
                $service_event_handler_commands[$x]["downtime"] = "";
            }
            if (!array_key_exists("downtime", $host_notification_handler_commands[$x])) {
                $host_notification_handler_commands[$x]["downtime"] = "";
            }
            if (!array_key_exists("downtime", $service_notification_handler_commands[$x])) {
                $service_notification_handler_commands[$x]["downtime"] = "";
            }
        }

        $component_url = get_component_url_base($globaleventhandler_component_name);

        $output = "";

        $eventhandlersok = globaleventhandler_component_checkeventhandlers();
        if (!$eventhandlersok) {
            $output .= 
                "<font color='red'><b>" . _("WARNING") . ":</b> " . _("Event handlers are currently disabled.  This will prevent event handler commands from working!") . "</font>";
        }

        $output .= '

<p>' . _('Define commands to be locally executed on this Nagios XI server when host and service state changes or notifications occur.  Recommended only for advanced users.') . '</p>

<script type="text/javascript">
$(document).ready(function() {
$("#tabs").tabs();
});
</script>

<div id="tabs">
<ul>
    <li><a href="#statechanges-tab">State Changes</a></li>
    <li><a href="#notifications-tab">Notifications</a></li>
</ul>

<div id="statechanges-tab">

<h5 class="ul">' . _('Host State Change Handler Commands') . '</h5>

<p>' . _('Commands to be executed when host state changes occur.') . ' </p>
    
<table class="table table-condensed table-bordered table-striped table-auto-width">
    <thead>
        <tr>
            <th>' . _('Sequence') . '</th>
            <th>' . _('Enabled') . '</th>
            <th>' . _('Command') . '</th>
            <th>' . _('Don&#39t Run in Downtime') . '</th>
        </tr>
    </thead>
    <tbody>';



        for ($x = 0; $x <= 2; $x++) {

            $output .= '
        <tr>
            <td class="center">' . ($x + 1) . '</td>
            <td class="center">
                <input type="checkbox" id="host_event_handler_commands[' . $x . '][enabled]" name="host_event_handler_commands[' . $x . '][enabled]" ' . is_checked($host_event_handler_commands[$x]["enabled"], 1) . '>
            </td>
            <td>
                <input type="text" size="80" name="host_event_handler_commands[' . $x . '][command]" value="' . htmlspecialchars($host_event_handler_commands[$x]["command"]) . '" class="form-control">
            </td>
            <td class="center">
                <input type="checkbox" id="downtime" name="host_event_handler_commands[' . $x . '][downtime]" value="1"' . is_checked($host_event_handler_commands[$x]['downtime'], 1) . '>
            </td>
        </tr>';
        }



        $output .= '
    </tbody>
</table>

<h5 class="ul">' . _('Service State Change Handler Commands') . '</h5>

<p>' . _('Commands to be executed when service state changes occur.') . ' </p>

<table class="table table-condensed table-bordered table-striped table-auto-width">
    <thead>
        <tr>
            <th>' . _('Sequence') . '</th>
            <th>' . _('Enabled') . '</th>
            <th>' . _('Command') . '</th>
            <th>' . _('Don&#39t Run in Downtime') . '</th>
        </tr>
    </thead>
    <tbody>';



        for ($x = 0; $x <= 2; $x++) {

            $output .= '
        <tr>
            <td class="center">' . ($x + 1) . '</td>
            <td class="center">
                <input type="checkbox" id="service_event_handler_commands[' . $x . '][enabled]" name="service_event_handler_commands[' . $x . '][enabled]" ' . is_checked($service_event_handler_commands[$x]["enabled"], 1) . '>
            </td>
            <td>
                <input type="text" size="80" name="service_event_handler_commands[' . $x . '][command]" value="' . htmlspecialchars($service_event_handler_commands[$x]["command"]) . '" class="form-control">
            </td>
            <td class="center">
                <input type="checkbox"id="downtime" name="service_event_handler_commands[' . $x . '][downtime]" value="1" ' . is_checked($service_event_handler_commands[$x]['downtime'], 1) . '>
            </td>
        </tr>';
        }



        $output .= '
    </tbody>
</table>

</div>

<div id="notifications-tab">

<h5 class="ul">' . _('Host Notification Handler Commands') . '</h5>

<p>' . _('Commands to be executed when host notifications occur.') . '</p>

<table class="table table-condensed table-bordered table-striped table-auto-width">
    <thead>
        <tr>
            <th>' . _('Sequence') . '</th>
            <th>' . _('Enabled') . '</th>
            <th>' . _('Command') . '</th>
        </tr>
    </thead>
    <tbody>';



        for ($x = 0; $x <= 2; $x++) {

            $output .= '
        <tr>
            <td class="center">' . ($x + 1) . '</td>
            <td class="center">
                <input type="checkbox" id="host_notification_handler_commands[' . $x . '][enabled]" name="host_notification_handler_commands[' . $x . '][enabled]" ' . is_checked($host_notification_handler_commands[$x]["enabled"], 1) . '>
            </td>
            <td>
                <input type="text" size="80" name="host_notification_handler_commands[' . $x . '][command]" value="' . htmlspecialchars($host_notification_handler_commands[$x]["command"]) . '" class="form-control">
            </td>
        </tr>
    ';
        }



        $output .= '
    </tbody>
</table>

<h5 class="ul">' . _('Service Notification Handler Commands') . '</h5>

<p>' . _('Commands to be executed when service notifications occur.') . '</p>

<table class="table table-condensed table-bordered table-striped table-auto-width">
    <thead>
        <tr>
            <th>' . _('Sequence') . '</th>
            <th>' . _('Enabled') . '</th>
            <th>' . _('Command') . '</th>
        </tr>
    </thead>
    <tbody>';



        for ($x = 0; $x <= 2; $x++) {

            $output .= '
        <tr>
            <td class="center">' . ($x + 1) . '</td>
            <td class="center">
                <input type="checkbox" id="service_notification_handler_commands[' . $x . '][enabled]" name="service_notification_handler_commands[' . $x . '][enabled]" ' . is_checked($service_notification_handler_commands[$x]["enabled"], 1) . '>
            </td>
            <td>
                <input type="text" size="80" name="service_notification_handler_commands[' . $x . '][command]" value="' . htmlspecialchars($service_notification_handler_commands[$x]["command"]) . '" class="form-control">
            </td>
        </tr>';
        }



        $output .= '
    </tbody>
</table>

</div>

</div>';

        break;


    case COMPONENT_CONFIGMODE_SAVESETTINGS:

        // Get variables
        $host_event_handler_commands           = grab_array_var($inargs, "host_event_handler_commands", "");
        $service_event_handler_commands        = grab_array_var($inargs, "service_event_handler_commands", "");
        $host_notification_handler_commands    = grab_array_var($inargs, "host_notification_handler_commands", "");
        $service_notification_handler_commands = grab_array_var($inargs, "service_notification_handler_commands", "");

        // Validate variables
        $errors = 0;
        $errmsg = array();

        // Handle errors
        if ($errors > 0) {
            $outargs[COMPONENT_ERROR_MESSAGES] = $errmsg;
            $result = 1;
            return '';
        }

        // Save settings
        $settings = array(
            "host_event_handler_commands"           => $host_event_handler_commands,
            "service_event_handler_commands"        => $service_event_handler_commands,
            "host_notification_handler_commands"    => $host_notification_handler_commands,
            "service_notification_handler_commands" => $service_notification_handler_commands,
        );
        set_option("globaleventhandler_component_options", serialize($settings));

        // Info messages
        $okmsg                            = array();
        $okmsg[]                          = "Settings updated.";
        $outargs[COMPONENT_INFO_MESSAGES] = $okmsg;

        break;


    default:
        break;
    }

    return $output;
}


////////////////////////////////////////////////////////////////////////
// EVENT HANDLER AND NOTIFICATION FUNCTIONS
////////////////////////////////////////////////////////////////////////


register_callback(CALLBACK_EVENT_PROCESSED, 'globaleventhandler_component_eventhandler');


function globaleventhandler_component_eventhandler($cbtype, $args)
{
    if (get_option("enable_subsystem_logging", true)) {
        echo "*** GLOBAL HANDLER...\n";
        print_r($args);
    }

    $event_type        = grab_array_var($args, "event_type", "");
    if ($event_type != EVENTTYPE_STATECHANGE && $event_type != EVENTTYPE_NOTIFICATION) {
        return;
    }

    $meta              = grab_array_var($args, "event_meta", array());
    $notification_type = grab_array_var($meta, "notification-type", "");
    $handler_type      = grab_array_var($meta, "handler-type", "");
    $contact           = grab_array_var($meta, "contact", "");

    // Load settings
    $settings_raw = get_option("globaleventhandler_component_options");
    if (empty($settings_raw)) {
        return;
    }
    
    $settings = unserialize($settings_raw);

    if (empty($settings)) {
        return;
    }

    if ($event_type == EVENTTYPE_STATECHANGE) {
        $type = $handler_type;
        $handler = "event";
    } else /* if ($event_type == EVENTTYPE_NOTIFICATION) */ {
        $type = $notification_type;
        $handler = "notification";
    }

    // Ensure that the notification/handler type is valid
    if (!globaleventhandler_valid_type($type)) {
        return;
    }

    // Will be either host_notification_handler_commands or
    //                service_notification_handler_commands or
    //                host_event_handler_commmands or
    //                service_event_handler_commands
    $settings_key = "{$type}_{$handler}_handler_commands";

    // Make sure what we have what we need
    if (empty($settings[$settings_key]) || !is_array($settings[$settings_key])) {
        return;
    }

    // Loop through all notification or handler commands
    foreach ($settings[$settings_key] as $seq => $vals) {

        // Only process commands that are enabled
        // and non-null
        if (!isset($vals["enabled"]) || empty($vals["command"])) {
            continue;
        }

        if ($event_type == EVENTTYPE_STATECHANGE) {

            $downtime = grab_array_var($vals, "downtime", 0);

            if (globaleventhandler_component_retrievedowntime($type, $meta) && $downtime == 1) {
                echo ucfirst($type) . " is in scheduled downtime... EVENT DOWNTIME=1 \n";
                continue;
            }
        }

        globaleventhandler_component_run_command($vals["command"], $meta);
    }
}


////////////////////////////////////////////////////////////////////////
// COMMAND FUNCTIONS
////////////////////////////////////////////////////////////////////////


function globaleventhandler_component_run_command($cmd, $meta)
{
    if (get_option("enable_subsystem_logging", true)) {
        echo "RUNNING GLOBAL EVENT HANDLER...\n";
        echo "CMD=$cmd\n";
        echo "META:\n";
        print_r($meta);
    }

    // Process variables present in the command line
    $cmdline = process_notification_text($cmd, $meta);

    // Execute the command
    exec($cmdline);

    return 0;
}


////////////////////////////////////////////////////////////////////////
// MISC FUNCTIONS
////////////////////////////////////////////////////////////////////////


function globaleventhandler_component_checkeventhandlers()
{
    $xml = get_xml_program_status();

    if ($xml) {

        $v = intval($xml->programstatus->event_handlers_enabled);

        if ($v == 1) {
            return true;
        }
    }

    return false;
}


function globaleventhandler_component_retrievedowntime($handler_type, $meta)
{
    // We don't actually need this, since by the time this function is
    // called, we've already checked for a valid type
    // But if you plan on using this function from another component,
    // you should uncomment the following block:
    /*
    if (!globaleventhandler_valid_type($handler_type)) {
        return false;
    }
    */

    // Use a variable function so we don't need
    // a switch/if statement for two blocks of code
    // that do the exact same thing except different
    // function calls
    // Will either be get_host_status_xml_output or
    //                get_service_status_xml_output
    $xml_cb = "get_{$handler_type}_status_xml_output";
    if (!function_exists($xml_cb)) {
        return false;
    }

    // The array we'll pass to $xml_cb
    $req = array("host_name" => $meta["host"]);
    if ($handler_type == "service") {
        $req["name"] = $meta["service"];
    }

    // Likewise with the variable function, the
    // status property we're looking for changes
    // based on the handler_type
    // Will either be hoststatus or servicestatus
    $status = $handler_type . "status";

    $obj = simplexml_load_string($xml_cb($req));

    // Check for the existence of the appropriate property
    // before attempting to access it
    if (!isset($obj->$status->scheduled_downtime_depth)) {
        return false;
    }

    $downtime = intval($obj->$status->scheduled_downtime_depth);
    if ($downtime > 0) {
        return true;
    }

    return false;
}


function globaleventhandler_valid_type($type)
{
    if ($type == "host" || $type == "service") {
        return true;
    }

    return false;
}
