<?php
//
// MongoDB Server Config Wizard
// Copyright (c) 2013-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

mongodbserver_configwizard_init();

function mongodbserver_configwizard_init()
{
    $name = "mongodbserver";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.0.1",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor a MongoDB Server"),
        CONFIGWIZARD_DISPLAYTITLE => _("MongoDB Server"),
        CONFIGWIZARD_FUNCTION => "mongodbserver_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "mongodb.png",
        CONFIGWIZARD_FILTER_GROUPS => array('database', 'server'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );
    register_configwizard($name, $args);
}

/**
 * @return int
 *          0 = good to go
 *          1 = prereqs non-existent
 *          2 = may need to upgrade pymongo
 */
function mongodbserver_configwizard_check_prereqs()
{
    // Plugin doesn't exist
    if (!file_exists("/usr/local/nagios/libexec/check_mongodb.py")) {
        return 1; // prereqs non-existent
    }

    $preferred_version = '3.4.0';

    $found_yum = false;
    $yum_output = array();

    $found_pip = false;
    $pip_output = array();

    // See if pymongo is installed and get the version number(s)
    $yum_command = "yum list installed | grep pymongo | awk '{ print $2 }'";
    exec($yum_command, $yum_output, $yum_return);

    // See if pymongo is installed via pip and get the version number
    $pip_command = "pip show pymongo --no-cache-dir";
    exec($pip_command, $pip_output, $pip_return);

    // If neither yum nor pip returned anything, there is no need to continue
    if ($yum_return !== 0 || $pip_return !== 0) {
        return 1; // prereqs non-existent
    }

    // Check pip output
    if ($pip_return === 0) {
        foreach ($pip_output as $line) {

            // We're only interested in lines that start with "Version:"
            if (strpos($line, "Version:") === 0) {
                $pip_version = trim(str_replace("Version:", "", $line));

                if (version_compare($preferred_version, $pip_version) < 1) {
                    return 0; // good to go!
                }
            }
        }
    }

    // Check yum output
    if ($yum_return === 0) {
        foreach ($yum_output as $line) {
            $yum_version = trim($line);

            if (version_compare($preferred_version, $yum_version) < 1) {
                return 0; // good to go
            }
        }
    }

    return 2; // need to upgrade pymongo
}


/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function mongodbserver_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    $wizard_name = "mongodbserver";

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    // See what python version is installed
    $v_command = "python --version | grep -o '\s[0-9]\.[0-9]\+'";
    exec($v_command, $v_output, $v_return);

    // decide which python version
    $c_script = "check_mongodb_server_3";
    $service = "xiwizard_mongodbserver_service_3";
    if (floatval($v_output[0]) < 3.0) {
        $service = "xiwizard_mongodbserver_service";
        $c_script = "check_mongodb_server";
    }

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            $check_prereqs = mongodbserver_configwizard_check_prereqs();

            if ($check_prereqs == 1) {
                // Redirect output to buffer -- we can load all 'printed' text into a string by calling ob_get_clean() right before we 'break;' for this case.
                ob_start();
?>
            <div class="alert alert-danger w-50">
                <div class="align-items-center d-flex pb-3">
                    <i class="material-symbols-outlined md-middle md-padding">warning</i>
                    <span class="fw-bold pe-2"><?= _('Error') ?>:</span> <?= _('The python pymongo module is missing from the Nagios XI server.') ?>
                </div>
                <div class="card card-p">
                    <p class="text-start"><?= _('To use this wizard you must install pymongo on your server. If you are using CentOS or RHEL you can run:') ?></p>
                    <pre><code>pip install pymongo</code></pre>
                </div>
            </div>
<?php
                $output = ob_get_clean();

            } else {
                $address = grab_array_var($inargs, "ip_address", "");
                $port = grab_array_var($inargs, "port", "27017");
                $username = grab_array_var($inargs, "username", "");
                $password = grab_array_var($inargs, "password", "");

                if (!empty($services_serial)) {
                    $services = json_decode(base64_decode($services_serial), true);
                }

                $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial");

                if (!empty($serviceargs_serial)) {
                    $serviceargs = json_decode(base64_decode($serviceargs_serial), true);
                }

                # Get the existing host/node configurations.
                # TODO: Include passwords/secrets?
                $nodes = get_configwizard_hosts($wizard_name);

                ########################################################################################
                # Load the html
                # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
                #   to load the PHP from the Step1 file into the $output string.
                ########################################################################################
                ob_start();
                include __DIR__.'/steps/step1.php';
                $output = ob_get_clean();
            }

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address", "");
            $port = grab_array_var($inargs, "port", "");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");

            // Swap out eligible user macros if detected
            $address = nagiosccm_replace_user_macros($address);
            $port = nagiosccm_replace_user_macros($port);
            $username = nagiosccm_replace_user_macros($username);
            $password = nagiosccm_replace_user_macros($password);

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (have_value($address) == false)
                $errmsg[$errors++] = "No address specified.";
            if (have_value($username) == false)
                $errmsg[$errors++] = "No username specified.";
            if (have_value($password) == false)
                $errmsg[$errors++] = "No password specified.";

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $port = grab_array_var($inargs, "port", "");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");

            $ha = @gethostbyaddr($address);

            if ($ha == "") {
                $ha = $address;
            }

            $hostname = grab_array_var($inargs, "hostname", $ha);

            $services = grab_array_var($inargs, "services", array(
                "check_connection" => "on",
                "free_connections" => "on",
                "memory_usage" => "on",
                "mapped_memory" => "on",
                "lock_time" => "on",
                "flush_average" => "on",
                "last_flush" => "on",
                "index_miss_ratio" => "on",
                "num_dbs" => "on",
                "num_collections" => "on",
                "num_queries" => "on",
                "rep_lag" => "off",
                "rep_lag_percent" => "off",
                "rep_state" => "off"
            ));

            $serviceargs = grab_array_var($inargs, "serviceargs", array(
                "check_connection_warning" => "2",
                "check_connection_critical" => "4",
                "free_connections_warning" => "70",
                "free_connections_critical" => "85",
                "memory_usage_warning" => "1",
                "memory_usage_critical" => "2",
                "mapped_memory_warning" => "1",
                "mapped_memory_critical" => "2",
                "lock_time_warning" => "5",
                "lock_time_critical" => "10",
                "flush_average_warning" => "100",
                "flush_average_critical" => "200",
                "last_flush_warning" => "200",
                "last_flush_critical" => "400",
                "index_miss_ratio_warning" => "0.005",
                "index_miss_ratio_critical" => "0.01",
                "num_dbs_warning" => "300",
                "num_dbs_critical" => "500",
                "num_collections_warning" => "300",
                "num_collections_critical" => "500",
                "num_queries_warning" => "150",
                "num_queries_critical" => "200",
                "rep_lag_warning" => "15",
                "rep_lag_critical" => "30",
                "rep_lag_percent_warning" => "50",
                "rep_lag_percent_critical" => "75",
                "rep_state_warning" => "0",
                "rep_state_critical" => "0"
            ));

            // Replace defaults with the given values
            $services_serial = grab_array_var($inargs, "services_serial");

            if (!empty($services_serial)) {
                $services = json_decode(base64_decode($services_serial), true);
            }

            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial");

            if (!empty($serviceargs_serial)) {
                $serviceargs = json_decode(base64_decode($serviceargs_serial), true);
            }

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $hostname = nagiosccm_replace_user_macros($hostname);
            $port = grab_array_var($inargs, "port", "");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");

            $services = grab_array_var($inargs, "services", array());
            $serviceargs = grab_array_var($inargs, "serviceargs", array());

            // check for errors
            $errors = 0;
            $errmsg = array();

            if (is_valid_host_name($hostname) == false) {
                $errmsg[$errors++] = "Invalid host name.";
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $port = grab_array_var($inargs, "port", "");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");

            $services = grab_array_var($inargs, "services", array());
            if (empty($services)) {
                $services_serial = grab_array_var($inargs, "services_serial");
            } else {
                $services_serial = base64_encode(json_encode($services));
            }
            $serviceargs = grab_array_var($inargs, "serviceargs", array());
            if (empty($serviceargs)) {
                $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial");
            } else {
                $serviceargs_serial = base64_encode(json_encode($serviceargs));
            }

            $output = '

        <input type="hidden" name="ip_address" value="' . encode_form_val($address) . '" />
        <input type="hidden" name="hostname" value="' . encode_form_val($hostname) . '" />
        <input type="hidden" name="port" value="' . encode_form_val($port) . '" />
        <input type="hidden" name="username" value="' . encode_form_val($username) . '" />
        <input type="hidden" name="password" value="' . encode_form_val($password) . '" />
        <input type="hidden" name="services_serial" value="' . $services_serial . '" />
        <input type="hidden" name="serviceargs_serial" value="' . $serviceargs_serial . '" />

            ';
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $port = grab_array_var($inargs, "port", "");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");

            $services = grab_array_var($inargs, "services", array());
            if (empty($services)) {
                $services_serial = grab_array_var($inargs, "services_serial");
            } else {
                $services_serial = base64_encode(json_encode($services));
            }
            $serviceargs = grab_array_var($inargs, "serviceargs", array());
            if (empty($serviceargs)) {
                $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial");
            } else {
                $serviceargs_serial = base64_encode(json_encode($serviceargs));
            }

            $output = '

        <input type="hidden" name="ip_address" value="' . encode_form_val($address) . '" />
        <input type="hidden" name="hostname" value="' . encode_form_val($hostname) . '" />
        <input type="hidden" name="port" value="' . encode_form_val($port) . '" />
        <input type="hidden" name="username" value="' . encode_form_val($username) . '" />
        <input type="hidden" name="password" value="' . encode_form_val($password) . '" />
        <input type="hidden" name="services_serial" value="' . $services_serial . '" />
        <input type="hidden" name="serviceargs_serial" value="' . $serviceargs_serial . '" />

            ';
            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            $hostname = grab_array_var($inargs, "hostname", "");
            $address = grab_array_var($inargs, "ip_address", "");
            $port = grab_array_var($inargs, "port", "");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");

            $services_serial = grab_array_var($inargs, "services_serial", "");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            $services = json_decode(base64_decode($services_serial), true);
            $serviceargs = json_decode(base64_decode($serviceargs_serial), true);

            // save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["hostname"] = $hostname;
            $meta_arr["ip_address"] = $address;
            $meta_arr["port"] = $port;
            $meta_arr["username"] = $username;
            $meta_arr["services"] = $services;
            $meta_arr["serviceargs"] = $serviceargs;
            save_configwizard_object_meta($wizard_name, $hostname, "", $meta_arr);

            $objs = array();

            if (!host_exists($hostname)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_mongodbserver_host",
                    "host_name" => $hostname,
                    "address" => $address,
                    "icon_image" => "mongodb.png",
                    "statusmap_image" => "mongodb.png",
                    "_xiwizard" => $wizard_name,
                );
            }

            foreach ($services as $svcvar => $svcval) {

                switch ($svcvar) {

                    case "check_connection":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MongoDB Connection",
                            "use" => $service,
                            "check_command" => $c_script . "!connect!" . $port . "!" . $serviceargs["check_connection_warning"] . "!" . $serviceargs["check_connection_critical"] . "!" . $username . "!" . $password,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "free_connections":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MongoDB Free Connections",
                            "use" => $service,
                            "check_command" => $c_script . "!connections!" . $port . "!" . $serviceargs["free_connections_warning"] . "!" . $serviceargs["free_connections_critical"] . "!" . $username . "!" . $password,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "memory_usage":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MongoDB Memory Usage",
                            "use" => $service,
                            "check_command" => $c_script . "!memory!" . $port . "!" . $serviceargs["memory_usage_warning"] . "!" . $serviceargs["memory_usage_critical"] . "!" . $username . "!" . $password,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "mapped_memory":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MongoDB Mapped Memory Usage",
                            "use" => $service,
                            "check_command" => $c_script . "!memory_mapped!" . $port . "!" . $serviceargs["mapped_memory_warning"] . "!" . $serviceargs["mapped_memory_critical"] . "!" . $username . "!" . $password,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "lock_time":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MongoDB Lock Time",
                            "use" => $service,
                            "check_command" => $c_script . "!lock!" . $port . "!" . $serviceargs["lock_time_warning"] . "!" . $serviceargs["lock_time_critical"] . "!" . $username . "!" . $password,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "flush_average":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MongoDB Flush Time Average",
                            "use" => $service,
                            "check_command" => $c_script . "!flushing!" . $port . "!" . $serviceargs["flush_average_warning"] . "!" . $serviceargs["flush_average_critical"] . "!" . $username . "!" . $password,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "last_flush":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MongoDB Last Flush Time",
                            "use" => $service,
                            "check_command" => $c_script . "!last_flush_time!" . $port . "!" . $serviceargs["last_flush_warning"] . "!" . $serviceargs["last_flush_critical"] . "!" . $username . "!" . $password,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "index_miss_ratio":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MongoDB Index Miss Ratio",
                            "use" => $service,
                            "check_command" => $c_script . "!index_miss_ratio!" . $port . "!" . $serviceargs["index_miss_ratio_warning"] . "!" . $serviceargs["index_miss_ratio_critical"] . "!" . $username . "!" . $password,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "num_dbs":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MongoDB Databases",
                            "use" => $service,
                            "check_command" => $c_script . "!databases!" . $port . "!" . $serviceargs["num_dbs_warning"] . "!" . $serviceargs["num_dbs_critical"] . "!" . $username . "!" . $password,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "num_collections":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MongoDB Collections",
                            "use" => $service,
                            "check_command" => $c_script . "!collections!" . $port . "!" . $serviceargs["num_collections_warning"] . "!" . $serviceargs["num_collections_critical"] . "!" . $username . "!" . $password,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "num_queries":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MongoDB Queries Per Second",
                            "use" => $service,
                            "check_command" => $c_script . "!queries_per_second!" . $port . "!" . $serviceargs["num_queries_warning"] . "!" . $serviceargs["num_queries_critical"] . "!" . $username . "!" . $password,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "rep_lag":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MongoDB Replication Lag",
                            "use" => $service,
                            "check_command" => $c_script . "!replication_lag!" . $port . "!" . $serviceargs["rep_lag_warning"] . "!" . $serviceargs["rep_lag_critical"] . "!" . $username . "!" . $password,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "rep_lag_percent":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MongoDB Replication Lag Percentage",
                            "use" => $service,
                            "check_command" => $c_script . "!replication_lag_percent!" . $port . "!" . $serviceargs["rep_lag_percent_warning"] . "!" . $serviceargs["rep_lag_percent_critical"] . "!" . $username . "!" . $password,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "rep_state":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MongoDB Replicaset State",
                            "use" => $service,
                            "check_command" => $c_script . "!replset_state!" . $port . "!" . $serviceargs["rep_state_warning"] . "!" . $serviceargs["rep_state_critical"] . "!" . $username . "!" . $password,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    default:
                        break;
                }
            }

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:
            break;
    }

    return $output;
}
