<?php
//
// Websensor Config Wizard
// Copyright (c) 2008-2024 - Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

esensors_websensor_configwizard_init();

function esensors_websensor_configwizard_init()
{
    $name = "esensors_websensor";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.0.1",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor temperature, humidity, and light levels on a Esensors Websensor."),
        CONFIGWIZARD_DISPLAYTITLE => _("Esensors Websensor"),
        CONFIGWIZARD_FUNCTION => "esensors_websensor_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "em01.png",
        CONFIGWIZARD_FILTER_GROUPS => array('network'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );
    register_configwizard($name, $args);
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function esensors_websensor_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    $wizard_name = "esensors_websensor";

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            $address = grab_array_var($inargs, "ip_address", "");

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address", "");
            $address_replaced = nagiosccm_replace_user_macros($address);
            $model = grab_array_var($inargs, "model", "");

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (have_value($address_replaced) == false) {
                $errmsg[$errors++] = _("No address specified.");
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $model = grab_array_var($inargs, "model", "");

            $ha = @gethostbyaddr($address);

            if ($ha == "") {
                $ha = $address;
            }

            $hostname = grab_array_var($inargs, "hostname", $ha);

            $temp_warning_low = "60";
            $temp_warning_high = "85";
            $temp_critical_low = "50";
            $temp_critical_high = "95";

            $humidity_warning_low = "15";
            $humidity_warning_high = "80";
            $humidity_critical_low = "10";
            $humidity_critical_high = "90";

            $illumination_warning_low = "";
            $illumination_warning_high = "70";
            $illumination_critical_low = "";
            $illumination_critical_high = "80";

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $hostname_replaced = nagiosccm_replace_user_macros($hostname);

            $model = grab_array_var($inargs, "model");

            // check for errors
            $errors = 0;
            $errmsg = array();

            if (is_valid_host_name($hostname_replaced) == false)
                $errmsg[$errors++] = _("Invalid host name.");

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;


        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $model = grab_array_var($inargs, "model");
            $services = grab_array_var($inargs, "services");
            $serviceargs = grab_array_var($inargs, "serviceargs");

            $output = '

        <input type="hidden" name="ip_address" value="' . encode_form_val($address) . '">
        <input type="hidden" name="hostname" value="' . encode_form_val($hostname) . '">
        <input type="hidden" name="model" value="' . encode_form_val($model) . '">
        <input type="hidden" name="services_serial" value="' . encode_form_val(base64_encode(json_encode($services))) . '">
        <input type="hidden" name="serviceargs_serial" value="' . encode_form_val(base64_encode(json_encode($serviceargs))) . '">

            ';
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:


            $output = '
            ';
            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            $hostname = grab_array_var($inargs, "hostname", "");
            $address = grab_array_var($inargs, "ip_address", "");
            $hostaddress = $address;

            $model = grab_array_var($inargs, "model");

            $services_serial = grab_array_var($inargs, "services_serial", "");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            $services = json_decode(base64_decode($services_serial), true);
            $serviceargs = json_decode(base64_decode($serviceargs_serial), true);

            // save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["hostname"] = $hostname;
            $meta_arr["ip_address"] = $address;
            $meta_arr["model"] = $model;
            $meta_arr["services"] = $services;
            $meta_arr["serivceargs"] = $serviceargs;
            save_configwizard_object_meta($wizard_name, $hostname, "", $meta_arr);

            $objs = array();

            if (!host_exists($hostname)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_websensor_host",
                    "host_name" => $hostname,
                    "address" => $hostaddress,
                    "icon_image" => "em01.png",
                    "statusmap_image" => "em01.png",
                    "_xiwizard" => $wizard_name,
                );
            }

            // see which services we should monitor
            foreach ($services as $svc => $svcstate) {

                //echo "PROCESSING: $svc -> $svcstate<BR>\n";

                switch ($svc) {

                    case "ping":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Ping",
                            "use" => "xiwizard_websensor_ping_service",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "temp":

                        $wl = grab_array_var($serviceargs, "temp_warning_low");
                        if ($wl == "")
                            $wl = "x";
                        $wh = grab_array_var($serviceargs, "temp_warning_high");
                        if ($wh == "")
                            $wh = "x";
                        $cl = grab_array_var($serviceargs, "temp_critical_low");
                        if ($cl == "")
                            $cl = "x";
                        $ch = grab_array_var($serviceargs, "temp_critical_high");
                        if ($ch == "")
                            $ch = "x";

                        switch ($model) {
                            case "EM08-T":
                                $warn = $wl . " " . $wh;
                                $crit = $cl . " " . $ch;
                                $objs[] = array(
                                    "type" => OBJECTTYPE_SERVICE,
                                    "host_name" => $hostname,
                                    "service_description" => "Temperature",
                                    "use" => "xiwizard_websensor_service",
                                    "check_command" => "check_em08_temp!" . $warn . "!" . $crit . "",
                                    "_xiwizard" => $wizard_name,
                                );
                                break;
                            default:
                                $warn = $wl . "/" . $wh;
                                $crit = $cl . "/" . $ch;
                                $objs[] = array(
                                    "type" => OBJECTTYPE_SERVICE,
                                    "host_name" => $hostname,
                                    "service_description" => "Temperature",
                                    "use" => "xiwizard_websensor_service",
                                    "check_command" => "check_em01_temp!" . $warn . "!" . $crit . "",
                                    "_xiwizard" => $wizard_name,
                                );
                                break;
                        }
                        break;

                    case "humidity":

                        $wl = grab_array_var($serviceargs, "humidity_warning_low");
                        if ($wl == "")
                            $wl = "x";
                        $wh = grab_array_var($serviceargs, "humidity_warning_high");
                        if ($wh == "")
                            $wh = "x";
                        $cl = grab_array_var($serviceargs, "humidity_critical_low");
                        if ($cl == "")
                            $cl = "x";
                        $ch = grab_array_var($serviceargs, "humidity_critical_high");
                        if ($ch == "")
                            $ch = "x";

                        switch ($model) {
                            case "EM08-T":
                                $warn = $wl . " " . $wh;
                                $crit = $cl . " " . $ch;
                                $objs[] = array(
                                    "type" => OBJECTTYPE_SERVICE,
                                    "host_name" => $hostname,
                                    "service_description" => "Humidity",
                                    "use" => "xiwizard_websensor_service",
                                    "check_command" => "check_em08_humidity!" . $warn . "!" . $crit . "",
                                    "_xiwizard" => $wizard_name,
                                );
                                break;
                            default:
                                $warn = $wl . "/" . $wh;
                                $crit = $cl . "/" . $ch;
                                $objs[] = array(
                                    "type" => OBJECTTYPE_SERVICE,
                                    "host_name" => $hostname,
                                    "service_description" => "Humidity",
                                    "use" => "xiwizard_websensor_service",
                                    "check_command" => "check_em01_humidity!" . $warn . "!" . $crit . "",
                                    "_xiwizard" => $wizard_name,
                                );
                                break;
                        }
                        break;

                    case "illumination":

                        $wl = grab_array_var($serviceargs, "illumination_warning_low");
                        if ($wl == "")
                            $wl = "x";
                        $wh = grab_array_var($serviceargs, "illumination_warning_high");
                        if ($wh == "")
                            $wh = "x";
                        $cl = grab_array_var($serviceargs, "illumination_critical_low");
                        if ($cl == "")
                            $cl = "x";
                        $ch = grab_array_var($serviceargs, "illumination_critical_high");
                        if ($ch == "")
                            $ch = "x";

                        switch ($model) {
                            case "EM08-T":
                                $warn = $wl . " " . $wh;
                                $crit = $cl . " " . $ch;
                                $objs[] = array(
                                    "type" => OBJECTTYPE_SERVICE,
                                    "host_name" => $hostname,
                                    "service_description" => "Illumination",
                                    "use" => "xiwizard_websensor_service",
                                    "check_command" => "check_em08_light!" . $warn . "!" . $crit . "",
                                    "_xiwizard" => $wizard_name,
                                );
                                break;
                            default:
                                $warn = $wl . "/" . $wh;
                                $crit = $cl . "/" . $ch;
                                $objs[] = array(
                                    "type" => OBJECTTYPE_SERVICE,
                                    "host_name" => $hostname,
                                    "service_description" => "Illumination",
                                    "use" => "xiwizard_websensor_service",
                                    "check_command" => "check_em01_light!" . $warn . "!" . $crit . "",
                                    "_xiwizard" => $wizard_name,
                                );
                                break;
                        }
                        break;
                    default:
                        break;
                }
            }

            //echo "OBJECTS:<BR>";
            //print_r($objs);
            //exit();

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:
            break;
    }

    return $output;
}
