#!/bin/bash -e
#
# Script to automatically upgrade and configure NagVis
#
# NagVis Integration Component - Upgrade
# Copyright (c) 2018-2025 - Nagios Enterprises, LLC. All rights reserved.
#
# NOTE: This script ALWAYS "upgrades" NagVis with each Nagios XI upgrade, which is why many people have broken versions.
#

BASEDIR=$(dirname $(readlink -f $0))
. /usr/local/nagiosxi/var/xi-sys.cfg

NAGVIS_VER="1.9.47c"
NAGIOS_NAGVIS_VER="$NAGVIS_VER-nagios"
TS=$(date +%s)

# Get the version of the current NagVis install
INSTALLED_NAGVIS_VER=$(grep CONST_VERSION /usr/local/nagvis/share/server/core/defines/global.php | sed -E "s/.*CONST_VERSION', '([^']+)'.*/\1/")

echo NAGVIS_VER $NAGVIS_VER
echo INSTALLED_NAGVIS_VER $INSTALLED_NAGVIS_VER

if [ "$NAGVIS_VER" == "$INSTALLED_NAGVIS_VER" ]; then
	echo "NagVis is up to date"
	exit 0
fi


# Check if NagVis is installed
if [ ! -f /usr/local/nagvis/share/index.php ]; then
    echo "NagVis is not installed"
    exit 0
fi

# Check whether we have sufficient privileges
if [ $(whoami) != "root" ]; then
    echo "This script needs to be run as root/superuser." >&2
    exit 0
fi

# Config updates for 1.9.8 (and patches for security)
update_config_and_mods() {

	# Replace links
	sed -i 's|^;host_downtime_url.*|host_downtime_url="[html_cgi]/../nagioscore/ui/cmd.php?cmd_typ=55\&host=[name]"|' /usr/local/nagvis/etc/nagvis.ini.php
	sed -i 's|^;host_ack_url.*|host_ack_url="[html_cgi]/../nagioscore/ui/cmd.php?cmd_typ=96\&host=[name]\&force_check"|' /usr/local/nagvis/etc/nagvis.ini.php
	sed -i 's|^;service_downtime_url.*|service_downtime_url="[html_cgi]/../nagioscore/ui/cmd.php?cmd_typ=56\&host=[name]\&service=[service_description]"|' /usr/local/nagvis/etc/nagvis.ini.php
	sed -i 's|^;service_ack_url.*|service_ack_url="[html_cgi]/../nagioscore/ui/cmd.php?cmd_typ=7\&host=[name]\&service=[service_description]\&force_check"|' /usr/local/nagvis/etc/nagvis.ini.php

}

echo "Backing up old NagVis install ..."

# First, let's make a backup
mkdir -p /store/backups/nagvis
cd /store/backups/nagvis
tar -zcf nagvis-upgrade-$TS.tar.gz /usr/local/nagvis

# Install pre-reqs (only needed on old installs - since we only supported CentOS we don't need to check
# for other operating system installs)
if [ "$distro" == "CentOS" ] || [ "$distro" == "RedHatEnterpriseServer" ] || [ "$distro" == "OracleServer" ]; then
	yum install rsync -y
fi

# Run the install
(
	cd $BASEDIR
	tar zxf nagvis-$NAGIOS_NAGVIS_VER.tar.gz
	cd nagvis-$NAGIOS_NAGVIS_VER
	echo | ./install.sh -b /usr/bin -u $apacheuser -g $apachegroup -w "$httpdconfdir" -i ndo2db -c y -a y -q
	cd ..
	rm -rf nagvis-$NAGIOS_NAGVIS_VER
)

# Update the configuration/do mods
update_config_and_mods

# Update NagVis login to us LogonSession instead of LogonMixed
sed -i 's/logonmodule=.*/logonmodule="LogonSession"/' /usr/local/nagvis/etc/nagvis.ini.php
sed -i 's/logonenvvar="/;logonenvvar="/' /usr/local/nagvis/etc/nagvis.ini.php

# Update apache config to remove Basic Auth
sed -i 's/AuthName/#Authname/' $httpdconfdir/nagvis.conf
sed -i 's/AuthType/#AuthType/' $httpdconfdir/nagvis.conf
sed -i 's/AuthUserFile/#AuthUserFile/' $httpdconfdir/nagvis.conf

# Update settings for apache 2.4?
if [ "$dist" == "el7" ]; then
	echo "Updating for apache 2.4 if we need to..."
	sed -i 's/Require valid-user/Require all granted/' $httpdconfdir/nagvis.conf
else
	sed -i 's/Require valid-user/#Require valid-user/' $httpdconfdir/nagvis.conf
fi

# Remove livestatus backend
sed -i 's/\[backend_live_1\]/;\[backend_live_1\]/' /usr/local/nagvis/etc/nagvis.ini.php
sed -i 's/backendtype="mklivestatus"/;backendtype="mklivestatus"/' /usr/local/nagvis/etc/nagvis.ini.php

# Update ndomy_1 to nagiosxi
sed -i 's/\[backend_ndomy_1\]/\[backend_nagiosxi\]/' /usr/local/nagvis/etc/nagvis.ini.php
sed -i 's/backend="ndomy_1"/backend="nagiosxi"/' /usr/local/nagvis/etc/nagvis.ini.php

# Reload Apache configuration
if [ "$distro" == "CentOS" ] || [ "$distro" == "RedHatEnterpriseServer" ] || [ "$distro" == "OracleServer" ]; then
    service httpd reload
else
    service apache2 reload
fi

echo "NagVis upgraded to $NAGIOS_NAGVIS_VER"
