<?php
//
// Nagios Log Server Config Wizard
// Copyright (c) 2016-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

define("WIZARD_NAME", 'nagioslogserver');
define("WIZARD_ICON", WIZARD_NAME.'.png');

nagioslogserver_configwizard_init();

function nagioslogserver_configwizard_init()
{
    register_configwizard(WIZARD_NAME, array(
        CONFIGWIZARD_NAME => WIZARD_NAME,
        CONFIGWIZARD_VERSION => '2.0.2',
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _('Monitor a Nagios Log Server query.'),
        CONFIGWIZARD_DISPLAYTITLE => 'Nagios Log Server',
        CONFIGWIZARD_FUNCTION => WIZARD_NAME.'_configwizard_func',
        CONFIGWIZARD_PREVIEWIMAGE => WIZARD_ICON,
        CONFIGWIZARD_FILTER_GROUPS => array('nagios', 'server'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    ));
}

/**
 * Build a Log Server query URL.
 *
 * @param string $base Base URL to the Log Server Web UI (e.g., 'http://host.domain/nagioslogserver/').
 * @param string $path API controller/action path (e.g., 'check/get_queries' or 'system/cpu_status').
 * @param string $token Log Server API access token.
 *
 * @return string The assembled API URL.
 */
function build_api_url($base, $path, $token)
{
    if (substr($base, -1) != '/') $base .= '/'; // Make base url end in a slash.
    return "{$base}index.php/api/$path?token=$token";
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function nagioslogserver_configwizard_func($mode = '', $inargs = null, &$outargs = null, &$result = null)
{
    $result = 0; // Stage result status code: 0 okay; 1 ERROR!!! BAAAAHHHHHHHHHHD!!!!!!!!!!...
    $output = ''; // HTML output string.

    // Initialize output args - pass back the same data we got.
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:
            $address = grab_array_var($inargs, 'ip_address');
            $url = grab_array_var($inargs, 'url');
            $key = grab_array_var($inargs, 'key');
            $self_cert = grab_array_var($inargs, "self_cert");

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts(WIZARD_NAME);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            // Collect and validate submitted values.
            $address = grab_array_var($inargs, 'ip_address');
            $url = grab_array_var($inargs, 'url');
            $key = grab_array_var($inargs, 'key');
            $self_cert = grab_array_var($inargs, "self_cert", "");

            $address = nagiosccm_replace_user_macros($address);
            $url = nagiosccm_replace_user_macros($url);

            $errors = 0;
            $errmsg = array();

            if (empty($address)) {
                $errmsg[$errors++] = _('No host address specified.');
            }

            if (empty($url)) {
                $errmsg[$errors++] = _('No Log Server URL specified.');
            }

            if (empty($key)) {
                $errmsg[$errors++] = _('No API key specified.');
            }

            if (!$errors) {
                // CPU status as a test URL is shorter and simpler than subsystem status.
                $api_url = build_api_url($url, 'system/cpu_status', $key);
                $context = null;
                if ($self_cert == 'on') {
                    $self_signed_cert_context = array(
                        "ssl" => array(
                            "verify_peer"       => false,
                            "verify_peer_name"  => false,
                            ),
                        );

                    $context = stream_context_create($self_signed_cert_context);
                }

                // We should be able to get a response with the creds we have.
                $json = @file_get_contents($api_url, 0, $context);

                if (empty($json)) {
                    $errmsg[$errors++] = _('Unable to contact server at ') . $url . '. If you are using SSL and you have a self signed cert, checking the checkbox below to allow self signed certs may fix your issue.';
                } else {
                    $data = json_decode($json, true);

                    if (!$data || !is_array($data)) {
                        $errmsg[$errors++] = _('Server returned invalid output. (Is this a Nagios Log Server?)');
                    } else if (grab_array_var($data, 'error') && grab_array_var($data, 'type') == 'authentication') {
                        $errmsg[$errors++] = _('Bad API key. Message from server is') . ': "' . grab_array_var($data, 'message') . '"';
                    } else if (!array_key_exists('cpu_usage', $data)) {
                        $errmsg[$errors++] = _('Server didn\'t return expected output. (Is this a Nagios Log Server?)');
                    }
                }
            }

            if ($errors) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Collect our submitted values.
            $address = grab_array_var($inargs, 'ip_address');
            $ha = @gethostbyaddr($address);
            $hostname = grab_array_var($inargs, 'hostname', $ha ? $ha : $address);
            $url = grab_array_var($inargs, 'url');
            $key = grab_array_var($inargs, 'key');
            $self_cert = grab_array_var($inargs, "self_cert", "");

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;


        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:
            $address = grab_array_var($inargs, 'ip_address');
            $hostname = grab_array_var($inargs, 'hostname');
            $hostname = nagiosccm_replace_user_macros($hostname);
            $url = grab_array_var($inargs, 'url');
            $key = grab_array_var($inargs, 'key');

            $services = '';
            $serviceargs = '';

            // Use encoded data if user came back from future screen.
            $services_serial = grab_array_var($inargs, 'services_serial');

            if ($services_serial) {
                $services = json_decode(base64_decode($services_serial), true);
            }

            $serviceargs_serial = grab_array_var($inargs, 'serviceargs_serial');

            if ($serviceargs_serial) {
                $serviceargs = json_decode(base64_decode($serviceargs_serial), true);
            }

            // If no serialized data, use current request data if available.
            if (!$services) {
                $services = grab_array_var($inargs, 'services', array());
            }

            if (!$serviceargs) {
                $serviceargs = grab_array_var($inargs, 'serviceargs', array());
            }

            // Now validate the data.
            $errors = 0;
            $errmsg = array();

            if (!is_valid_host_name($hostname)) {
                $errmsg[$errors++] = _('Invalid host name.');
            }

            $have_id_queries = array_key_exists('id', $services) && count($services['id']);
            $have_text_queries = array_key_exists('query', $services) && count($services['query']);

            if (!$have_id_queries && !$have_text_queries) {
                $errmsg[$errors++] = _('You have not specified any queries to monitor.');
            }

            if ($have_id_queries) foreach ($services['id'] as $i => $v) {
                // We need all data for selected query ID rows.
                if (!$serviceargs['id'][$i]['name'])
                    $errmsg[$errors++] = _('Missing display name.');
                if (!$serviceargs['id'][$i]['id'])
                    $errmsg[$errors++] = _('Missing query ID.');
                $minutes = $serviceargs['id'][$i]['minutes'];
                if (!$minutes)
                    $errmsg[$errors++] = _('Minutes must be greater than zero.');
                else if (!is_numeric($minutes) || $minutes != intval($minutes))
                    $errmsg[$errors++] = _('Minutes must be an integer.');
                if (!$serviceargs['id'][$i]['warning'])
                    $errmsg[$errors++] = _('Missing warning range.');
                if (!$serviceargs['id'][$i]['critical'])
                    $errmsg[$errors++] = _('Missing critical range.');
            }

            if ($have_text_queries) foreach ($services['query'] as $i => $v) {
                // Validate non-empty common data first, then more intelligent
                // type-specific checks later.
                if (!$serviceargs['query'][$i]['name'])
                    $errmsg[$errors++] = _('Missing display name.');

                $query = $serviceargs['query'][$i]['query'];

                if (!$query) {
                    $errmsg[$errors++] = _('Missing query.');
                }

                $minutes = $serviceargs['query'][$i]['minutes'];

                if (!$minutes)
                    $errmsg[$errors++] = _('Minutes must be greater than zero.');
                else if (!is_numeric($minutes) || $minutes != intval($minutes))
                    $errmsg[$errors++] = _('Minutes must be an integer.');
                if (!$serviceargs['query'][$i]['warning'])
                    $errmsg[$errors++] = _('Missing warning range.');
                if (!$serviceargs['query'][$i]['critical'])
                    $errmsg[$errors++] = _('Missing critical range.');

                $type = $serviceargs['query'][$i]['type'];

                if (!$type) {
                    $errmsg[$errors++] = _('Missing query type.');
                } else if ($query) switch ($type) {
                    case 'text':
                        // Non-empty text: we'll escape shell characters when
                        // building the plugin command later, and the plugin
                        // URL encodes the query.
                        break;
                    case 'json':
                        // Validate that the JSON parses to an object.
                        if (!is_object(json_decode($query)))
                            $errmsg[$errors++] = _('Query JSON is invalid.');
                        break;
                    default:
                        $errmsg[$errors++] = _('Invalid query type') . ": '$type'";
                        break;
                }
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }
            break;

        case CONFIGWIZARD_MODE_GETSTAGE3HTML:
            $address = encode_form_val(grab_array_var($inargs, 'ip_address'));
            $hostname = encode_form_val(grab_array_var($inargs, 'hostname'));
            $url = encode_form_val(grab_array_var($inargs, 'url'));
            $key = encode_form_val(grab_array_var($inargs, 'key'));
            $self_cert = grab_array_var($inargs, "self_cert", "");

            $services = json_encode(grab_array_var($inargs, 'services'));
            $services_serial = grab_array_var($inargs, 'services_serial', base64_encode($services));

            $serviceargs = json_encode(grab_array_var($inargs, 'serviceargs'));
            $serviceargs_serial = grab_array_var($inargs, 'serviceargs_serial', base64_encode($serviceargs));

            $output = '
            <input type="hidden" name="ip_address" value="' . $address . '">
            <input type="hidden" name="hostname" value="' . $hostname . '">
            <input type="hidden" name="url" value="' . $url . '">
            <input type="hidden" name="key" value="' . $key . '">
            <input type="hidden" name="self_cert" value="' . $self_cert . '">
            <input type="hidden" name="services_serial" value="' . $services_serial . '">
            <input type="hidden" name="serviceargs_serial" value="' . $serviceargs_serial . '">
            <!-- SERVICES=' . $services . PHP_EOL . 'SERVICEARGS=' . $serviceargs . '-->';
            break;


        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:
            break;


        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:
            break;


        case CONFIGWIZARD_MODE_GETOBJECTS:
            $address = grab_array_var($inargs, 'ip_address');
            $hostname = grab_array_var($inargs, 'hostname');
            $url = grab_array_var($inargs, 'url');
            $key = grab_array_var($inargs, 'key');
            $self_cert = grab_array_var($inargs, "self_cert");

            $services_serial = grab_array_var($inargs, 'services_serial');
            $serviceargs_serial = grab_array_var($inargs, 'serviceargs_serial');

            $services = json_decode(base64_decode($services_serial), true);
            $serviceargs = json_decode(base64_decode($serviceargs_serial), true);

            // save data for later use in re-entrance
            save_configwizard_object_meta(WIZARD_NAME, $hostname, '', array(
                'hostname' => $hostname,
                'ip_address' => $address,
                'url' => $url,
                'key' => $key,
                'self_cert' => $self_cert,
                'services' => $services,
                'serivceargs' => $serviceargs,
            ));

            $objs = array();

            if (!host_exists($hostname)) {
                $objs[] = array(
                    'type' => OBJECTTYPE_HOST,
                    'use' => 'xiwizard_nagioslogserver_host',
                    'host_name' => $hostname,
                    'address' => $address,
                    'icon_image' => WIZARD_ICON,
                    'statusmap_image' => WIZARD_ICON,
                    '_xiwizard' => WIZARD_NAME,
                );
            }

            // The base API URL for loading queries.
            $api_url = build_api_url($url, 'check/get_queries', $key) . '&id=';

            // Escape and build the common command arguments.
            $url = escapeshellarg($url);
            $key = escapeshellarg($key);
            $cmdargs_base = "--url=$url --apikey=$key";

            // Figure out the 'services' we should monitor.
            foreach ($services as $service_type => $selected_services) {
                foreach (array_keys($selected_services) as $i) {
                    $name = $serviceargs[$service_type][$i]['name'];
                    $mins = escapeshellarg($serviceargs[$service_type][$i]['minutes']);
                    $warn = escapeshellarg($serviceargs[$service_type][$i]['warning']);
                    $crit = escapeshellarg($serviceargs[$service_type][$i]['critical']);

                    $cmdargs = "$cmdargs_base --minutes=$mins --warn=$warn --crit=$crit";

                    switch ($service_type) {
                        case 'id':
                            $id = $serviceargs['id'][$i]['id'];
                            $result = @load_url($api_url.$id, array("method" => "post", "disable_verifypeer" => true), false , $self_cert);
                            $result = json_decode($result, true);
                            $query = is_array($result) && isset($result['raw'])
                                ? $result['raw'] : ' "Failed to load query"';
                            $cmdargs .= ' --query=' . escapeshellarg($query);
                            // If querying by ID we would add this instead:
                            //$cmdargs .= ' --id=' . $serviceargs['id'][$i]['id'];
                            break;

                        case 'query':
                            $query_type = $serviceargs['query'][$i]['type'];
                            $query_text = $serviceargs['query'][$i]['query'];
                            setlocale(LC_CTYPE, "en_US.UTF-8");
                            $query_text = escapeshellcmd($query_text);
                            $query_text = addslashes($query_text);

                            switch ($query_type) {
                                case 'text':
                                    $query_type = 'string';
                                    break;
                                case 'json':
                                    $query_type = 'query';
                                    break;
                                default:
                                    // Skipping bad values for now, should show a message somewhere...
                                    # TODO: continue does the same thing as break, inside of a switch.
                                    # Is "break" the intended function or should it be "continue 2" or continue 3?
                                    #continue;
                                    break;
                            }

                            $cmdargs .= " --$query_type='$query_text'";
                            break;

                        default:
                            // Skipping bad values for now, should show a message somewhere...
                            # TODO: continue does the same thing as break, inside of a switch.
                            # Is "break" the intended function or should it be "continue 2"?
                            #continue;
                            break;
                    }

                    $objs[] = array(
                        'type' => OBJECTTYPE_SERVICE,
                        'host_name' => $hostname,
                        'service_description' => $name,
                        'use' => 'xiwizard_nagioslogserver_service',
                        'check_command' => 'check_xi_service_nagioslogserver!' . $cmdargs,
                        '_xiwizard' => WIZARD_NAME,
                    );
                }
            }

            // Return the object definitions to the wizard.
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;
            break;


        default:
            break;
    }

    return $output;
}
