<?php
//
// Custom Variables Tab Component
// Copyright (c) 2010-2024 Nagios Enterprises, LLC. All rights reserved.
//  
// $Id: freevariabletab.inc.php 38 2010-06-07 22:28:29Z swilkerson $

require_once(dirname(__FILE__).'/../componenthelper.inc.php');

$customvariabletab_component_name = "customvariabletab";
customvariabletab_component_init();


////////////////////////////////////////////////////////////////////////
// COMPONENT INIT FUNCTIONS
////////////////////////////////////////////////////////////////////////


function customvariabletab_component_init()
{
    global $customvariabletab_component_name;
    $versionok = customvariabletab_component_checkversion();
    
    $desc = "";
    if (!$versionok) {
        $desc = "<b>"._("Error: This component requires Nagios XI 2024R1.1 or later.")."</b>";
    }
    
    $args = array(
        COMPONENT_NAME => $customvariabletab_component_name,
        COMPONENT_AUTHOR => "Nagios Enterprises, LLC",
        COMPONENT_DESCRIPTION => _("Adds a new tabs on host and service detail screens to show custom  variables from config.  ").$desc,
        COMPONENT_TITLE => _("Custom  Variable Tab"),
        COMPONENT_VERSION => '1.2.1',
        COMPONENT_CONFIGFUNCTION => "customvariabletab_component_config_func",
        COMPONENT_REQUIRES_VERSION => 60100
    );
        
    register_component($customvariabletab_component_name, $args);
    
    if ($versionok) {
        register_callback(CALLBACK_SERVICE_TABS_INIT, 'customvariabletab_component_addtab');
        register_callback(CALLBACK_HOST_TABS_INIT, 'customvariabletab_component_addtab');
    }
}


///////////////////////////////////////////////////////////////////////////////////////////
// MISC FUNCTIONS
///////////////////////////////////////////////////////////////////////////////////////////


function customvariabletab_component_checkversion()
{
    if(!function_exists('get_product_release'))
        return false;
    if(get_product_release()<60100)
        return false;
    if(!function_exists('get_nagios_session_protector'))
        return false;
    return true;
}


function customvariabletab_component_config_func($mode, $inargs, &$outargs, &$result)
{
    $result = 0;
    $output = '';
    $component_name = "customvariabletab";

    switch ($mode)
    {
        case COMPONENT_CONFIGMODE_GETSETTINGSHTML:
        
            $settings_raw = get_option("customvariabletab_component_options");
            if ($settings_raw == "") {
                $settings = array(
                    "enabled" => 1,
                );
            } else
                $settings = unserialize($settings_raw);

            // initial values
            $enabled = grab_array_var($settings, "enabled", "1");

            // values passed to us
            $enabled = checkbox_binary(grab_array_var($inargs, "enabled", $enabled));
            

            $output .= '            
<h5 class="ul">' . _('Custom Variable Tab Settings') . '</h5>

<table>
    <tr>
        <td class="checkbox">
            <label>
                <input type="checkbox" id="enabled" name="enabled" ' . is_checked($enabled, 1) . '>
                ' . _('Enable Custom Variable Tab') . '
            </label>
        </td>
    </tr>
</table';

            break;

        case COMPONENT_CONFIGMODE_SAVESETTINGS:
            // get variables
            $enabled = checkbox_binary(grab_array_var($inargs, "enabled", ""));

            // validate variables
            $errors = 0;
            $errmsg = array();
            if ($enabled == 1) {
                /*
                if(have_value($user_dn)==false){
                    $errmsg[$errors++]="No user DN specified.";
                    }
                */
            }

            // handle errors
            if ($errors > 0) {
                $outargs[COMPONENT_ERROR_MESSAGES] = $errmsg;
                $result = 1;
                return '';
            }

            // save settings
            $settings = array(
                "enabled" => $enabled,
            );
            set_option("customvariabletab_component_options", serialize($settings));
            break;

        default:
            break;
    }

    return $output;
}


function customvariabletab_component_addtab($cbtype, &$cbdata)
{
    // get our settings
    $settings_raw = get_option("customvariabletab_component_options");
    if ($settings_raw == "") {
        $settings = array(
            "enabled" => 1,
        );
    } else
        $settings = unserialize($settings_raw);

    // initial values
    $enabled = grab_array_var($settings, "enabled");

    // bail out if we're not enabled...
    if ($enabled != 1) {
        return;
    }
    
    
    $hostname = grab_array_var($cbdata, "host");
    $servicename = grab_array_var($cbdata, "service");

    $content = '<div class="infotable_title">'._('Custom Variables').'</div>';

    $customvariables = customvariabletab_component_get_customvariables($hostname, $servicename);

    if ($customvariables) {
        $content .= '<table class="table table-bordered table-striped" style="max-width: 800px;"><tr><th style="width: 200px;">'._('Name').'</th><th>'._('Value').'</th></tr>';
        
        foreach ($customvariables->customvar as $customvariable) {
            $content .= '<tr><td>'.$customvariable->name.'</td><td>'.$customvariable->value.'</td></tr>'; 
        }

        $content .= '</table>';
    } else {
        $content .= _("No custom variables currently set");

        
    }


    if(is_neptune()) {
        $newtab = array(
            "id" => "customvariabletab",
            "title" => _("Custom Variables"),
            "content" => $content,
            "icon" => '  '
        );
    } else {
        $newtab = array(
            "id" => "customvariabletab",
            "title" => _("Custom Variables"),
            "content" => $content,
            "icon" => '<i class="material-symbols-outlined md-18 md-300 md-middle">list_alt</i>'
        );
    }

    $cbdata["tabs"][] = $newtab;
}

    
function customvariabletab_component_get_customvariables($hostname, $servicename=null)
{
    $args = array(
        "host_name" => $hostname
    );
    
    if (!$servicename) {
        $x = get_xml_custom_host_variable_status($args);
        if ($x->customhostvarstatus->customvars) {
            return $x->customhostvarstatus->customvars;
        }
    } else {
        $args["service_description"] = $servicename;
        $x = get_xml_custom_service_variable_status($args);
        if ($x->customservicevarstatus->customvars) {
            return $x->customservicevarstatus->customvars;
        }
    }

    return false;
}
