<?php
//
// Nagios XI Operations Center Component
// Copyright (c) 2011-2021 Nagios Enterprises, LLC. All rights reserved.
//

require_once(dirname(__FILE__) . '/../../common.inc.php');

// Initialization and session start
pre_init();
init_session();
grab_request_vars();
check_prereqs();

// Check authentication
check_authentication(false);

// Grab page variables
$host = grab_request_var("host", "");
$service = grab_request_var("service", "");
$hostgroup = grab_request_var("hostgroup", "");
$servicegroup = grab_request_var("servicegroup", "");
$state = grab_request_var("state", "");

// Grab extra filters
$filters = grab_request_var('filters', array());
$update_flag = grab_request_var('update_flag', 0);
$clear = grab_request_var('clear', 0);
if ($clear) {
    $filters = array();
    $update_flag = 1;
}

// If the user clicked the update button, set user metas
if ($update_flag) {
    // If the hide_not_dis is in the array, the user has turned that filter on
    if (in_array('hide_not_dis', $filters)) {
        // Turn filter on
        set_user_meta(0, 'nocscreen_hide_not_disabled', 1, false);
    } else {
        set_user_meta(0, 'nocscreen_hide_not_disabled', 0, false);
    }

    // If the hide_handled is in the array, the user has turned that filter on
    if (in_array('hide_handled', $filters)) {
        // Turn filter on
        set_user_meta(0, 'nocscreen_hide_handled', 1, false);
    } else {
        set_user_meta(0, 'nocscreen_hide_handled', 0, false);
    }

    // If the hide_soft is in the array, the user has turned that filter on
    if (in_array('hide_soft', $filters)) {
        // Turn filter on
        set_user_meta(0,'nocscreen_hide_soft_states', 1, false);
    } else {
        set_user_meta(0,'nocscreen_hide_soft_states', 0, false);
    }
}

// Grab user metas after they have been updated
// ** There is probably a better way to do this **
$hide_not_dis = get_user_meta(0, 'nocscreen_hide_not_disabled');
$hide_handled = get_user_meta(0, 'nocscreen_hide_handled');
$hide_soft = get_user_meta(0, 'nocscreen_hide_soft_states');
$host_orderby = get_user_meta(0, 'nocscreen_host_orderby', "last_state_change:a");
$service_orderby = get_user_meta(0, 'nocscreen_service_orderby', "last_state_change:a");
$show_filters = get_user_meta(0, 'nocscreen_show_filters', 1);

$refresh = intval(get_user_meta(0, 'nocscreen_page_refresh', 10));
if ($refresh < 1) {
    $refresh = 1;
}

$title = _("Operations Center");
do_page_start(array("page_title" => $title, "jquery_plugins" => array("sumoselect")), true);
?>

<style type="text/css">
    #topbar { margin-top: 10px; vertical-align: top; }
    h1 { margin: 0; padding: 0; }
    h1 img { margin-right: 10px; }
    #leftside { float: left; }
    #rightside { float: right; }
    .strong { font-weight: bold; }
    #lastUpdate { color: #777; font-size: 11px; margin: 10px 0; }
    div.duration { width: 125px; }
    #noc_filter{ margin: 0; }
    .noc-filters { margin: 5px 0 15px 0; }
    .filters-btn { margin-left: 10px; }
    /* Sometimes the OK button has camelcase, sometimes it doesn't. Don't ask me why. */
    .btnOk {background-color: #4D89F9; border-color: #3371E6; color: white;}
    .btnOk:hover {background-color: #3371E6 !important;}
    .btnok {background-color: #4D89F9; border-color: #3371E6; color: white;}
    .btnok:hover {background-color: #3371E6 !important;}
    .sound-select-grid { display: grid; grid-template-columns: 1fr 1fr 1fr 1fr }
    .sound-select { grid-row: span 1; grid-column: span 1 }
    .condensed-checkbox[type=checkbox] { vertical-align: bottom; margin-top: 0px; margin: 0px 0px 0px; height: 13px; box-shadow: none;}
</style>

<script type="text/javascript">
var INT_ID;
$(document).ready(function () {

    //$('.refresh').html('<i class="fa fa-spinner fa-14 fa-spin"></i>');

    // Initialize our sumoselects
    // These are all done independently so we can act upon them more easily
    extraFilterSelect = $('#extra-filters-select').SumoSelect({
        okCancelInMulti: true,
        captionFormatAllSelected: "<?php echo _('All filters selected'); ?>",
        csvDispCount: 2,
        placeholder: '<?php echo _('Select Additional Filters'); ?>'
    });

    stateSelect = $('#state-select').SumoSelect({});

    hostSelect = $('#host-select').SumoSelect({
        triggerChangeCombined: true,
        csvDispCount: 2,
        search: true,
        searchText: 'Search...'
    });

    serviceSelect = $('#service-select').SumoSelect({
        triggerChangeCombined: true,
        csvDispCount: 2,
        search: true,
        searchText: 'Search...'
    });

    servicegroupSelect = $('#servicegroup-select').SumoSelect({
        triggerChangeCombined: true,
        csvDispCount: 2,
        search: true,
        searchText: 'Search...'
    });

    hostgroupSelect = $('#hostgroup-select').SumoSelect({
        triggerChangeCombined: true,
        csvDispCount: 2,
        search: true,
        searchText: 'Search...'
    });

    // Sounds stuff needs to exist before NOC content.
    // Rest of sound-related code at bottom of script
    bind_sounds();
    display_sound_preferences();

    // Main NOC content and loop
    noc_load_content();
    INT_ID = setInterval(noc_load_content, <?php echo intval($refresh) * 1000; ?>);

    // Hide/show filters
    $('.filters-btn').click(function() {
        if ($('.noc-filters').is(':visible')) {
            $('.noc-filters').hide();
            $('.filters-btn').html('<span class="material-icons md-16">filter_alt</span> <?php echo _("Show Filters"); ?>');
            $.post('nocscreenapi.php', { 'cmd': 'setfilter', 'show': 0 });
        } else {
            $('.noc-filters').show();
            $('.filters-btn').html('<span class="material-icons md-16">filter_alt</span> <?php echo _("Hide Filters"); ?>');
            $.post('nocscreenapi.php', { 'cmd': 'setfilter', 'show': 1 });
        }
    });

    // Clear hostgroup and servicegroup selects if a host is selected
    $(hostSelect).change(function ()
    {
        // Check for empty string to avoid infinite loop
        if ($(this).val() !== '') {
            // Clear the hostgroup select
            $('#hostgroup-select > .placeholder').prop('disabled', false);
            hostgroupSelect.sumo.selectItem('');
            hostgroupSelect.sumo.unSelectAll();
            $('#hostgroup-select > .placeholder').prop('disabled', true);
            hostgroupSelect.sumo.reload();

            // Clear the servicegroup
            $('#servicegroup-select > .placeholder').prop('disabled', false);
            servicegroupSelect.sumo.selectItem('');
            servicegroupSelect.sumo.unSelectAll();
            $('#servicegroup-select > .placeholder').prop('disabled', true);
            servicegroupSelect.sumo.reload();
        } else {
            return;
        }
    });

    // Clear host and servicegroup selects if a hostgroup is selected
    $(hostgroupSelect).change(function ()
    {
        // Check for empty string to avoid infinite loop
        if ($(this).val() !== '') {
            // Clear the host select
            $('#host-select > .placeholder').prop('disabled', false);
            hostSelect.sumo.selectItem('');
            hostSelect.sumo.unSelectAll();
            $('#host-select > .placeholder').prop('disabled', true);
            hostSelect.sumo.reload();

            // Clear the servicegroup
            $('#servicegroup-select > .placeholder').prop('disabled', false);
            servicegroupSelect.sumo.selectItem('');
            servicegroupSelect.sumo.unSelectAll();
            $('#servicegroup-select > .placeholder').prop('disabled', true);
            servicegroupSelect.sumo.reload();
        } else {
            return false;
        }
        
    });

    // Clear host and servicegroup selects if a hostgroup is selected
    $(servicegroupSelect).change(function ()
    {
        // Check for empty string to avoid infinite loop
        if ($(this).val() !== '') {
            // Clear the host select
            $('#host-select > .placeholder').prop('disabled', false);
            hostSelect.sumo.selectItem('');
            hostSelect.sumo.unSelectAll();
            $('#host-select > .placeholder').prop('disabled', true);
            hostSelect.sumo.reload();

            // Clear the servicegroup
            $('#hostgroup-select > .placeholder').prop('disabled', false);
            hostgroupSelect.sumo.selectItem('');
            hostgroupSelect.sumo.unSelectAll();
            $('#hostgroup-select > .placeholder').prop('disabled', true);
            hostgroupSelect.sumo.reload();
        } else {
            return false;
        }

    });
    
    $('#hosttable .host_order').live('click', function() {
        $('#orderby_update').val(1);
        if ($('#host_orderby').val() == 'name:a') {
            $('#host_orderby').val('name:d');
        } else {
            $('#host_orderby').val('name:a');
        }
        noc_load_content();
    });
    
    $('#hosttable .duration_order').live('click', function() {
        $('#orderby_update').val(1);
        if ($('#host_orderby').val() == 'last_state_change:a') {
            $('#host_orderby').val('last_state_change:d');
        } else {
            $('#host_orderby').val('last_state_change:a');
        }
        noc_load_content();
    });
    
    $('#servicetable .host_order').live('click', function() {
        $('#orderby_update').val(1);
        if ($('#service_orderby').val() == 'host_name:a') {
            $('#service_orderby').val('host_name:d');
        } else {
            $('#service_orderby').val('host_name:a');
        }
        noc_load_content();
    });
    
    $('#servicetable .service_order').live('click', function() {
        $('#orderby_update').val(1);
        if ($('#service_orderby').val() == 'service_description:a') {
            $('#service_orderby').val('service_description:d');
        } else {
            $('#service_orderby').val('service_description:a');
        }
        noc_load_content();
    });
    
    $('#servicetable .duration_order').live('click', function() {
        $('#orderby_update').val(1);
        if ($('#service_orderby').val() == 'last_state_change:a') {
            $('#service_orderby').val('last_state_change:d');
        } else {
            $('#service_orderby').val('last_state_change:a');
        }
        noc_load_content();
    });

    $('#submit-button').click(function() {
        $('#noc_filter').append('<input type="hidden" id="update-flag" name="update_flag" value="1"/>');
    })

    // Settings dropdown
    $('.btn-update-settings').click(function() {
        var refresh = parseInt($('.page-refresh').val());
        if (refresh) {
            clearInterval(INT_ID);
            var rval = refresh * 1000;
            INT_ID = setInterval(noc_load_content, rval);

            // Set refresh value in user meta
            $.post('nocscreenapi.php', { 'cmd': 'setrefresh', 'refresh': refresh });
        }
        $("#settings-dropdown").fadeOut("fast");
    });

    $('.btn-close-settings').click(function() {
        $("#settings-dropdown").fadeOut("fast");
    });

    if ($('#open-dtform').length > 0) {
        var p = $('#open-dtform').position();
        var left = Math.floor(p.left+8) - ($('#settings-dropdown').outerWidth()/2);
        var top = Math.floor(p.top+14);
        $('#settings-dropdown').css('left', left+'px');
        $('#settings-dropdown').css('top', top+'px');

        $("#open-dtform").click(function () {
            var d = $("#settings-dropdown").css("display");
            if (d == "none") {
                $("#settings-dropdown").fadeIn("fast");
            } else {
                $("#settings-dropdown").fadeOut("fast");
            }
        });
    }

    // Begin sounds-stuff
    $('.sound-btn').click(function () {
        $('#configure-sound').modal('show');
    });

    $('.select-sound').click(function () {
        var name = $(this).data('name');
        $('#selected-sound').html(name);
        $('input[name=sound-selected][value="'+name+'"]').prop('checked', true);
    })

    // The actual 'custom audio upload' button. 
    // jQuery doesn't seem to work with XHR level 2, so this is using mostly native JS.
    $('#audio-upload-submit').click(function (e) {
        e.preventDefault();

        $('#upload-failed-msg').html('');

        var formdata = new FormData();

        var files = $('input[name=uploadedfile]')[0].files;

        for (var i = 0; i < files.length; i++) {
            file = files[i];
            if (!file.type.match('audio.*')) {
                var errmsg = '<?php echo _("File %s is not an audio file.");?>';
                errmsg = errmsg.replace('%s', file.name);
                $('#upload-failed-msg').html(errmsg);
                return;
            }

            formdata.append('audio[]', file, file.name);
        }

        // form data was set without issues. Start the AJAX call.
        var xhr = new XMLHttpRequest();
        xhr.open('POST', 'nocajax.php?mode=audio-upload', true);

        xhr.onload = function (e) {
            var responseData = JSON.parse(e.target.responseText);
            if (responseData.hasOwnProperty('error')) {
                $('#upload-failed-msg').html(responseData.errmsg);
            }
            noc_load_audio();
        }

        xhr.send(formdata);
    });

    $('input[name=uploadedfile]').change(function() {
        var fnames = [];
        for (var i = 0; i < this.files.length; i++) {
            fnames.push(this.files[i].name);
        }
        $('.filename-readonly').val(fnames.join(', '));
    });

    $('#clear-sound-prefs').click(function () {
        save_sound_preferences('clear_sound_prefs');
    });

    // Saves user-level sound preferences (which sounds get played via which events)
    $('#save-sound-prefs').click(function () {
        // the 'short name' of the sound we've chosen. 
        // IDs for the rest of the program prepend to this name.
        var sound = $('#selected-sound').html();
        save_sound_preferences(sound);
    });
});

function save_sound_preferences(sound) {
    var events = [];
    $('input.sound-user-prefs:checked').each(function () {
        // data('name') here is like 'host-up' or 'service-unknown'
        events.push($(this).data('name'));
    });


    var args = {
        'events': events,
        'sound': sound
    };


    $.post('nocajax.php?mode=save-audio-prefs', args, function (data) {
        if (typeof data === 'undefined') {
            // an unknown error has occurred
            return;
        }
        else if (data.hasOwnProperty('error') && data.error) {
            $('#upload-failed-msg').html(data.errmsg);
        }
        else if (data.hasOwnProperty('success') && data.success) {
            $('#upload-failed-msg').css({'color': 'green'});
            $('#upload-failed-msg').html(data.successmsg);
        }

        display_sound_preferences();
    }, 'json');
}

// This retrieves the user's sound preferences via ajax and updates the "event -> sound"-linking table.
function display_sound_preferences() {

    $.get('nocajax.php?mode=get-audio-prefs', {}, function (data) {

        // This div doesn't do anything, except allow us to determine whether the linker has been loaded.
        var sound_linker_html = '<div></div>';

        for (which in data) {
            $('#current-' + which).html(data[which]);
            sound_linker_html += '<div id="link-' + which + '" data-sound="' + data[which] + '"></div>'
        }

        $('#sound-linker').html(sound_linker_html);

    }, 'json');
}

// Retrieves a bunch of sound metadata, 
// which is then used to render the actual <audio>s 
// as well as the play/delete table within the modal.
function noc_load_audio() {
    $.get('nocajax.php?mode=load-audio', {}, function(data) {
        $('#upload-throbber').hide();
        sound_container_html = '';
        sound_table_html = '';

        for (var i = 0; i < data.length; i++) {

            sound_container_html += '<audio id="sound-' + data[i].short_name + '">';
            sound_container_html += '<source src="' + data[i].web_path + '" type="' + data[i].mime + '">';
            sound_container_html += '</audio>';

            if (i % 2 == 0) {
                sound_table_html += '<tr>';
            }
            sound_table_html += '<td style="width: 40%; cursor: pointer;" class="select-sound" data-name="'+ data[i].short_name + '"><label><input type="radio" name="sound-selected" style="vertical-align: bottom; margin: 4px 4px 0 4px;" value="'+ data[i].short_name + '">' + data[i].short_name + '</label></td>';
            sound_table_html += '<td style="width: 10%; text-align: right;"><button type="button" data-name="' + data[i].short_name + '" class="btn btn-xs btn-default btn-play-sound btn-alignment"><span class="material-icons md-16" style="padding: 0px;">play_arrow</span></button>';

            <?php if (is_admin()) { ?>
            sound_table_html += '<button type="button" class="btn btn-xs btn-default btn-delete-sound btn-alignment" data-file="'+ data[i].file +'"><span class="material-icons md-16" style="color: red; padding:0px;">close</span></button>';
            <?php } ?>

            sound_table_html += '</td>';

            if (i % 2 == 1) {
                sound_table_html += '</tr>';
            }
        }

        if (data.length % 2 == 1) {
            sound_table_html += '</tr>';
        }

        $('#sound-container').html(sound_container_html);
        $('#sound-table').html(sound_table_html);
        bind_sounds();
    }, 'json');
    $('#upload-throbber').show();
}

// binds the 'play' and 'delete' buttons for each sound in the configure sounds menu.
// gets called on page load and each time a sound is uploaded/deleted.
function bind_sounds() {
    $('.btn-play-sound').off('click');
    $('.btn-play-sound').click(function() {
        var name = $(this).data('name');
        // idk if jquery works for this part
        var audio_element = document.getElementById('sound-' + name);
        // cloneNode allows the same sound to play over itself multiple times.
        // Otherwise, you can only play one instance of a specific audio file at a time.
        audio_element.cloneNode(true).play();
    });

    $('.btn-delete-sound').off('click');
    $('.btn-delete-sound').click(function() {
        if (confirm('Do you want to delete this sound file?')) {
            query = { 'file' : $(this).data('file') };
            $.post('nocajax.php?mode=delete-audio', query, function(data) {
                if (data.hasOwnProperty('error')) {
                    $('#upload-failed-msg').html(data.errmsg);
                }
                noc_load_audio();
            }, 'json');
        }
    });
}

function noc_load_content()
{
    var datastring = $("#noc_filter").serialize();

    // Summary bar info
    $.ajax({
        url: 'nocscreenapi.php?summary=true',
        data: datastring
    }).done(function (html) {
        $('#rightside').html(html);
        $('.tooltip.fade.left.in').remove();
    });

    // Get orderby settings
    datastring += '&orderby_update=' + $('#orderby_update').val();
    datastring += '&host_orderby=' + $('#host_orderby').val();
    datastring += '&service_orderby=' + $('#service_orderby').val();
    $('#orderby_update').val(0);

    // Main NOC content
    $.ajax({
        url: 'nocscreenapi.php',
        data: datastring
    }).done(function (html) {
        var d = new Date();
        $('#lastUpdate').empty();
        $('#lastUpdate').append(( 'Last Update: ' + d.toString() ));
        $('#content').html(html);

        noc_play_sounds();
    });

}

// only used in noc_maybe_play_sound
classlist_regexes = {
    'host': /(host)(.*)/i,
    'service': /(service)(.*)/i
};

var sounds_last_host_length = 0;
var sounds_last_service_length = 0;
function noc_play_sounds() {

    // Tables are rendered on the backend, so go into each row to parse necessary info for sounds.
    var host_rows = $('#hosttable > tbody > tr.hostrow');
    host_rows.each(function () {

        noc_maybe_play_sound(this, 'host', 'host_name');
    });

    /* This doesn't catch every 'up' state, and with filters it might
     * also catch other non-ok states, but the NOC screen doesn't render
     * OK/UP objects, so that data isn't available to us.
     */
    if (host_rows.length < sounds_last_host_length) {
        noc_play_sound('host', 'up');
    }
    sounds_last_host_length = host_rows.length;

    var service_rows = $('#servicetable > tbody > tr.servicerow');
    service_rows.each(function() {
        
        noc_maybe_play_sound(this, 'service', 'description');
    });

    /* This doesn't catch every 'ok' state, and with filters it might
     * also catch other non-ok states, but the NOC screen doesn't render
     * OK/UP objects, so that data isn't available to us.
     */
    if (service_rows.length < sounds_last_service_length) {
        noc_play_sound('service', 'ok');
    }
    sounds_last_service_length = service_rows.length;
}

// Reads the NOC table, plays the relevant sounds if any states changed since last reload.
// row_this: the 'this' from selecting each table row on the noc screen
// type: 'host' or 'service'
// name: 'host_name' or 'description' (this is a class on one of the <td>s)
function noc_maybe_play_sound(row_this, type, name)
{
    duration = $(row_this).children('td').children('div.duration').html();
    // Check for duration up to '10s'. If it's more, don't play the sound.
    skip_sound = !duration.match(/^(10|[0-9])s/i);
    if (skip_sound) {
        return;
    }

    // the <td> with a class 'host_name' or 'description' has only one other class, 
    // which can be parsed to tell you type (which we have anyways) and the current state of the object.
    // type and state determine which sound gets played
    var candidate_classes = $(row_this).children('td.' + name)[0].classList;
    var matches;
    for (var i = 0; i < candidate_classes.length; i++) {
        if (candidate_classes[i] !== name) {

            matches = candidate_classes[i].match(classlist_regexes[type]);
            matches = matches.slice(1);
        }
    }

    noc_play_sound(matches[0], matches[1]);
}

// sounds get loaded in asynchonously wrt table content (which calls this function), so 
// this waits for the sound-linking table to get updated before trying to play the sound.
function noc_play_sound(type, state)
{
    var check_exists = setInterval(function() {

        link_html = $('#sound-linker').html();
        var sound_name = $('#link-' + type + '-' + state).data('sound');

        var audio = $('#sound-' + sound_name);
        if (audio.length) {
            audio[0].play();
            clearInterval(check_exists);
        }
        else if (link_html) {
            clearInterval(check_exists);
        }
    }, 1000);
}
</script>

<?php
// Include graph explorer if we have it
if (function_exists('graphexplorer_component_js_include')) {
    graphexplorer_component_js_include();
}
?>

<?php if (is_neptune()) { ?>

<div id="topbar" style="display:flex; flex-direction:column;">
    <div id="leftside" class="neptune-flex">
        <span class="flex items-center" style="gap:5px;">
            <h1 class="fl"><?php echo _("Operations Center"); ?></h1>
            <a class="tt-bind settings-dropdown" id="open-dtform" data-placement="right" title="<?php echo _('Edit page settings'); ?>"><span class="material-symbols-outlined neptune-icon-md-btn">settings</span></a>
        </span>

        <div id="settings-dropdown">
            <div class="content">       
                <div class="input-group" style="width: 200px;">
                    <label class="input-group-addon"><?php echo _('Refresh every'); ?></label>
                    <input type="text" class="page-refresh form-control" value="<?php echo $refresh; ?>" placeholder="30">
                    <label class="input-group-addon"><?php echo _('sec'); ?></label>
                </div>
                <div style="margin-top: 10px">
                    <button type="button" class="btn btn-update-settings btn-xs btn-primary fl btn-alignment"><span class="material-icons md-16" style="padding: 0px;">check</span> <?php echo _('Update'); ?></button>
                    <button type="button" class="btn btn-close-settings btn-xs btn-default fr btn-alignment"><span class="material-icons md-16" style="padding: 0px;">close</span> <?php echo _('Close'); ?></button>
                    <div class="clear"></div>
                </div>
            </div>
        </div>

        <div class="flex items-center" style="gap:10px;">
            <button type="button" class="filters-btn btn btn-xs btn-default neptune-flex-center" style="gap: 5px;"><span class="material-icons md-18" style="padding: 0px;">filter_alt</span><?php if ($show_filters) { echo _('Hide Filters'); } else { echo _('Show Filters'); } ?></button>
            <button type="button" class="sound-btn btn btn-xs btn-default neptune-flex-center" style="gap: 5px;"><span class="material-icons md-18" style="padding: 0px;">volume_up</span> <?php echo _("Configure Sound"); ?></button>
        </div>
        <div class="clear"></div>
    </div>
    <?php echo neptune_section_spacer(); ?>
    <div id="rightside"></div>
    <div class="clear"></div>
    <?php echo neptune_section_spacer(); ?>
    <div id="lastUpdate"></div>
</div>

<?php } else { ?>

    <div id="topbar">
    <div id="leftside" class="leftside-op-center">
        <img src="<?php echo theme_image('nagiosxi-logo-small.png'); ?>" class="fl" alt="<?php echo _("Nagios XI"); ?>">
        <h1 class="fl" style="margin: 0 15px;"><?php echo _("Operations Center"); ?></h1>
        <a class="tt-bind settings-dropdown" id="open-dtform" data-placement="right" title="<?php echo _('Edit page settings'); ?>">
            <span class="material-symbols-outlined">settings</span>
        </a>        
        <div id="settings-dropdown">
            <div class="content">
                <div class="input-group" style="width: 200px;">
                    <label class="input-group-addon"><?php echo _('Refresh every'); ?></label>
                    <input type="text" class="page-refresh form-control" value="<?php echo $refresh; ?>" placeholder="30">
                    <label class="input-group-addon"><?php echo _('sec'); ?></label>
                </div>
                <div style="margin-top: 10px">
                    <button type="button" class="btn btn-update-settings btn-xs btn-primary fl btn-alignment"><span class="material-icons md-16" style="padding: 0px;">check</span> <?php echo _('Update'); ?></button>
                    <button type="button" class="btn btn-close-settings btn-xs btn-default fr btn-alignment"><span class="material-icons md-16" style="padding: 0px;">close</span> <?php echo _('Close'); ?></button>
                    <div class="clear"></div>
                </div>
            </div>
        </div>
        <button type="button" class="filters-btn btn btn-xs btn-default btn-alignment"><span class="material-icons md-16" style="padding: 0px;">filter_alt</span> <?php if ($show_filters) { echo _('Hide Filters'); } else { echo _('Show Filters'); } ?></button>
        <button type="button" class="sound-btn btn btn-xs btn-default btn-alignment"><span class="material-icons md-16" style="padding: 0px;">volume_up</span> <?php echo _("Configure Sound"); ?></button>
        <div class="clear"></div>
    </div>
    <div id="rightside"></div>
    <div class="clear"></div>
    <div id="lastUpdate"></div>
</div>



<?php } ?>

<?php if(is_neptune()) { ?>

<div class="noc-filters neptune-well" <?php if (!$show_filters) { echo 'style="display: none;"'; } ?>>    
    <form id="noc_filter" class="form-inline noc-filter-form" method="get" action="">
        <div class="input-group">
            <label  class="neptune-input-group-addon-item input-group-addon"><?php echo _('Host'); ?></label>
            <select class="hide" style="width: 150px;" name="host" id="host-select">
                <option  value="">&nbsp;</option>
                <?php
                $args = array('brevity' => 1, 'orderby' => 'host_name:a');
                $oxml = get_xml_host_objects($args);
                if ($oxml) {
                    foreach ($oxml->host as $hostobject) {
                        $name = strval($hostobject->host_name);
                        echo "<option value='" . $name . "' " . is_selected($host, $name) . ">$name</option>\n";
                    }
                }
                ?>
            </select>
        </div>
 
        <div class="input-group">
            <label class="input-group-addon"><?php echo _('Hostgroup'); ?></label>
            <select class="hide" style="width: 150px;" name="hostgroup" id="hostgroup-select">
                <option value="">&nbsp;</option>
                <?php
                $args = array('orderby' => 'hostgroup_name:a');
                $oxml = get_xml_hostgroup_objects($args);
                if ($oxml) {
                    foreach ($oxml->hostgroup as $hg) {
                        $name = strval($hg->hostgroup_name);
                        echo "<option value='" . $name . "' " . is_selected($hostgroup, $name) . ">$name</option>\n";
                    }
                }
                ?>
            </select>
        </div>

        <div class="input-group">
            <label class="input-group-addon"><?php echo _('Servicegroup'); ?></label>
            <select class="hide" style="width: 150px;" name="servicegroup" id="servicegroup-select">
                <option value="">&nbsp;</option>
                <?php
                $args = array('orderby' => 'servicegroup_name:a');
                $oxml = get_xml_servicegroup_objects($args);
                if ($oxml) {
                    foreach ($oxml->servicegroup as $sg) {
                        $name = strval($sg->servicegroup_name);
                        echo "<option value='" . $name . "' " . is_selected($servicegroup, $name) . ">$name</option>\n";
                    }
                }
                ?>
            </select>
        </div>

        <div class="input-group">
            <label class="input-group-addon"><?php echo _('Service State'); ?></label>
            <select class="hide" style="width: 100px;" name="state" id="state-select">
                <option value="">&nbsp;</option>
                <option value="1" <?php echo is_selected($state, 1); ?> ><?php echo _('Warning') ?></option>
                <option value="2" <?php echo is_selected($state, 2); ?> ><?php echo _('Critical') ?></option>
                <option value="3" <?php echo is_selected($state, 3); ?> ><?php echo _('Unknown') ?></option>
            </select>
        </div>

        <div class="input-group">
            <label class="input-group-addon neptune-input-group-addon-item"><?php echo _('Additional Filters'); ?></label>
            <select style="height:36px;" class="hide" style="width: 200px;" name="filters[]" id="extra-filters-select" multiple>
                <option value="hide_not_dis" <?php echo is_selected($hide_not_dis, 1); ?> ><?php echo _('Hide notifications disabled') ?></option>
                <option value="hide_handled" <?php echo is_selected($hide_handled, 1); ?> ><?php echo _('Hide handled') ?></option>
                <option value="hide_soft" <?php echo is_selected($hide_soft, 1); ?> ><?php echo _('Hide soft states') ?></option>
            </select>
        </div>
        <button type="submit" class="reporttimesubmitbutton btn btn-sm btn-primary" name="reporttimesubmitbutton" id="submit-button"><span class="material-icons md-16" style="padding: 0px;">check</span> <?php echo _("Filter"); ?></button>
        <a href="noc.php?clear=1" class="neptune-btn btn btn-sm btn-default"><span class="material-icons md-16" style="padding: 0px;">close</span><?php echo _('Clear filters'); ?></a>

    </form>

    <!-- These are just here to be accessed -->
    <input id="host_orderby" type="hidden" name="host_orderby" value="<?php echo $host_orderby; ?>">
    <input id="service_orderby" type="hidden" name="service_orderby" value="<?php echo $service_orderby; ?>">
    <input id="orderby_update" type="hidden" name="orderby_update" value="0">

    <div class="clear"></div>
</div>

<?php } else { ?>
    <div class="noc-filters well" <?php if (!$show_filters) { echo 'style="display: none;"'; } ?>>    
        <form id="noc_filter" style="display:flex; flex-wrap:wrap; gap:10px;" class="form-inline" method="get" action="">

            <div class="input-group">
                <label class="input-group-addon"><?php echo _('Host'); ?></label>
                <select class="hide" style="width: 150px;" name="host" id="host-select">
                    <option value="">&nbsp;</option>
                    <?php
                    $args = array('brevity' => 1, 'orderby' => 'host_name:a');
                    $oxml = get_xml_host_objects($args);
                    if ($oxml) {
                        foreach ($oxml->host as $hostobject) {
                            $name = strval($hostobject->host_name);
                            echo "<option value='" . $name . "' " . is_selected($host, $name) . ">$name</option>\n";
                        }
                    }
                    ?>
                </select>
            </div>
    
            <div class="input-group">
                <label class="input-group-addon"><?php echo _('Hostgroup'); ?></label>
                <select class="hide" style="width: 150px;" name="hostgroup" id="hostgroup-select">
                    <option value="">&nbsp;</option>
                    <?php
                    $args = array('orderby' => 'hostgroup_name:a');
                    $oxml = get_xml_hostgroup_objects($args);
                    if ($oxml) {
                        foreach ($oxml->hostgroup as $hg) {
                            $name = strval($hg->hostgroup_name);
                            echo "<option value='" . $name . "' " . is_selected($hostgroup, $name) . ">$name</option>\n";
                        }
                    }
                    ?>
                </select>
            </div>

            <div class="input-group">
                <label class="input-group-addon"><?php echo _('Servicegroup'); ?></label>
                <select class="hide" style="width: 150px;" name="servicegroup" id="servicegroup-select">
                    <option value="">&nbsp;</option>
                    <?php
                    $args = array('orderby' => 'servicegroup_name:a');
                    $oxml = get_xml_servicegroup_objects($args);
                    if ($oxml) {
                        foreach ($oxml->servicegroup as $sg) {
                            $name = strval($sg->servicegroup_name);
                            echo "<option value='" . $name . "' " . is_selected($servicegroup, $name) . ">$name</option>\n";
                        }
                    }
                    ?>
                </select>
            </div>

            <div class="input-group">
                <label class="input-group-addon"><?php echo _('Service State'); ?></label>
                <select class="hide" style="width: 100px;" name="state" id="state-select">
                    <option value="">&nbsp;</option>
                    <option value="1" <?php echo is_selected($state, 1); ?> ><?php echo _('Warning') ?></option>
                    <option value="2" <?php echo is_selected($state, 2); ?> ><?php echo _('Critical') ?></option>
                    <option value="3" <?php echo is_selected($state, 3); ?> ><?php echo _('Unknown') ?></option>
                </select>
            </div>

            <div class="input-group">
                <label class="input-group-addon"><?php echo _('Additional Filters'); ?></label>
                <select class="hide" style="width: 200px;" name="filters[]" id="extra-filters-select" multiple>
                    <option value="hide_not_dis" <?php echo is_selected($hide_not_dis, 1); ?> ><?php echo _('Hide notifications disabled') ?></option>
                    <option value="hide_handled" <?php echo is_selected($hide_handled, 1); ?> ><?php echo _('Hide handled') ?></option>
                    <option value="hide_soft" <?php echo is_selected($hide_soft, 1); ?> ><?php echo _('Hide soft states') ?></option>
                </select>
            </div>

            <button type="submit" class="reporttimesubmitbutton btn btn-sm btn-primary btn-alignment" name="reporttimesubmitbutton" id="submit-button"><span class="material-icons md-16" style="padding: 0px;">check</span> <?php echo _("Filter"); ?></button>
            <a href="noc.php?clear=1" class="btn btn-sm btn-default btn-alignment"><span class="material-icons md-16" style="padding: 0px;">close</span> <?php echo _('Clear filters'); ?></a>

        </form>

        <!-- These are just here to be accessed -->
        <input id="host_orderby" type="hidden" name="host_orderby" value="<?php echo $host_orderby; ?>">
        <input id="service_orderby" type="hidden" name="service_orderby" value="<?php echo $service_orderby; ?>">
        <input id="orderby_update" type="hidden" name="orderby_update" value="0">

        <div class="clear"></div>
    </div>
<?php } ?>



<div id="content"></div>

<?php $sounds = get_alarm_sounds(); ?>
<div class="hide" id="sound-container">
    <?php for ($i = 0; $i < count($sounds); $i++) { ?>
    <audio id="<?php echo 'sound-' . $sounds[$i]['short_name']; ?>">
        <source src="<?php echo $sounds[$i]['web_path'];?>" type="<?php echo $sounds[$i]['mime'];?>">
    </audio>
    <?php } ?>
</div>
<div class="hide" id="sound-linker">
    <!-- Populated by display_sound_preferences() -->
</div>
<div class="modal" id="configure-sound">
    <div class="modal-dialog" style="width: 800px">
        <div class="modal-content" style="border-radius: 0px;">
            <div class="modal-header">
                <button type="button" class="close btn-alignment" data-dismiss="modal" aria-hidden="true">×</button>
                <h3 style="margin-top: 7px; margin-bottom: 6px;"><?php echo _("Configure Sound"); ?></h3>
            </div>
            <div class="modal-body">
                <h5><?php echo _("Select Sound:");?></h4>
                <div style="max-height: 200px; overflow-y: auto">
                    <table class="table table-condensed table-striped table-no-border">
                        <tbody id="sound-table">
                            <?php for ($i = 0; $i < count($sounds); $i++) {
                                if ($i % 2 == 0) {
                                    echo '<tr class="primary-foreground">';
                                }
                            ?>
                                <td class="select-sound" data-name="<?php echo encode_form_val($sounds[$i]['short_name']); ?>" style="width: 40%; cursor: pointer;">
                                    <label class="neptune-margin-config-sound">
                                        <input type="radio" name="sound-selected" style="vertical-align: bottom; margin: 4px 4px 0 4px;" value="<?php echo encode_form_val($sounds[$i]['short_name']); ?>">
                                        <?php echo encode_form_val($sounds[$i]['short_name']); ?>
                                    </label>
                                </td>
                                <td class="flex items-center btn-alignment gap-2" style="width: 10%; text-align: right;">
                                    <button type="button" class="btn btn-xs btn-default btn-play-sound btn-alignment" data-name="<?php echo encode_form_val($sounds[$i]['short_name']); ?>"><span class="material-icons md-16" style="padding: 0px;">play_arrow</span></button>
                                    <?php if (is_admin()) { ?>
                                    <button type="button" class="btn btn-xs btn-default btn-delete-sound btn-alignment" data-file="<?php echo encode_form_val($sounds[$i]['file']); ?>"><span class="material-icons md-16" style="color: red; padding:0px;">close</span></button>
                                    <?php } ?>
                                </td>
                            <?php 
                                if ($i % 2 == 1) {
                                    echo '</tr>';
                                }
                            }
                            if (count($sounds) % 2 == 1) {
                                echo '</tr>';
                            }
                            ?>
                        </tbody>
                    </table>
                    <div class="text-medium"><?php echo _("Currently Selected:"); ?> <div id="selected-sound" style="display: inline-block"></div></div>
                </div>

                <h5><?php echo _("Play for these changes:"); ?></h5>
                <div class="sound-select-grid gap-2">
                    <div class="sound-select">
                        <input type="checkbox" id="soundselect-host-up" data-name="host-up" class="form-control condensed-checkbox anti-form-control sound-user-prefs anti-form-control" style="">
                        <label for="soundselect-host-up"><?php echo _('Host -> UP'); ?></label>
                        <div><?php echo sprintf(_('Currently: %s'), '');?><span id="current-host-up"></span></div>
                    </div>
                    <div class="sound-select">
                        <input type="checkbox" id="soundselect-host-down" data-name="host-down" class="form-control condensed-checkbox sound-user-prefs anti-form-control" style="">
                        <label for="soundselect-host-down"><?php echo _('Host -> DOWN'); ?></label>
                        <div><?php echo sprintf(_('Currently: %s'), '');?><span id="current-host-down"></span></div>
                    </div>
                    <div class="sound-select">
                        <input type="checkbox" id="soundselect-host-unreachable" data-name="host-unreachable" class="form-control condensed-checkbox sound-user-prefs anti-form-control" style="">
                        <label for="soundselect-host-unreachable"><?php echo _('Host -> UNREACHABLE'); ?></label>
                        <div><?php echo sprintf(_('Currently: %s'), '');?><span id="current-host-unreachable"></span></div>
                    </div>
                    <div class="sound-select">
                    </div>
                    <div class="sound-select">
                        <input type="checkbox" id="soundselect-service-ok" data-name="service-ok" class="form-control condensed-checkbox sound-user-prefs anti-form-control" style="">
                        <label for="soundselect-service-ok"><?php echo _('Service -> OK'); ?></label>
                        <div><?php echo sprintf(_('Currently: %s'), '');?><span id="current-service-ok"></span></div>
                    </div>
                    <div class="sound-select">
                        <input type="checkbox" id="soundselect-service-warning" data-name="service-warning" class="form-control condensed-checkbox sound-user-prefs anti-form-control" style="">
                        <label for="soundselect-service-warning"><?php echo _('Service -> WARNING'); ?></label>
                        <div><?php echo sprintf(_('Currently: %s'), '');?><span id="current-service-warning"></span></div>
                    </div>
                    <div class="sound-select">
                        <input type="checkbox" id="soundselect-service-critical" data-name="service-critical" class="form-control condensed-checkbox sound-user-prefs anti-form-control" style="">
                        <label for="soundselect-service-critical"><?php echo _('Service -> CRITICAL'); ?></label>
                        <div><?php echo sprintf(_('Currently: %s'), '');?><span id="current-service-critical"></span></div>
                    </div>
                    <div class="sound-select">
                        <input type="checkbox" id="soundselect-service-unknown" data-name="service-unknown" class="form-control condensed-checkbox sound-user-prefs anti-form-control" style="">
                        <label for="soundselect-service-unknown"><?php echo _('Service -> UNKNOWN'); ?></label>
                        <div><?php echo sprintf(_('Currently: %s'), '');?><span id="current-service-unknown"></span></div>
                    </div>
                </div>

                <div style="text-align: center; margin-top: 15px" class="flex items-center gap-2">
                    <button type="button" class="btn btn-sm btn-primary" id="save-sound-prefs" style=""><?php echo _("Save"); ?></button>
                    <button type="button" class="btn btn-sm btn-default" id="clear-sound-prefs"><?php echo _("Clear Selected Preferences"); ?></button>
                </div>

                <h5><?php echo _("Upload an Audio File");?></h5>

                <form enctype="multipart/form-data" action="components.php" method="post" style="margin: 0;">
                    <?php echo get_nagios_session_protector(); ?>
                    <input type="hidden" name="upload" value="1">
                    <input type="hidden" name="MAX_FILE_SIZE" value="<?php echo get_php_upload_max_filesize(); ?>">

                    <div style="text-align: left;">
                        <div class="" style="display: inline-block; vertical-align: bottom; border-right: 0px">
                            <span class="btn btn-sm btn-default btn-file">
                                <?php echo _('Browse'); ?>&hellip; <input type="file" name="uploadedfile">
                            </span>
                        </div>
                        <div style="display: inline-block; vertical-align: bottom; margin-left: -5px;">
                            <input type="text" class="form-control filename-readonly" style="width: 200px;" readonly>
                        </div>
                        <div style="display: inline-block">
                            <button type="submit" id="audio-upload-submit" class="btn btn-sm btn-primary"><?php echo _('Upload'); ?></button>
                        </div>
                        <div id="upload-failed-msg" class="" style="display: inline-block; height: 15px; font-weight: bold; color: red; vertical-align: middle; margin-left: 20px;">
                            
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <div class="text-small" style="vertical-align: bottom; display: inline-block; text-align: left; width: 80%; font-size: 0.8rem"><?php echo sprintf(_('All base sounds were recorded by Daniel Simon and retrieved from soundbible.com under CC-BY-3.0-US. %1$sSee full license.%2$s'), '<a href="https://creativecommons.org/licenses/by/3.0/us/" target="_blank" style="vertical-align: inherit; display: inherit; text-align: inherit;">', '</a>'); ?>
                </div>
                <div style="display: inline-block; width: calc(20% - 5px);">
                    <button type="button" class="btn btn-sm btn-default" data-dismiss="modal"><?php echo _('Close');?></button>
                </div>
            </div>
        </div>
    </div>
</div>

<?php
do_page_end(true);
