<?php
//
// DUO Two Factor Authentication Component
// Copyright (c) 2018-2022 Nagios Enterprises, LLC. All rights reserved.
//

use Duo\DuoUniversal\Client;
use Duo\DuoUniversal\DuoException;

require_once(dirname(__FILE__).'/../componenthelper.inc.php');

if (version_compare(PHP_VERSION, '7.3.0', '>=')) {
    require_once(__DIR__ . '/vendor/autoload.php');
}

// The default timeout for DUO Two Factor Authentication
define("DEFAULT_DUO_2FA_TIMEOUT", 60);

duo_component_init();

function duo_component_init()
{
    $version_ok = duo_component_checkversion();

    $desc_err_msg = "";
    if ($version_ok != true) {
        $desc_err_msg = "<br><strong>" . _("Error: This component requires Nagios XI 5.5.0 or later.") . "</strong>";
    }

    $args = array(
        COMPONENT_NAME             => "duo",
        COMPONENT_VERSION          => "1.1.2",
        COMPONENT_AUTHOR           => "Nagios Enterprises, LLC",
        COMPONENT_DESCRIPTION      => _("Allows DUO to be used as a Two Factor Authentication source for Nagios XI") . $desc_err_msg,
        COMPONENT_TITLE            => _("DUO 2FA Integration"),
        COMPONENT_CONFIGFUNCTION   => "duo_config_func",
        COMPONENT_REQUIRES_VERSION => 5500,
    );

    $registered = register_component("duo", $args);

    // we only register the callbacks if the version is ok AND duo is enabled
    if ($registered == true && $version_ok == true) {

        // the actual 2fa callbacks
        register_callback(CALLBACK_TWO_FACTOR_AUTH, 'duo_check_2fa');
        register_callback(CALLBACK_TWO_FACTOR_POST, 'duo_check_2fa_post');

        // register callbacks for user management
        register_callback(CALLBACK_USER_EDIT_PROCESS, 'duo_manage_users');
        register_callback(CALLBACK_USER_EDIT_HTML_AUTH_SETTINGS, 'duo_manage_users');
    }
}

/////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////
// Two Factor Authentication Functions
/////////////////////////////////////////////////////////////////////////////////////////

function duo_check_2fa($cbtype, &$cbargs)
{
    if (duo_2fa_enabled() == false) {
        $cbargs["info_messages"][0] = _("DUO two factor authentication disabled globally, skipping DUO 2FA");
        $cbargs["2fa_skip"] = 1;
        $cbargs["2fa_disabled"] = 1;
        return;
    }

    if (duo_2fa_good_to_go() != true) {
        $cbargs["info_messages"][0] = _("DUO settings not configured properly, skipping DUO 2FA");
        $cbargs["2fa_skip"] = 1;
        return;        
    }

    $credentials = grab_array_var($cbargs, "credentials", array());

    $username = grab_array_var($credentials, "username");
    if (empty($username)) {
        $cbargs["info_messages"][0] = _("Username cannot be blank, skipping DUO 2FA");
        $cbargs["2fa_skip"] = 1;
        return;
    }

    $user_id = get_user_id($username);
    if (empty($user_id)) {
        $cbargs["info_messages"][0] = _("User ($username) doesn't seem to exist, skipping DUO 2FA");
        $cbargs["2fa_skip"] = 1;
        return;
    }

    $duo_2fa = get_user_meta($user_id, "duo_2fa", "enabled");
    if ($duo_2fa == "skip") {
        $cbargs["info_messages"][0] = _("User ($username) configured to skip DUO 2FA");
        $cbargs["2fa_skip"] = 1;
        return;
    }

    $duo_ikey = duo_2fa_ikey();
    $duo_skey = duo_2fa_skey();
    $duo_host = duo_2fa_host();
    $duo_akey = duo_2fa_akey();

    $duo_use_legacy_duowebsdk = get_option('duo_use_legacy_duowebsdk', false);

    // set some session data that we can check against once postback occurs
    $_SESSION["duo_username"] = $username;
    $_SESSION["duo_timestamp"] = time();

    if (version_compare(PHP_VERSION, '7.3.0', '>=') && !$duo_use_legacy_duowebsdk) {
        try {
            $duo_client = new Client(
                $duo_ikey,
                $duo_skey,
                $duo_host,
                get_external_url().'login.php'
            );
        } catch (DuoException $e) {
            throw new ErrorException("*** Duo config error. Verify the values in duo.conf are correct ***\n" . $e->getMessage());
        }

        // Set up a few more session values
        $state = $duo_client->generateState();
        $_SESSION['duo_state'] = $state;
        $prompt_uri = $duo_client->createAuthUrl($username, $state);
        header('Location: '.$prompt_uri);
        exit(0);

    } else {
        // if we made it this far, we are good to go on loading the duo iframe
        do_page_start((array("page_title" => _("DUO Two Factor Authentication"))));
        require_once(get_component_dir_base("duo") . "/lib/Duo-Web-v2.php");
        $sig_request = Duo\Web::signRequest($duo_ikey, $duo_skey, $duo_akey, $username);
    }

    ?>
    <script type="text/javascript" src="<?php echo get_component_url_base("duo"); ?>/lib/Duo-Web-v2.min.js?<?php echo get_build_id(); ?>"></script>
    <style>
    #duo_iframe {
        width: 620px;
        height: 330px;
        background: transparent;
        margin: 100px auto;
        display: block;
        border: none;
    }
    </style>
    <iframe id="duo_iframe" data-host="<?php echo $duo_host; ?>" data-sig-request="<?php echo $sig_request; ?>"></iframe>
    <?php

    do_child_end();
    exit(0);
}


function duo_check_2fa_post($cbtype, &$cbargs)
{
    if (duo_2fa_enabled() != true) {
        $cbargs["info_messages"][0] = _("DUO two factor authentication disabled globally, skipping DUO 2FA");
        $cbargs["2fa_skip"] = 1;
        $cbargs["2fa_disabled"] = 1;
        return;
    }

    // Get version we are using (sdk v2 vs v4)
    $duo_use_legacy_duowebsdk = get_option('duo_use_legacy_duowebsdk', false);

    if (version_compare(PHP_VERSION, '7.3.0', '>=') && !$duo_use_legacy_duowebsdk) {
        $response_state = grab_request_var("state", "");
        $response_duo_code = grab_request_var("duo_code", "");
    } else {
        $sig_response = grab_request_var("sig_response");
        if (empty($sig_response)) {
            return;
        }
    }

    if (duo_2fa_good_to_go() != true) {
        return;
    }

    $duo_ikey = duo_2fa_ikey();
    $duo_skey = duo_2fa_skey();
    $duo_akey = duo_2fa_akey();
    $duo_host = duo_2fa_host();

    // do we have some set session data?
    $session_username = grab_array_var($_SESSION, "duo_username", false);
    $session_timestamp = grab_array_var($_SESSION, "duo_timestamp", 0);

    if (version_compare(PHP_VERSION, '7.3.0', '>=') && !$duo_use_legacy_duowebsdk) {
        $session_state = grab_array_var($_SESSION, "duo_state", "");
        if (empty($response_state) || empty($response_duo_code) || $session_state != $response_state) {
            return;
        }
    }

    // this wasn't a postback
    if (empty($session_username) || empty($session_timestamp)) {
        return;
    }

    // if the session is too old we just force them to go again
    if ($session_timestamp < (time() - duo_2fa_timeout())) {
        return;
    }

    // if we got a sig_response, it means the user
    // has ALREADY authenticated - so we just get the data
    // and fake a login
    if (version_compare(PHP_VERSION, '7.3.0', '>=') && !$duo_use_legacy_duowebsdk) {

        try {
            $duo_client = new Client(
                $duo_ikey,
                $duo_skey,
                $duo_host,
                get_external_url().'login.php'
            );
        } catch (DuoException $e) {
            throw new ErrorException("*** Duo config error. Verify the values in duo.conf are correct ***\n" . $e->getMessage());
        }

        try {
            $token = $duo_client->exchangeAuthorizationCodeFor2FAResult($response_duo_code, $session_username);

            // Verify we have a valid token and username matches the token provided
            if (!is_array($token) || $token['sub'] != $session_username || $token['auth_result']['result'] != 'allow') {
                return;
            } else {
                $username = $token['sub'];
            }

        } catch (DuoException $e) {
            return;
        }

    } else {
        require_once(get_component_dir_base("duo") . "/lib/Duo-Web-v2.php");
        $username = Duo\Web::verifyResponse($duo_ikey, $duo_skey, $duo_akey, $sig_response);

        // if the username doesn't match what was set in the session
        // then some trickery of some kind is happening. no bueno
        if ($username != $session_username) {;
            return;
        }
    }

    // at this point we are good to go for logging in
    // but we are essentially tricking the system
    // and bypassing the login.php page entirely

    // we have to fake a lot of stuff because of where
    // the callback is in relation to where the actual
    // session beginning starts

    $user_id = get_user_id($username);

    // this shouldn't be
    if (is_null($user_id) || $user_id <= 0) {
        return;
    }

    // most of the rest of this code
    // is taken directly from the function
    // `do_login` in login.php
    // some of it is missing, for brevity
    // and because i think other places
    // will take care of it as well
    init_session(false, true, true);
    
    $sid = user_generate_session();
    $_SESSION["session_id"] = $sid;

    change_user_attr($user_id, "login_attempts", 0);
    change_user_attr($user_id, "last_attempt", 0);

    // set session variables
    $_SESSION["user_id"] = $user_id;
    $_SESSION["username"] = strtolower($username); // <-- Username must be lowercase or Nagios Core permissions will get screwed up

    if (user_has_agreed_to_license() == false) {
        $_SESSION["agreelicense"] = 1;
    }

    get_user_meta_session_vars(true);

    $_SESSION["has_seen_login_alerts"] = false;
    
    // Set last login time
    change_user_attr(0, 'last_login', time());

    send_to_audit_log(_("User logged in successfully via DUO Two Factor Authentication"), AUDITLOGTYPE_SECURITY);

    header("Location: index.php");
    exit(0);
}

/////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////
// Configuration & Management Functions
/////////////////////////////////////////////////////////////////////////////////////////

function duo_config_func($mode, $inargs, &$outargs, &$result)
{
    switch ($mode) {
    case COMPONENT_CONFIGMODE_GETSETTINGSHTML:

        $duo_enabled = duo_2fa_enabled();
        $duo_ikey = duo_2fa_ikey();
        $duo_skey = duo_2fa_skey();
        $duo_host = duo_2fa_host();
        $duo_timeout = duo_2fa_timeout();
        $duo_use_legacy_duowebsdk = get_option('duo_use_legacy_duowebsdk', false);

        $output = '
            <div class="alert alert-info" style="margin-top: 10px;">' . _("Follow the ") . '<strong>' . _("First Steps") . '</strong>' . _(" located at ") .
            '<a href="https://duo.com/docs/duoweb" target="_blank">https://duo.com/docs/duoweb</a>. ' . 
            _("All you need to continue is an integration key, secret key, and API hostname.") . '</div>';

        if (version_compare(PHP_VERSION, '7.3.0', '<')) {
            $output .= '<div class="alert alert-danger">'._('Your PHP version on this system is below 7.3 so this Nagios XI system is only able to use the Web SDK v2 which may work but is deprecated. To use Web SDK v4 you must upgrade your PHP version to 7.3 or higher.').'</div>';
        } else {
            $output .= '<div class="alert alert-warning">
                <b>'._('READ THIS BEFORE ENABLING').'</b><br><br>'._('Your Nagios XI system must have a valid hostname with SSL and not an IP address for DUO 2FA to work. The hostname does not need to be web-accessible and your SSL certificate does not need to be valid but the hostname does need to resolve for those who log into the system. You can set this in Admin > System Settings > Internal URL. The system will use External URL instead if you have one specified. Example: https://nagiosxi.internal/nagiosxi/').'
            </div>';
        }

        $output .= '<p>' . _("You can disable (skip) two DUO authentication on a per user basis in Admin > User Management.") . '</p>';
        $output .= '

            <h5 class="ul">' . _('Duo Settings') . '</h5>
            <table class="table table-condensed table-no-border table-auto-width">

                <tr>
                    <td></td>
                    <td class="checkbox">
                        <label>
                            <input type="checkbox" class="checkbox" name="duo_enabled" ' . is_checked($duo_enabled, true) . '>
                            '._('Enable Duo Two Factor Authentication').'
                        </label>
                    </td>
                </tr>

                <tr>
                    <td class="vt">
                        <label for="duo_ikey">' . _('DUO Integration Key') . ':</label>
                    </td>
                    <td>
                        <input type="text" size="40" name="duo_ikey" id="duo_ikey" value="' . htmlentities($duo_ikey) . '" class="form-control">
                    </td>
                </tr>

                <tr>
                    <td class="vt">
                        <label for="duo_skey">' . _('DUO Secret Key') . ':</label>
                    </td>
                    <td>
                        <input type="text" size="40" name="duo_skey" id="duo_skey" value="' . htmlentities($duo_skey) . '" class="form-control">
                    </td>
                </tr>

                <tr>
                    <td class="vt">
                        <label for="duo_host">' . _('DUO API Hostname') . ':</label>
                    </td>
                    <td>
                        <input type="text" size="40" name="duo_host" id="duo_host" value="' . htmlentities($duo_host) . '" class="form-control">
                    </td>
                </tr>

                <tr><td colspan="2">&nbsp;</td><tr>

                <tr>
                    <td class="vt">
                        <label for="duo_timeout">' . _('DUO Authentication Timeout') . ':</label>
                    </td>
                    <td>
                        <input type="text" size="16" name="duo_timeout" id="duo_timeout" value="' . htmlentities($duo_timeout) . '" class="form-control">
                    </td>
                </tr>';

             if (version_compare(PHP_VERSION, '7.3.0', '>=')) {
                $output .= '
                <tr>
                    <td></td>
                    <td class="checkbox">
                        <label>
                            <input type="checkbox" class="checkbox" name="duo_use_legacy_duowebsdk" ' . is_checked($duo_use_legacy_duowebsdk, true) . '>
                            '._('Use Legacy WebSDK 2 (may not work properly)').'
                        </label>
                    </td>
                </tr>';
            }

            $output .= '</table>';
            return $output;
        break;

    case COMPONENT_CONFIGMODE_SAVESETTINGS:

        $duo_enabled = checkbox_binary(grab_array_var($inargs, "duo_enabled", false));
        $duo_use_legacy_duowebsdk = checkbox_binary(grab_array_var($inargs, "duo_use_legacy_duowebsdk", false));

        $duo_ikey = grab_array_var($inargs, "duo_ikey", "");
        $duo_skey = grab_array_var($inargs, "duo_skey", "");
        $duo_host = grab_array_var($inargs, "duo_host", "");
        $duo_timeout = grab_array_var($inargs, "duo_timeout", DEFAULT_DUO_2FA_TIMEOUT);

        set_option("duo_enabled", $duo_enabled);
        set_option("duo_ikey", $duo_ikey);
        set_option("duo_skey", $duo_skey);
        set_option("duo_host", $duo_host);
        set_option("duo_timeout", $duo_timeout);
        set_option("duo_use_legacy_duowebsdk", $duo_use_legacy_duowebsdk);

        break;

    default:
        break;
    }
}

function duo_manage_users($cbtype, $cbargs)
{
    if (duo_2fa_enabled() != true) {
        return;
    }

    switch ($cbtype) {
    case CALLBACK_USER_EDIT_HTML_AUTH_SETTINGS:

        $user_id = grab_array_var($cbargs, "user_id");
        $add = grab_array_var($cbargs, "add");

        $duo_2fa = "enabled";
        if ($add == false && $user_id > 0) {
            $duo_2fa = get_user_meta($user_id, "duo_2fa", "enabled");
        }

        echo '
            <tr>
                <td><label for="duo_2fa">' . _("DUO 2FA") . ': </label>

                    <i tabindex="1" class="fa fa-question-circle pop" data-content="' . 
                    _("DUO Two Factor Authentication is currently enabled.") . '"></i>

                </td>
                <td>

                    <select name="duo_2fa" id="duo_2fa" class="form-control">
                        <option value="enabled" ' . is_selected($duo_2fa, "enabled") . '>' . _("Enabled") . '</option>
                        <option value="skip" ' . is_selected($duo_2fa, "skip") . '>' . _("Skip") . '</option>
                    </select>

                </td>
            </tr>
            ';

        break;

    case CALLBACK_USER_EDIT_PROCESS:

        $user_id = intval(grab_array_var($cbargs, "user_id"));

        if ($user_id <= 0) {
            return;
        }

        $duo_2fa = grab_request_var("duo_2fa", "");
        if (!empty($duo_2fa)) {

            // we only care if its set specifically to 'skip'
            if ($duo_2fa != "skip") {
                delete_user_meta($user_id, "duo_2fa");
                return;
            }

            set_user_meta($user_id, "duo_2fa", "skip");
        }

        break;
    }
}

/////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////
// Helper Functions
/////////////////////////////////////////////////////////////////////////////////////////

function duo_2fa_good_to_go()
{
    $duo_ikey = duo_2fa_ikey();
    $duo_skey = duo_2fa_skey();
    $duo_host = duo_2fa_host();
    $duo_akey = duo_2fa_akey();

    /* we only return true if EVERY condition is satisfied */
    if (duo_2fa_enabled()
        && (!empty($duo_ikey)    && strlen($duo_ikey) == 20)
        && (!empty($duo_skey)    && strlen($duo_skey) == 40)
        && (!empty($duo_host)    && (strpos($duo_host, "duosecurity.com") !== false || strpos($duo_host, "duofederal.com") !== false))
        && (!empty($duo_akey))) {

        return true;
    }
    
    return false;
}

function duo_2fa_enabled()
{
    $duo_enabled = get_option("duo_enabled", false);
    if (!empty($duo_enabled) && $duo_enabled == true) {
        return true;
    }

    return false;
}

function duo_2fa_reset_akey()
{
    $duo_akey = random_string(64);
    set_option("duo_akey", $duo_akey);
    return $duo_akey;
}

function duo_2fa_ikey()
{
    return get_option("duo_ikey", false);
}

function duo_2fa_skey()
{
    return get_option("duo_skey", false);
}

function duo_2fa_host()
{
    return get_option("duo_host", false);
}

function duo_2fa_akey()
{
    $duo_2fa_akey = get_option("duo_akey", false);

    if (empty($duo_2fa_akey) || strlen($duo_2fa_akey) < 40) {
        $duo_2fa_akey = duo_2fa_reset_akey();
    }
    
    return $duo_2fa_akey;
}

function duo_2fa_timeout()
{
    if (!defined("DEFAULT_DUO_2FA_TIMEOUT")) {
        define("DEFAULT_DUO_2FA_TIMEOUT", 60);
    }

    $duo_timeout = intval(get_option("duo_timeout", DEFAULT_DUO_2FA_TIMEOUT));
    if ($duo_timeout > 0) {
        return $duo_timeout;
    }

    return 60;
}

function duo_component_checkversion()
{
    if (function_exists('get_product_release')) {
        if (get_product_release() >= 5500) {
            return true;
        }
    }

    return false;
}
