<?php
//
// Better Bullet Map (BBMap) Component
// Copyright (c) 2010-2024 Nagios Enterprises, LLC. All rights reserved.
//

require_once(dirname(__FILE__) . '/../componenthelper.inc.php');
include_once(dirname(__FILE__) . '/dashlet.inc.php');

$bbmap_component_name = "bbmap";
bbmap_component_init();

////////////////////////////////////////////////////////////////////////
// COMPONENT INIT FUNCTIONS
////////////////////////////////////////////////////////////////////////

function bbmap_component_init()
{
    global $bbmap_component_name;

    $versionok = bbmap_component_checkversion();

    $desc = "";

    if (!$versionok) {
        $desc = "<b>Error: " . _("This component requires Nagios XI 2024R1.1 or later.") . "</b>";
    }

    $args = array(
        COMPONENT_NAME => $bbmap_component_name,
        COMPONENT_VERSION => '2.0.2',
        COMPONENT_AUTHOR => "Nagios Enterprises, LLC",
        COMPONENT_DESCRIPTION => _("Displays a grid-like view of host and service status. ") . $desc,
        COMPONENT_TITLE => "BBMap"
    );

    register_component($bbmap_component_name, $args);

    if ($versionok) {
        register_callback(CALLBACK_MENUS_INITIALIZED, 'bbmap_component_addmenu');
    }

    $args = array();
    $args[DASHLET_NAME] = "bbmap";
    $args[DASHLET_TITLE] = "BBMap";
    $args[DASHLET_FUNCTION] = "bbmap_dashlet_func";
    $args[DASHLET_DESCRIPTION] = _("Displays a grid-like view of host and service status.");
    $args[DASHLET_FILTER_GROUPS] = array("status");
    $args[DASHLET_WIDTH] = "1000";
    $args[DASHLET_HEIGHT] = "520";
    $args[DASHLET_INBOARD_CLASS] = "bbmap_map_inboard";
    $args[DASHLET_OUTBOARD_CLASS] = "bbmap_map_outboard";
    $args[DASHLET_CLASS] = "bbmap_map";
    $args[DASHLET_AUTHOR] = "Nagios Enterprises, LLC";
    $args[DASHLET_COPYRIGHT] = "Dashlet Copyright &copy; 2011-2024 Nagios Enterprises. All rights reserved.";
    $args[DASHLET_HOMEPAGE] = "http://www.nagios.com";
    $args[DASHLET_SHOWASAVAILABLE] = true;
    $args[DASHLET_ISCUSTOM] = false;
    register_dashlet($args[DASHLET_NAME], $args);
}


///////////////////////////////////////////////////////////////////////////////////////////
// MISC FUNCTIONS
///////////////////////////////////////////////////////////////////////////////////////////

function bbmap_component_checkversion()
{
    if (!function_exists('get_product_release')) {
        return false;
    }

    if (get_product_release() < 60100) {
        return false;
    }

    return true;
}

function bbmap_component_addmenu($arg = null)
{
    global $bbmap_component_name;

    $mi = find_menu_item(MENU_HOME, "menu-home-section-maps", "id");

    if ($mi == null) {
        return;
    }

    $order = grab_array_var($mi, "order", "");

    if ($order == "") {
        return;
    }

    $neworder = $order + 1;

    add_menu_item(MENU_HOME, array(
        "type" => "link",
        "title" => "BBmap",
        "id" => "menu-home-bbmap",
        "order" => $neworder,
        "opts" => array(
            "href" => get_base_url() . 'includes/components/bbmap/',
            "icon" => "fa-th-large"
        )
    ));
}


///////////////////////////////////////////////////////////////////////////////////////////
// AJAX FUNCTIONS
///////////////////////////////////////////////////////////////////////////////////////////

function xicore_ajax_get_bbmap_dashlet_html($args = null)
{
    $host = grab_array_var($args, "host", "");
    $hostgroup = grab_array_var($args, "hostgroup", "");
    $servicegroup = grab_array_var($args, "servicegroup", "");
    $hardStates = grab_array_var($args, "hardStates", "");
    $id = uniqid();

    // Special "all" stuff
    if ($hostgroup == "all") {
        $hostgroup = "";
    }

    if ($servicegroup == "all") {
        $servicegroup = "";
    }

    if ($host == "all") {
        $host = "";
    }

    // Can do hostgroup OR servicegroup OR host
    if ($hostgroup != "") {
        $servicegroup = "";
        $host = "";
    } else if ($servicegroup != "") {
        $host = "";
    }

    // Limit hosts by hostgroup or host
    $host_ids = array();
    $host_ids_str = "";

    // Limit by hostgroup
    if ($hostgroup != "") {
        $host_ids = get_hostgroup_member_ids($hostgroup);

    // Limit by host
    } else if ($host != "") {
        $host_ids[] = get_host_id($host);
    }

    $y = 0;

    foreach ($host_ids as $hid) {
        if ($y > 0)
            $host_ids_str .= ",";
        $host_ids_str .= $hid;
        $y++;
    }

    // imit service by servicegroup
    $service_ids = array();
    $service_ids_str = "";

    if ($servicegroup != "") {
        $service_ids = get_servicegroup_member_ids($servicegroup);
    }

    $y = 0;

    foreach ($service_ids as $sid) {
        if ($y > 0) {
            $service_ids_str .= ",";
        }

        $service_ids_str .= $sid;
        $y++;
    }

    // Get service status from backend
    $backendargs = array();
    $backendargs["cmd"] = "getservicestatus";
    $backendargs["limitrecords"] = false; // don't limit records
    $backendargs["combinedhost"] = true; // get host status too
    $backendargs["brevity"] = 1; // we don't everything

    // Host id limiters
    if ($host_ids_str != "") {
        $backendargs["host_id"] = "in:" . $host_ids_str;
    }

    // Service id limiters
    if ($service_ids_str != "") {
        $backendargs["service_id"] = "in:" . $service_ids_str;
    }

    // Order by host name, service description.
    $backendargs["orderby"] = "host_name:a,service_description:a";

    $xml = get_xml_service_status($backendargs);

    // Get service names
    $servicenames = array();

    foreach ($xml->servicestatus as $ss) {
        $sn = strval($ss->name);

        if (!in_array($sn, $servicenames)) {
            $servicenames[] = $sn;
        }
    }

    sort($servicenames);
    $cols = count($servicenames);

    $pretitle = "";

    if ($host != "") {
        $pretitle = _("Host") . " '$host' ";
    } else if ($hostgroup != "") {
        $pretitle = _("Hostgroup") . " '$hostgroup' ";
    } else if ($servicegroup != "") {
        $pretitle = _("Servicegroup") . " '$servicegroup' ";
    }

    /*
     * Using DataTables Javascript module to handle the fancy stuff.
     * - Fixed header
     * - Fixed left column
     * - Filter
     * - Sort by column
     * - Stripe
     * - Hover
     *
     * There are many other features available in this module, like pagination,
     * limit number of entries, AJAX, etc. 
     *
     * https://datatables.net/
     *
     * NOTE: Some features may require more modules to be included in the .min.* files.
     */
?>
    <script type="text/javascript">
        // Create the DataTable object.
        var bbmapTable = new DataTable('#bbmap-<?php echo $id; ?>', {
            fixedColumns: {
                left: 1
            },
            paging: false,
            scrollCollapse: false,
            scrollY: '100%',
            //scrollY: '200px', // Debugging vertical scrollbar
            scrollX: true,
            // Required: "Attaches" the DataTables header to the real table.
            //           Otherwise on horizontal scroll, the body of the table
            //           scrolls too far to the left.
            "initComplete": function(settings, json) {
                //alert( 'DataTables has finished its initialisation.' );

                $('.header_row:last-child')
                    .css({
                        height: "1px"
                    })
            }
        });

    </script>

    <div class="infotable_title"><?= $pretitle ?><?= _('Status Grid') ?></div>
<?php
    $tableclass = "display nowrap table table-condensed table-striped table-bordered table-hover table-auto-width";
    if (is_neptune()) {
        $tableclass = "bbmap-neptune display nowrap table-striped table-hover table-auto-width";
    }
?>
<?php
$theme = get_theme();
$bgcolor   = 'inherit';
$textcolor = 'inherit';
switch ($theme) {

    case 'xi5dark':
        $bgcolor   = '#111';
        $textcolor = '#fff';
        break;

    case 'neptune':
        $bgcolor   = '#131B25';
        // $bgcolor   = '#0e141b';
        $textcolor = '#fff';
        break;

    case 'neptunelight':
        $bgcolor   = '#fff';
        $textcolor = '#000';
        break;

    case 'xi5':
    default:
        $bgcolor   = '#fff';
        $textcolor = '#000';
        break;
}


?>

<script>
    // Get PHP theme variable into JavaScript
    var currentTheme = '<?php echo $theme; ?>';
    var tableElement = document.getElementById('bbmap-<?php echo $id; ?>');
    // Only proceed if we're in a Neptune theme
    if (currentTheme === 'neptune' || currentTheme === 'neptunelight') {
       
        var background_color;
        
        // Set background color based on theme
        if (currentTheme === 'neptune') {
            background_color = '#0e141b';
        } else if (currentTheme === 'neptunelight') {
            background_color = '#f6f6f6';
        }
        var isInboard = tableElement.closest('.bbmap_map_inboard') !== null;
        var inboard = tableElement.closest('.bbmap_map_inboard');
        var infotable = inboard.querySelectorAll('th.infotable_hosts');
        // Function to update background colors
        function updateBackgrounds() {
            

            if (isInboard) {
                inboard.style.setProperty('background-color', background_color);
                infotable[0].style.setProperty('background-color', background_color);
            }
        }
        // Initial attempt
        updateBackgrounds();

        // Retry after a short delay to ensure DOM is fully loaded
        setTimeout(updateBackgrounds, 100);
        
        // Final attempt after DataTables initialization
        bbmapTable.on('init', function() {
            setTimeout(updateBackgrounds, 100);
        });
    
}
    else {
        var isInboard = tableElement.closest('.bbmap_map_inboard') !== null;
        if (isInboard) {
        // Get background color with timeout to ensure elements are loaded
        function updateBackgrounds() {
            var inboard = tableElement.closest('.bbmap_map_inboard');
            if (currentTheme === 'xi5dark') {
                background_color = '#111';
            } else {
                background_color = '#fff';
            }
            inboard.style.setProperty('background-color', background_color);
            }
        // Initial attempt
        updateBackgrounds();

        // Retry after a short delay to ensure DOM is fully loaded
        setTimeout(updateBackgrounds, 100);
        
        // Final attempt after DataTables initialization
        bbmapTable.on('init', function() {
            setTimeout(updateBackgrounds, 100);
        });
    }
}

</script>
        <table id="bbmap-<?php echo $id; ?>" class="<?php echo $tableclass ?>">
            <thead>
                <tr class="header_row">
                <th class="infotable_hosts" style="position: sticky; left: 0; background-color: <?php echo $bgcolor; ?>; color: <?php echo $textcolor; ?>; z-index: 10; white-space: nowrap;"></th>
<?php
    if ($cols) {
        for ($x = 0; $x < $cols; $x++) {
?>
        <th class="rotate"><?= bbmap_get_service_title($servicenames[$x]) ?></th>
<?php
        }
    }
?>
                </tr>
            </thead>
            <tbody>
<?php
    if ($xml) {
        $base_url = get_base_url();
        $status_url = $base_url . "includes/components/xicore/status.php";
        $image_url = $base_url . "includes/components/bbmap/images/";

        $lasthost = "";
        $coloutput = array();

        if (count($xml->servicestatus) == 0) {
?>
            <tr><th>No Hosts</th><td>No Services</td></tr>
<?php
        }

//for ($idx = 0; $idx < 10; $idx++) {   // For debugging scroll Y.
        foreach ($xml->servicestatus as $ss) {

            $thishost = strval($ss->host_name);

            if ($thishost != $lasthost) {

                // End last row
                if ($lasthost != "") {

                    for ($x = 0; $x < $cols; $x++) {
                        $co = "";

                        if (array_key_exists($x, $coloutput)) {
                            $co = $coloutput[$x];
                        }
?>
                        <td><?= $co ?></td>
<?php
                    }
?>
                    </tr>

<?php
                    // Clear columnn output
                    unset($coloutput);
                    $coloutput = array();
                }

                $hosttext = strval($ss->host_status_text);
                $hoststate = intval($ss->host_current_state);
                $hosthbc = intval($ss->host_has_been_checked);
                $hoststatetype = intval($ss->host_state_type);
                $hostlasthardstate = intval($ss->host_last_hard_state);

                $statetext = "";
                $img = "";

                $relevanthoststate = $hoststate;
                if ($hardStates && $hoststatetype == 0) {
                    $relevanthoststate = $hostlasthardstate;
                }

                switch ($relevanthoststate) {
                    case 0:
                        if ($hosthbc == 1) {
                            $imgtitle = _("Host Up") . ": " . encode_form_val($hosttext);
                            $img = '<span class="material-symbols-outlined md-18 md-ok md-400 md-middle md-button tt-bind" title="' . $imgtitle . '">check_circle</span>';
                        } else {
                            $imgtitle = _("Host Pending") . ": " . encode_form_val($hosttext);
                            $img = '<span class="material-symbols-outlined md-18 md-pending md-400 md-middle md-button tt-bind" title="' . $imgtitle . '">schedule</span>';
                        }
                        break;
                    case 1:
                        $imgtitle = _("Host Down") . ": " . encode_form_val($hosttext);
                        $img = '<span class="material-symbols-outlined md-18 md-critical md-400 md-middle md-button tt-bind" title="' . $imgtitle . '">cancel</span>';
                        break;
                    case 2:
                        $imgtitle = _("Host Unreachable") . ": " . encode_form_val($hosttext);
                        $img = '<span class="material-symbols-outlined md-18 md-unknown md-400 md-middle md-button tt-bind" title="' . $imgtitle . '">help</span>';
                        break;
                    default:
                        $img = "";
                        break;
                }

                // Check for acknowledgements, scheduled downtime.
                if ($hoststate != 0) {
                    $ack = intval($ss->host_problem_acknowledged);
                    $sdd = intval($ss->host_scheduled_downtime_depth);                    
                    if ($ack == 1) {
                        $imgtitle = _("Host Acknowledged") . ": " . encode_form_val($hosttext);
                        $img = '<span class="material-symbols-outlined md-18 md-pending md-400 md-middle md-button tt-bind" title="' . $imgtitle . '">build_circle</span>';
                    } else if ($sdd > 0) {
                        $imgtitle = _("Host Scheduled Downtime") . ": " . encode_form_val($hosttext);
                        $img = '<span class="material-symbols-outlined md-18 md-pending md-400 md-middle md-button tt-bind" title="' . $imgtitle . '">build_circle</span>';
                    }
                }

                // Start new row
?>

                <tr><th class="infotable_hosts" style="position: sticky; left: 0; background-color: <?php echo $bgcolor; ?>; color: <?php echo $textcolor; ?>; z-index: 10; white-space: nowrap;">
                <a style="font-weight: normal;" href="<?= $status_url ?>?show=hostdetail&host=<?= urlencode($thishost) ?>" target="_blank"><?php echo "$img" ?><span class="tt-bind" title="<?= $thishost ?>"><?= $thishost ?></span></a>
<?php
            }

            // Remember last host.
            $lasthost = $thishost;

            // Service status
            $service = strval($ss->name);

            $servicetext = strval($ss->status_text);
            $servicestate = intval($ss->current_state);
            $servicehbc = intval($ss->has_been_checked);
            $servicestatetype = intval($ss->state_type);
            $servicelasthardstate = intval($ss->last_hard_state);

            $statetext = "";

            $relevantservicestate = $servicestate;
            if ($hardStates && $servicestatetype == 0) {
                $relevantservicestate = $servicelasthardstate;
            }

            switch ($relevantservicestate) {
                case 0:
                    if ($servicehbc == 1) {
                        $imgtitle = "" . htmlentities($service, ENT_COMPAT, 'UTF-8') . " Ok: " . htmlentities($servicetext, ENT_COMPAT, 'UTF-8');
                        $img = '<span class="material-symbols-outlined md-18 md-ok md-400 md-middle md-button tt-bind" title="' . $imgtitle . '">check_circle</span>';
                    } else {
                        $imgtitle = "" . htmlentities($service, ENT_COMPAT, 'UTF-8') . " Pending: " . htmlentities($servicetext, ENT_COMPAT, 'UTF-8');
                        $img = '<span class="material-symbols-outlined md-18 md-pending md-400 md-middle md-button tt-bind" title="' . $imgtitle . '">build_circle</span>';
                    }
                    break;
                case 1:
                    $imgtitle = "" . htmlentities($service, ENT_COMPAT, 'UTF-8') . " Warning: " . htmlentities($servicetext, ENT_COMPAT, 'UTF-8');
                        $img = '<span class="material-symbols-outlined md-18 md-warning md-400 md-middle md-button tt-bind" title="' . $imgtitle . '">warning</span>';
                    break;
                case 2:
                    $imgtitle = "" . htmlentities($service, ENT_COMPAT, 'UTF-8') . " Critical: " . htmlentities($servicetext, ENT_COMPAT, 'UTF-8');
                    $img = '<span class="material-symbols-outlined md-18 md-critical md-400 md-middle md-button tt-bind" title="' . $imgtitle . '">error</span>';
                    break;
                case 3:
                    $imgtitle = "" . htmlentities($service, ENT_COMPAT, 'UTF-8') . " Unknown: " . htmlentities($servicetext, ENT_COMPAT, 'UTF-8');
                    $img = '<span class="material-symbols-outlined md-18 md-unknown md-400 md-middle md-button tt-bind" title="' . $imgtitle . '">help</span>';
                    break;
                default:
                    $img = "";
                    break;
            }

            // Check for acknowledgements, scheduled downtime.
            if ($servicestate != 0) {
                $ack = intval($ss->problem_acknowledged);
                $sdd = intval($ss->scheduled_downtime_depth);

                if ($ack == 1) {
                    $imgtitle = "" . htmlentities($service, ENT_COMPAT, 'UTF-8') . " Acknowledged: " . htmlentities($servicetext, ENT_COMPAT, 'UTF-8');
                    $img = '<span class="material-symbols-outlined md-18 md-pending md-400 md-middle md-button tt-bind" title="' . $imgtitle . '">build_circle</span>';
                } else if ($sdd > 0) {
                    $imgtitle = "" . htmlentities($service, ENT_COMPAT, 'UTF-8') . " Scheduled Downtime: " . htmlentities($servicetext, ENT_COMPAT, 'UTF-8');
                    $img = '<span class="material-symbols-outlined md-18 md-pending md-400 md-middle md-button tt-bind" title="' . $imgtitle . '">build_circle</span>';
                }
            }

            // Check for host acknowledgements, scheduled downtime.
            if ($hoststate != 0) {
                $ack = intval($ss->host_problem_acknowledged);
                $sdd = intval($ss->host_scheduled_downtime_depth);

                if ($ack == 1) {
                    $imgtitle = "" . htmlentities($service, ENT_COMPAT, 'UTF-8') . " Host Problem Acknowledged: " . htmlentities($servicetext, ENT_COMPAT, 'UTF-8');
                    $img = '<span class="material-symbols-outlined md-18 md-pending md-400 md-middle md-button tt-bind" title="' . $imgtitle . '">build_circle</span>';
                } else if ($sdd > 0) {
                    $imgtitle = "" . htmlentities($service, ENT_COMPAT, 'UTF-8') . " Host in Scheduled Downtime: " . htmlentities($servicetext, ENT_COMPAT, 'UTF-8');
                    $img = '<span class="material-symbols-outlined md-18 md-pending md-400 md-middle md-button tt-bind" title="' . $imgtitle . '">build_circle</span>';
                }
            }

            // What column does this go in?
            $col = array_search($service, $servicenames);

            $coloutput[$col] = '<a href="' . $status_url . '?show=servicedetail&host=' . urlencode($thishost) . '&service=' . urlencode($service) . '" target="_blank">' . $img . '</a>';
        }
//} // For debugging scroll Y.

        // End last row.
        if ($lasthost != "") {
            for ($x = 0; $x < $cols; $x++) {
                $co = "";

                if (array_key_exists($x, $coloutput)) {
                    $co = $coloutput[$x];
                }
?>
                <td><?= $co ?></td>
<?php
            }
?>
            </tr>
<?php
            // Clear columnn output.
            unset($coloutput);
            $coloutput = array();
        }
    } else {
?>
        <tr><td  colspan="<?= ($cols + 1) ?>"><?= _('No data to display') ?></td></tr>
<?php
    }
?>
    </tbody>
    </table></div></div>

    <div class="ajax_date"><?= _('Last Updated') ?>: <?= get_datetime_string(time()) ?></div>
<?php
    return;
}

function bbmap_get_service_title($s)
{
    $title = "<div class='rotate'><span class='tt-bind' title='$s'>$s</span></div>";
    return $title;
}
