<?php
//
// Windows SNMP Config Wizard
// Copyright (c) 2011-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

windowssnmp_configwizard_init();

function windowssnmp_configwizard_init()
{
    $name = "windowssnmp";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.1.0",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor a Microsoft&reg; Windows workstation or server using SNMP."),
        CONFIGWIZARD_DISPLAYTITLE => _("Windows SNMP"),
        CONFIGWIZARD_FUNCTION => "windowssnmp_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "win_server.png",
        CONFIGWIZARD_FILTER_GROUPS => array('windows'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );
    register_configwizard($name, $args);
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function windowssnmp_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    global $cfg;

    $wizard_name = "windowssnmp";

    // Initialize return code and output
    $result = 0;
    $output = "";
    $snmptimeout = 10000000; //10 sec.

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            $address = grab_array_var($inargs, "ip_address", "");
            $port = grab_array_var($inargs, "port", "161");
            $osversion = grab_array_var($inargs, "osversion", "");
            $snmpcommunity = grab_array_var($inargs, "snmpcommunity", "public");
            $snmpversion = grab_array_var($inargs, "snmpversion", "2c");
            $snmpopts = "";
            $snmpopts_serial = grab_array_var($inargs, "snmpopts_serial", "");

            if ($snmpopts_serial != "") {
                $snmpopts = json_decode(base64_decode($snmpopts_serial), true);
            }

            if (!is_array($snmpopts)) {
                $snmpopts_default = array(
                    "v3_security_level" => "authPriv",
                    "v3_username" => "",
                    "v3_auth_password" => "",
                    "v3_privacy_password" => "",
                    "v3_auth_proto" => "",
                    "v3_priv_proto" => ""
                );

                $snmpopts = grab_array_var($inargs, "snmpopts", $snmpopts_default);
            }

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address", "");
            $port = grab_array_var($inargs, "port", 161);
            $osversion = grab_array_var($inargs, "osversion", "");
            $snmpcommunity = grab_array_var($inargs, "snmpcommunity", "");
            $snmpversion = grab_array_var($inargs, "snmpversion", "");

            $address = nagiosccm_replace_user_macros($address);
            $snmpcommunity = nagiosccm_replace_user_macros($snmpcommunity);

            $snmpopts_serial = grab_array_var($inargs, "snmpopts_serial", "");

            if ($snmpopts_serial == "") {
                $snmpopts = grab_array_var($inargs, "snmpopts");
            } else {
                $snmpopts = json_decode(base64_decode($snmpopts_serial), true);
            }

            // Add port to address if we need to
            $runaddress = $address;

            if (!empty($port) && $port != 161) {
                $runaddress = $address.":".$port;
            }

            // Check for errors
            $errors = [];

            if (have_value($address) == false) {
                $errors[] = _("No address specified.");
            } else if (!valid_ip($address)) {
                $errors[] = _("Invalid IP address.");
            }
            if (have_value($osversion) == false) {
                $errors[] = _("No operating system specified.");
            }
            if (have_value($snmpcommunity) == false && $snmpversion != "3") {
                $errors[] = _("No SNMP community specified.");
            }

            $test_timeout = 5000000;
            $test_oid = "iso.3.6.1.2.1.1.1.0"; // Simple, generic OID that should be very easy to responde to

            if ($snmpversion == "3") {
                $securitylevel = grab_array_var($snmpopts, "v3_security_level", "");
                $username = grab_array_var($snmpopts, "v3_username", "");
                $authproto = grab_array_var($snmpopts, "v3_auth_proto", "");
                $privproto = grab_array_var($snmpopts, "v3_priv_proto", "");
                $authpassword = grab_array_var($snmpopts, "v3_auth_password", "");
                $privacypassword = grab_array_var($snmpopts, "v3_privacy_password", "");

                // Run through user macros to account for illegal characters
                $username_replaced = nagiosccm_replace_user_macros($username);
                $authpassword_replaced = nagiosccm_replace_user_macros($authpassword);
                $privacypassword_replaced = nagiosccm_replace_user_macros($privacypassword);

                if (!$username ) {
                    $errors[] = "Username required for v3";
                }

                if ($securitylevel != 'noAuthNoPriv' && ! $authpassword) {
                    $errors[] = "Authentication password required for v3 authNoPriv and authPriv ";
                }

                if ($securitylevel == 'authPriv'  && ! $privacypassword) {
                    $errors[] = "Privacy password required for v3 authPriv ";
                }
                $test_reponse = snmp3_real_walk($runaddress, $username_replaced, $securitylevel, $authproto, $authpassword_replaced, $privproto, $privacypassword_replaced, $test_oid, $test_timeout);

            } else if ($snmpversion == "2c") {
                $test_reponse = snmp2_real_walk($runaddress, $snmpcommunity, $test_oid, $test_timeout);

            } else {
                $test_reponse = snmprealwalk($runaddress, $snmpcommunity, $test_oid, $test_timeout);
            }

            if (!$test_reponse) {
                $errors[] = "This server does not respond to SNMP requests (not snmp, or possibly wrong auth).";
            }


            if (count($errors)) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errors;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $port = intval(grab_array_var($inargs, "port", 161));
            $hostname = grab_array_var($inargs, "hostname", @gethostbyaddr($address));
            $osversion = grab_array_var($inargs, "osversion", "");
            $snmpcommunity = grab_array_var($inargs, "snmpcommunity", "");
            // Run snmpcommunity through user macros to account for illegal characters
            $snmpcommunity_replaced = nagiosccm_replace_user_macros($snmpcommunity);
            $snmpversion = grab_array_var($inargs, "snmpversion", "");

            $snmpopts_serial = grab_array_var($inargs, "snmpopts_serial", "");

            if ($snmpopts_serial == "") {
                $snmpopts = grab_array_var($inargs, "snmpopts");
            } else {
                $snmpopts = json_decode(base64_decode($snmpopts_serial), true);
            }

            // Add port to address if we need to
            $runaddress = $address;

            if (!empty($port) && $port != 161) {
                $runaddress = $address.":".$port;
            }

            $walksuccess = 1;
            $disk = array();
            $w_services = array();
            $process = array();

            // Populate disks that were walked then offer dropdowns A-Z for new fields
            $disk_oid = "HOST-RESOURCES-MIB::hrStorageDescr";
            $w_service_oid = "SNMPv2-SMI::enterprises.77.1.2.3.1.1";
            $process_oid = "HOST-RESOURCES-MIB::hrSWRunName";

            ////////////////////////////////////////////////////////////////////////
            // Walk the process and disk OIDs to display on stage 2 select inputs //

            if ($snmpversion == "3") {
                $securitylevel = grab_array_var($snmpopts, "v3_security_level", "");
                $username = grab_array_var($snmpopts, "v3_username", "");
                $authproto = grab_array_var($snmpopts, "v3_auth_proto", "");
                $privproto = grab_array_var($snmpopts, "v3_priv_proto", "");
                $authpassword = grab_array_var($snmpopts, "v3_auth_password", "");
                $privacypassword = grab_array_var($snmpopts, "v3_privacy_password", "");
                // Run through user macros to account for illegal characters
                $username_replaced = nagiosccm_replace_user_macros($username);
                $authpassword_replaced = nagiosccm_replace_user_macros($authpassword);
                $privacypassword_replaced = nagiosccm_replace_user_macros($privacypassword);

                $disk_table = snmp3_real_walk($runaddress, $username_replaced, $securitylevel, $authproto, $authpassword_replaced, $privproto, $privacypassword_replaced, $disk_oid, $snmptimeout);
                $w_service_table = snmp3_real_walk($runaddress, $username_replaced, $securitylevel, $authproto, $authpassword_replaced, $privproto, $privacypassword_replaced, $w_service_oid, $snmptimeout);
                $process_name_table = snmp3_real_walk($runaddress, $username_replaced, $securitylevel, $authproto, $authpassword_replaced, $privproto, $privacypassword_replaced, $process_oid, $snmptimeout);

            } else if ($snmpversion == "2c") {
                $disk_table = snmp2_real_walk($runaddress, $snmpcommunity_replaced, $disk_oid, $snmptimeout);
                $w_service_table = snmp2_real_walk($runaddress, $snmpcommunity_replaced, $w_service_oid, $snmptimeout);
                $process_name_table = snmp2_real_walk($runaddress, $snmpcommunity_replaced, $process_oid, $snmptimeout);

            } else {
                $disk_table = snmprealwalk($runaddress, $snmpcommunity_replaced, $disk_oid, $snmptimeout);
                $w_service_table = snmprealwalk($runaddress, $snmpcommunity_replaced, $w_service_oid, $snmptimeout);
                $process_name_table = snmprealwalk($runaddress, $snmpcommunity_replaced, $process_oid, $snmptimeout);
            }

            // If any walks fail, let user know
            if ($disk_table == false || $w_service_table == false || $process_name_table == false) {
                $walksuccess = 0;
            }

            if (!empty($disk_table)) {
                // disks
                foreach ($disk_table as $key => $val) {
                    preg_match("/(\w):\\\\/", $val, $name);

                    // Remove any without label/serial numbers (virtual, peripheral drives, etc.)
                    $matchCount = preg_match("/Label/", $val, $dcheck);

                    if ($matchCount == 0 || $dcheck[0] == "") {
                        continue;
                    }

                    if (isset($name[1])) {
                        $name = $name[1];
                        // use drive letter as name
                        array_push($disk, $name);
                    }
                }

                // trim repeated process names
                $disk = array_unique($disk);
            }

            if (!empty($process_name_table)) {
                // processes
                foreach ($process_name_table as $key => $val) {
                    preg_match('/"([^"]+)"/', $val, $name);
                    $name = $name[1];

                    // create array of names
                    array_push($process, '"' . $name . '"');
                }
                // trim repeated process names
                $process = array_unique($process);
                // get count
                $process_count = count($process);
                // list of processes
                $process = implode(",", $process);
            }

            // services
            if (!empty($w_service_table)) {
                foreach ($w_service_table as $key => $val) {
                    preg_match('/"([^"]+)"/', $val, $name);
                    $name = $name[1];

                    // create array of names
                    array_push($w_services, '"' .  $name . '"');
                }

                // trim repeated service names
                $w_services = array_unique($w_services);
                // get count
                $service_count = count($w_services);
                // list of services
                $w_services = implode(",", $w_services);
            }
            // end of walk section //
            /////////////////////////

            $services = "";
            $services_serial = grab_array_var($inargs, "services_serial", "");

            if ($services_serial != "") {
                $services = json_decode(base64_decode($services_serial), true);
            }

            if (!is_array($services)) {
                $services_default = array(
                    "ping" => 1,
                    "cpu" => 1,
                    "memory" => 1,
                    "pagefile" => 1,
                    "disk" => 1,
                );

                $services = grab_array_var($inargs, "services", $services_default);
            }

            $serviceargs = "";
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            if ($serviceargs_serial != "") {
                $serviceargs = json_decode(base64_decode($serviceargs_serial), true);
            }

            if (!is_array($serviceargs)) {
                $serviceargs_default = array(
                    "cpu_warning" => 80,
                    "cpu_critical" => 90,
                    "memory_warning" => 80,
                    "memory_critical" => 90,
                    "pagefile_warning" => 5,
                    "pagefile_critical" => 10,
                );

                for ($x = 0; $x < count($disk); $x++) {
                    if (!empty($disk)) { // we have snmpwalk data
                        $serviceargs_default["disk_warning"][$x] = 80;
                        $serviceargs_default["disk_critical"][$x] = 95;
                        $serviceargs_default["disk"][$x] = "";
                    } else { // we don't have snmpwalk data
                        $serviceargs_default["disk_warning"][$x] = 80;
                        $serviceargs_default["disk_critical"][$x] = 95;
                        $serviceargs_default["disk"][$x] = ($x == 0) ? "C" : "";
                    }
                }

                // Set scanned drives as default select option - after they are created
                if (!empty($disk)) {
                    $x = 0;

                    foreach ($disk as $key => $val) {
                        $serviceargs_default["disk"][$x] = ($x == $key) ? $val : "";
                        $x++;
                    }
                }

                for ($x = 0; $x < 4; $x++) {
                    if (!empty($process)) { // we have snmpwalk data
                        $serviceargs_default['processstate'][$x]['process'] = '';
                        $serviceargs_default['processstate'][$x]['name'] = '';
                        $services["processstate"][$x] = "";
                    } else { // we don't have snmpwalk data
                        if ($x == 0) {
                            $serviceargs_default['processstate'][$x]['process'] = 'explorer.exe';
                            $serviceargs_default['processstate'][$x]['name'] = 'Explorer';
                            $services["processstate"][$x] = ""; // defaults for checkboxes, enter on to be checked by default
                        } else {
                            $serviceargs_default['processstate'][$x]['process'] = '';
                            $serviceargs_default['processstate'][$x]['name'] = '';
                            $services["processstate"][$x] = ""; // defaults for checkboxes, enter on to be checked by default
                        }
                    }
                }

                for ($x = 0; $x < 4; $x++) {
                    if (!empty($w_services)) { // we have snmpwalk data
                        $serviceargs_default['servicestate'][$x]['service'] = "";
                        $serviceargs_default['servicestate'][$x]['name'] = "";
                        $services["servicestate"][$x] = "";
                    } else {
                        if ($x == 0) { // we don't have snmpwalk data
                            $serviceargs_default['servicestate'][$x]['service'] = "World Wide Web Publishing";
                            $serviceargs_default['servicestate'][$x]['name'] = "IIS Web Server";
                            $services["servicestate"][$x] = ""; // defaults for checkboxes, enter on to be checked by default
                        } elseif ($x == 1) {
                            $serviceargs_default['servicestate'][$x]['service'] = "Task Scheduler";
                            $serviceargs_default['servicestate'][$x]['name'] = "Task Scheduler";
                            $services["servicestate"][$x] = ""; // defaults for checkboxes, enter on to be checked by default
                        } elseif ($x == 2) {
                            $serviceargs_default['servicestate'][$x]['service'] = "Terminal Services";
                            $serviceargs_default['servicestate'][$x]['name'] = "Terminal Services";
                            $services["servicestate"][$x] = ""; // defaults for checkboxes, enter on to be checked by default
                        } else {
                            $serviceargs_default['servicestate'][$x]['service'] = "";
                            $serviceargs_default['servicestate'][$x]['name'] = "";
                            $services["servicestate"][$x] = ""; // defaults for checkboxes, enter on to be checked by default
                        }
                    }
                }

                $serviceargs = grab_array_var($inargs, "serviceargs", $serviceargs_default);
            }

            // List of disks
            $disk = implode(",", $disk);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $port = intval(grab_array_var($inargs, "port", 161));
            $hostname = grab_array_var($inargs, "hostname");
            $osversion = grab_array_var($inargs, "osversion", "");
            $snmpcommunity = grab_array_var($inargs, "snmpcommunity", "");
            $snmpversion = grab_array_var($inargs, "snmpversion", "");

            // Check for errors
            $errors = [];

            if (is_valid_host_name($hostname) == false) {
                $errors[] = "Invalid host name.";
            }

            if (count($errors)) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errors;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $port = intval(grab_array_var($inargs, "port", 161));
            $hostname = grab_array_var($inargs, "hostname");
            $osversion = grab_array_var($inargs, "osversion", "");
            $snmpcommunity = grab_array_var($inargs, "snmpcommunity", "");
            $snmpversion = grab_array_var($inargs, "snmpversion", "");
            $services = grab_array_var($inargs, "services");
            $serviceargs = grab_array_var($inargs, "serviceargs");

            $snmpopts_serial = grab_array_var($inargs, "snmpopts_serial", "");
            if ($snmpopts_serial == "")
                $snmpopts = grab_array_var($inargs, "snmpopts");
            else
                $snmpopts = json_decode(base64_decode($snmpopts_serial), true);

            $services_serial = grab_array_var($inargs, "services_serial", "");
            if ($services_serial == "")
                $services = grab_array_var($inargs, "services");
            else
                $services = json_decode(base64_decode($services_serial), true);

            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");
            if ($serviceargs_serial == "")
                $serviceargs = grab_array_var($inargs, "serviceargs");
            else
                $serviceargs = json_decode(base64_decode($serviceargs_serial), true);


            $output = '
        <input type="hidden" name="ip_address" value="' . encode_form_val($address) . '">
        <input type="hidden" name="port" value="' . encode_form_val($port) . '">
        <input type="hidden" name="hostname" value="' . encode_form_val($hostname) . '">
        <input type="hidden" name="osversion" value="' . encode_form_val($osversion) . '">
        <input type="hidden" name="snmpcommunity" value="' . encode_form_val($snmpcommunity) . '">
        <input type="hidden" name="snmpversion" value="' . encode_form_val($snmpversion) . '">
        <input type="hidden" name="snmpopts_serial" value="' . base64_encode(json_encode($snmpopts)) . '">
        <input type="hidden" name="services_serial" value="' . base64_encode(json_encode($services)) . '">
        <input type="hidden" name="serviceargs_serial" value="' . base64_encode(json_encode($serviceargs)) . '">
';
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:

            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            $hostname = grab_array_var($inargs, "hostname", "");
            $port = intval(grab_array_var($inargs, "port", 161));
            $address = grab_array_var($inargs, "ip_address", "");
            $osversion = grab_array_var($inargs, "osversion", "");
            $snmpcommunity = grab_array_var($inargs, "snmpcommunity", "");
            $snmpversion = grab_array_var($inargs, "snmpversion", "");
            $hostaddress = $address;

            $snmpopts_serial = grab_array_var($inargs, "snmpopts_serial", "");
            $snmpopts = json_decode(base64_decode($snmpopts_serial), true);

            $services_serial = grab_array_var($inargs, "services_serial", "");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            $services = json_decode(base64_decode($services_serial), true);
            $serviceargs = json_decode(base64_decode($serviceargs_serial), true);

            // save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["hostname"] = $hostname;
            $meta_arr["ip_address"] = $address;
            $meta_arr["osversion"] = $osversion;
            $meta_arr["snmpcommunity"] = $snmpcommunity;
            $meta_arr["snmpversion"] = $snmpversion;
            $meta_arr["services"] = $services;
            $meta_arr["serivceargs"] = $serviceargs;
            save_configwizard_object_meta($wizard_name, $hostname, "", $meta_arr);

            $objs = array();

            if (!host_exists($hostname)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_windowssnmp_host",
                    "host_name" => $hostname,
                    "address" => $hostaddress,
                    "icon_image" => "win_server.png",
                    "statusmap_image" => "win_server.png",
                    "_xiwizard" => $wizard_name,
                );
            }

            // determine SNMP args
            $snmpargs = "";
            if (!empty($port) && $port != 161) {
                $snmpargs .= " -p $port";
            }
            if ($snmpcommunity != "" && $snmpversion != "3")
                $snmpargs .= " -C " . escapeshellarg_nowrap($snmpcommunity);
            if ($snmpversion == "2c") {
                $snmpargs .= " --v2c";
            } else if ($snmpversion == "3") {

                $securitylevel = grab_array_var($snmpopts, "v3_security_level");
                $username = grab_array_var($snmpopts, "v3_username");
                $authproto = grab_array_var($snmpopts, "v3_auth_proto");
                $privproto = grab_array_var($snmpopts, "v3_priv_proto");
                $authpassword = grab_array_var($snmpopts, "v3_auth_password");
                $privacypassword = grab_array_var($snmpopts, "v3_privacy_password");

                if ($username != "")
                    $snmpargs .= " --login=" . escapeshellarg_nowrap($username);
                if ($authpassword != "")
                    $snmpargs .= " --passwd=" . escapeshellarg_nowrap($authpassword);
                if ($privacypassword != "")
                    $snmpargs .= " --privpass=" . escapeshellarg_nowrap($privacypassword);

                if ($authproto != "" && $privproto != "") {
                    $snmpargs .= " --protocols=" . escapeshellarg($authproto . "," . $privproto);
                } else if ($authproto != "" ) {
                    $snmpargs .= " --protocols=" . escapeshellarg($authproto);
                }
            }

            // see which services we should monitor
            foreach ($services as $svc => $svcstate) {

                //echo "PROCESSING: $svc -> $svcstate<BR>\n";

                switch ($svc) {

                    case "ping":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Ping",
                            "use" => "xiwizard_windowsserver_ping_service",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "cpu":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "CPU Usage",
                            "use" => "xiwizard_windowssnmp_load",
                            "check_command" => "check_xi_service_snmp_win_load!" . $snmpargs . " -w " . $serviceargs["cpu_warning"] . " -c " . $serviceargs["cpu_critical"] . " -f",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "memory":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Physical Memory Usage",
                            "use" => "xiwizard_windowssnmp_storage",
                            "check_command" => "check_xi_service_snmp_win_storage!" . $snmpargs . " -m 'Physical Memory' -w " . $serviceargs["memory_warning"] . " -c " . $serviceargs["memory_critical"] . " -f",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "pagefile":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Virtual Memory Usage",
                            "use" => "xiwizard_windowssnmp_storage",
                            "check_command" => "check_xi_service_snmp_win_storage!" . $snmpargs . " -m 'Virtual Memory' -w " . $serviceargs["pagefile_warning"] . " -c " . $serviceargs["pagefile_critical"] . " -f",
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "disk":
                        $donedisks = array();
                        $diskid = 0;
                        foreach ($serviceargs["disk"] as $diskname) {

                            if ($diskname == "")
                                continue;

                            //echo "HANDLING DISK: $diskname<BR>";

                            // we already configured this disk
                            if (in_array($diskname, $donedisks))
                                continue;
                            $donedisks[] = $diskname;

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $hostname,
                                "service_description" => "Drive " . $diskname . ": Disk Usage",
                                "use" => "xiwizard_windowssnmp_storage",
                                "check_command" => "check_xi_service_snmp_win_storage!" . $snmpargs . " -m ^" . $diskname . ": -w " . $serviceargs["disk_warning"][$diskid] . " -c " . $serviceargs["disk_critical"][$diskid] . " -f",
                                "_xiwizard" => $wizard_name,
                            );

                            $diskid++;
                        }
                        break;

                    case "servicestate":

                        $enabledservices = $svcstate;
                        foreach ($enabledservices as $sid => $sstate) {

                            $sname = $serviceargs["servicestate"][$sid]["service"];
                            $sdesc = $serviceargs["servicestate"][$sid]["name"];

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $hostname,
                                "service_description" => $sdesc,
                                "use" => "xiwizard_windowssnmp_service",
                                "check_command" => "check_xi_service_snmp_win_service!" . $snmpargs . " -r -n '" . $sname . "'",
                                "_xiwizard" => $wizard_name,
                            );
                        }
                        break;

                    case "processstate":

                        $enabledprocs = $svcstate;
                        foreach ($enabledprocs as $pid => $pstate) {

                            $pname = $serviceargs["processstate"][$pid]["process"];
                            $pdesc = $serviceargs["processstate"][$pid]["name"];

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $hostname,
                                "service_description" => $pdesc,
                                "use" => "xiwizard_windowssnmp_process",
                                "check_command" => "check_xi_service_snmp_win_process!" . $snmpargs . " -r -n '" . $pname . "'",
                                "_xiwizard" => $wizard_name,
                            );
                        }
                        break;


                    default:
                        break;
                }
            }

            //echo "OBJECTS:<BR>";
            //print_r($objs);
            //exit();

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:
            break;
    }

    return $output;
}
