<?php
//
// Windows Event Log Config Wizard
// Copyright (c) 2010-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

windowseventlog_configwizard_init();

function windowseventlog_configwizard_init()
{
    $name = "windowseventlog";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "3.0.3",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _('Monitor Windows event logs via NCPA.'),
        CONFIGWIZARD_DISPLAYTITLE => _('Windows Event Log'),
        CONFIGWIZARD_FUNCTION => 'windowseventlog_configwizard_func',
        CONFIGWIZARD_PREVIEWIMAGE => 'windowseventlog.png',
        CONFIGWIZARD_FILTER_GROUPS => array('windows'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );
    register_configwizard($name, $args);
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function windowseventlog_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    $wizard_name = "windowseventlog";

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {

        case CONFIGWIZARD_MODE_GETSTAGE1HTML:
            $ip_address = grab_array_var($inargs, "ip_address", "");
            $port = grab_array_var($inargs, "port", "5693");
            $token = grab_array_var($inargs, "token", "");
            $no_ssl_verify = grab_array_var($inargs, "no_ssl_verify", 1);

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            // Get variables that were passed to us
            $ip_address = grab_array_var($inargs, "ip_address", "");
            $port = grab_array_var($inargs, "port", "5693");
            $token = grab_array_var($inargs, "token", "");
            $no_ssl_verify = grab_array_var($inargs, "no_ssl_verify", 1);

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (have_value($ip_address) == false) {
                $errmsg[$errors++] = _("No address specified.");
            }

            if (have_value($port) == false) {
                $errmsg[$errors++] = _("No port number specified.");
            }

            // Test the connection if no errors
            if (empty($errors)) {
                $ip_address_replaced = nagiosccm_replace_user_macros($ip_address);
                $port_replaced = nagiosccm_replace_user_macros($port);
                $token_replaced = nagiosccm_replace_user_macros($token);

                // The URL we will use to query the NCPA agent, and do a walk
                // of all monitorable items.
                $query_url = "https://{$ip_address}:{$port}/testconnect/?token=".urlencode($token);
                $query_url_replaced = "https://{$ip_address_replaced}:{$port_replaced}/testconnect/?token=".urlencode($token_replaced);

                // Remove SSL verification or not
                $context = array("ssl" => array("verify_peer" => true, "verify_peer_name" => true));

                if ($no_ssl_verify) {
                    $context['ssl']['verify_peer'] = false;
                    $context['ssl']['verify_peer_name'] = false;
                }

                // All we want to do is test if we can hit this URL.
                $raw_json = file_get_contents($query_url_replaced, false, stream_context_create($context));

                if (empty($raw_json)) {
                    $errmsg[$errors++] = _("Unable to contact server at") . " {$query_url}.";
                } else {
                    $json = json_decode($raw_json, true);

                    if (!array_key_exists('value', $json)) {
                        $errmsg[$errors++] = _("Bad token for connection.");
                    }
                }

            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us
            $ip_address = grab_array_var($inargs, "ip_address", "");
            $port = grab_array_var($inargs, "port", "");
            $token = grab_array_var($inargs, "token", "");
            $no_ssl_verify = grab_array_var($inargs, "no_ssl_verify", 1);
            $hostname = grab_array_var($inargs, 'hostname', gethostbyaddr($ip_address));
            $default_mem_units = grab_array_var($inargs, 'default_mem_units', 'Gi');

            $services_serial = grab_array_var($inargs, "services_serial", "");

            if ($services_serial) {
                $services = json_decode(base64_decode($services_serial), true);
            }

            // Load default services
            if (empty($services)) {
                $services = array();

                $services[] = array(
                    'check' => 1,
                    'name' => _('Application Error Logs'),
                    'log' => 'Application',
                    'severity' => 'ERROR',
                    'event_id' => '',
                    'application' => '',
                    'computer_name' => '',
                    'category' => '',
                    'message' => '',
                );

                $services[] = array(
                    'check' => 1,
                    'name' => _('System Error Logs'),
                    'log' => 'System',
                    'severity' => 'ERROR',
                    'event_id' => '',
                    'application' => '',
                    'computer_name' => '',
                    'category' => '',
                    'message' => '',
                );

                $services[] = array(
                    'check' => 1,
                    'name' => _('Security Error Logs'),
                    'log' => 'Security',
                    'severity' => 'ERROR',
                    'event_id' => '',
                    'application' => '',
                    'computer_name' => '',
                    'category' => '',
                    'message' => '',
                );
            }

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:
            // Get variables that were passed to us
            $ip_address = grab_array_var($inargs, 'ip_address');
            $hostname = grab_array_var($inargs, 'hostname');
            $port = grab_array_var($inargs, 'port');
            $token = grab_array_var($inargs, 'token');

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (is_valid_host_name($hostname) == false) {
                $errmsg[$errors++] = "Invalid host name.";
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE3OPTS:
            $outargs[CONFIGWIZARD_OVERRIDE_OPTIONS]["check_interval"] = 5;
            $outargs[CONFIGWIZARD_OVERRIDE_OPTIONS]["retry_interval"] = 1;
            $outargs[CONFIGWIZARD_OVERRIDE_OPTIONS]["max_check_attempts"] = 1;
            break;

        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // Get variables that were passed to us
            $ip_address = grab_array_var($inargs, 'ip_address');
            $hostname = grab_array_var($inargs, 'hostname');
            $port = grab_array_var($inargs, 'port');
            $token = grab_array_var($inargs, 'token');
            $services = grab_array_var($inargs, 'services', array());

            $output = '
            <input type="hidden" name="ip_address" value="' . encode_form_val($ip_address) . '" />
            <input type="hidden" name="hostname" value="' . encode_form_val($hostname) . '" />
            <input type="hidden" name="port" value="' . encode_form_val($port) . '" />
            <input type="hidden" name="token" value="' . encode_form_val($token) . '" />
            <input type="hidden" name="services_serial" value="' . base64_encode(json_encode($services)) . '" />';

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:
            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:
            $output = '';
            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            // Get variables that were passed to us
            $ip_address = grab_array_var($inargs, 'ip_address');
            $hostname = grab_array_var($inargs, 'hostname');
            $port = grab_array_var($inargs, 'port');
            $token = grab_array_var($inargs, 'token');
            $services_serial = grab_array_var($inargs, "services_serial", "");
            $services = json_decode(base64_decode($services_serial), true);

            // Escape values for check_command line
            if (function_exists('nagiosccm_replace_command_line')) {
                $token = nagiosccm_replace_command_line($token, '', '!');
            }

            $objs = array();

            // Host
            if (!host_exists($hostname)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_ncpa_host",
                    "host_name" => $hostname,
                    "address" => $ip_address,
                    "icon_image" => "windowseventlog.png",
                    "statusmap_image" => "windowseventlog.png",
                    "_xiwizard" => $wizard_name
                );
            }

            // Common plugin opts
            $commonopts = '-t ' . escapeshellarg_nowrap($token) . ' ';
            if ($port) {
                $commonopts .= "-P $port ";
            }

            // Services
            foreach ($services as $svc) {

                $pluginopts = $commonopts;

                // Verify we want to add the service
                if (trim($svc['name']) == "" || empty($svc['check'])) {
                    continue;
                }

                // Create query for plugin
                $query = 'name='.$svc['log'].',logged_after=30m';

                if (!empty($svc['severity'])) {
                    $query .= ',severity='.$svc['severity'];
                }
                if (!empty($svc['event_id'])) {
                    $query .= ',event_id='.$svc['event_id'];
                }
                if (!empty($svc['application'])) {
                    $query .= ',application='.$svc['application'];
                }
                if (!empty($svc['computer_name'])) {
                    $query .= ',computer_name='.$svc['computer_name'];
                }
                if (!empty($svc['category'])) {
                    $query .= ',category='.$svc['category'];
                }
                if (!empty($svc['message'])) {
                    $query .= ',message='.$svc['message'];
                }

                $pluginopts .= " -M 'logs' -q '".$query."' -c 0";

                $objs[] = array(
                    'type' => OBJECTTYPE_SERVICE,
                    'host_name' => $hostname,
                    'service_description' => $svc['name'],
                    'use' => 'xiwizard_ncpa_service',
                    'max_check_attempts' => 1,
                    'icon_image' => 'windowseventlog.png',
                    "statusmap_image" => "windowseventlog.png",
                    'check_command' => "check_xi_ncpa!" . $pluginopts,
                    '_xiwizard' => $wizard_name
                );
            }

            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;
            break;

        default:
            break;
    }

    return $output;
}
