<?php
//
// Website URL Config Wizard
// Copyright (c) 2008-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

websiteurl_configwizard_init();

function websiteurl_configwizard_init()
{
    $name = "websiteurl";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.0.1",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor a specific web URL."),
        CONFIGWIZARD_DISPLAYTITLE => _("Website URL"),
        CONFIGWIZARD_FUNCTION => "websiteurl_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "www_server.png",
        CONFIGWIZARD_FILTER_GROUPS => array('website'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );
    register_configwizard($name, $args);
}

/**
 * @param string $mode
 * @param        $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function websiteurl_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    $wizard_name = "websiteurl";

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            $url = grab_array_var($inargs, "url", "http://");

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            // Get variables that were passed to us
            $url = grab_array_var($inargs, "url");
            $url = nagiosccm_replace_user_macros($url);

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (have_value($url) == false) {
                $errmsg[$errors++] = _("No URL specified.");
            } else if (!valid_url($url)) {
                $errmsg[$errors++] = _("Invalid URL.");
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us
            $url = grab_array_var($inargs, "url");
            $urlparts = parse_url($url);

            $hostname = grab_array_var($urlparts, "host");
            $urlscheme = grab_array_var($urlparts, "scheme");
            $port = grab_array_var($urlparts, "port");
            $username = grab_array_var($urlparts, "user");
            $password = grab_array_var($urlparts, "pass");

            if ($urlscheme == "https") {
                $ssl = "on";
            } else {
                $ssl = "off";
            }

            if ($port == "") {
                if ($ssl == "on") {
                    $port = 443;
                } else {
                    $port = 80;
                }
            }

            $basicauth = "";

            if ($username != "") {
                $basicauth = "on";
            }

            $ip = gethostbyname($hostname);

            $hostname = grab_array_var($inargs, "hostname", $hostname);
            $servicename = grab_array_var($inargs, "servicename", websiteurl_configwizard_url_to_name($url));
            $port = grab_array_var($inargs, "port", $port);
            $ssl = grab_array_var($inargs, "ssl", $ssl);
            $basicauth = grab_array_var($inargs, "basicauth", $basicauth);
            $username = grab_array_var($inargs, "username", $username);
            $password = grab_array_var($inargs, "password", $password);

            $services = grab_array_var($inargs, "services", array());
            $services_serial = grab_array_var($inargs, "services_serial");

            if ($services_serial != "") {
                $services = json_decode(base64_decode($services_serial), true);
            }

            $serviceargs = grab_array_var($inargs, "serviceargs", array());
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial");

            if ($serviceargs_serial != "") {
                $serviceargs = json_decode(base64_decode($serviceargs_serial), true);
            }

            if (count($services) == 0) {
                $services["http"] = "on";
                $services["httpcontent"] = "";
                $services["httpregex"] = "";
            }

            if (count($serviceargs) == 0) {
                $serviceargs["httpservicename"] = "URL Status";
                $serviceargs["httpcontentservicename"] = "URL Content";
                $serviceargs["httpregexservicename"] = "URL Content Regex";

                $serviceargs["httpcontentstr"] = "";
                $serviceargs["httpregexstr"] = "";
            }

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // Get variables that were passed to us
            $url = grab_array_var($inargs, "url");
            $hostname = grab_array_var($inargs, "hostname");
            $servicename = grab_array_var($inargs, "servicename");
            $services = grab_array_var($inargs, "services", array());
            $serviceargs = grab_array_var($inargs, "serviceargs", array());

            $hostname = nagiosccm_replace_user_macros($hostname);
            $port = nagiosccm_replace_user_macros($port);
            $username = nagiosccm_replace_user_macros($username);
            $password = nagiosccm_replace_user_macros($password);

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (is_valid_host_name($hostname) == false)
                $errmsg[$errors++] = _("Invalid host name.");

            if (is_valid_service_name($servicename) == false)
                $errmsg[$errors++] = _("Invalid service prefix.  Can only contain alphanumeric characters, spaces, and the following: <b>.\:_-</b>");

            if (have_value($url) == false)
                $errmsg[$errors++] = _("No URL specified.");
            else if (!valid_url($url))
                $errmsg[$errors++] = _("Invalid URL.");

            if (array_key_exists("httpcontent", $services)) {
                if (array_key_exists("httpcontentstr", $serviceargs)) {
                    if (have_value($serviceargs["httpcontentstr"]) == false)
                        $errmsg[$errors++] = _("You must specify a string to expect in the web page content.");
                }
                if ($serviceargs["httpcontentservicename"] == "")
                    $errmsg[$errors++] = _("You must specify a service name for the URL Content service.");
                else if (is_valid_service_name($serviceargs["httpcontentservicename"]) == false)
                    $errmsg[$errors++] = _("Invalid URL Content service name.");
            }

            if (array_key_exists("httpregex", $services)) {
                if (array_key_exists("httpregexstr", $serviceargs)) {
                    if (have_value($serviceargs["httpregexstr"]) == false)
                        $errmsg[$errors++] = _("You must specify a regular expression to expect in the web page content.");
                }
                if ($serviceargs["httpregexservicename"] == "")
                    $errmsg[$errors++] = _("You must specify a service name for the URL Content Regular Expression Match service.");
                else if (is_valid_service_name($serviceargs["httpregexservicename"]) == false)
                    $errmsg[$errors++] = _("Invalid URL Content Regular Expression Match service name.");
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;


        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // Get variables that were passed to us
            $url = grab_array_var($inargs, "url");
            $servicename = grab_array_var($inargs, "servicename");
            $hostname = grab_array_var($inargs, "hostname");
            $ip = grab_array_var($inargs, "ip");
            $ssl = grab_array_var($inargs, "ssl");
            $port = grab_array_var($inargs, "port");
            $username = grab_array_var($inargs, "username");
            $password = grab_array_var($inargs, "password");
            $services = grab_array_var($inargs, "services");
            $serviceargs = grab_array_var($inargs, "serviceargs");

            $services_serial = grab_array_var($inargs, "services_serial", base64_encode(json_encode($services)));
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", base64_encode(json_encode($serviceargs)));

            $output = '
        <input type="hidden" name="url" value="' . encode_form_val($url) . '">
        <input type="hidden" name="hostname" value="' . encode_form_val($hostname) . '">
        <input type="hidden" name="servicename" value="' . encode_form_val($servicename) . '">
        <input type="hidden" name="ip" value="' . encode_form_val($ip) . '">
        <input type="hidden" name="ssl" value="' . encode_form_val($ssl) . '">
        <input type="hidden" name="port" value="' . encode_form_val($port) . '">
        <input type="hidden" name="username" value="' . encode_form_val($username) . '">
        <input type="hidden" name="password" value="' . encode_form_val($password) . '">
        <input type="hidden" name="services_serial" value="' . $services_serial . '">
        <input type="hidden" name="serviceargs_serial" value="' . $serviceargs_serial . '">
';
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:

            $output = '

            ';
            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            $hostname = grab_array_var($inargs, "hostname", "");
            $servicename = grab_array_var($inargs, "servicename");
            $ip = grab_array_var($inargs, "ip", "");
            $url = grab_array_var($inargs, "url", "");
            $ssl = grab_array_var($inargs, "ssl");
            $port = grab_array_var($inargs, "port");
            $username = grab_array_var($inargs, "username");
            $password = grab_array_var($inargs, "password");

            $services_serial = grab_array_var($inargs, "services_serial", "");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            $services = json_decode(base64_decode($services_serial), true);
            $serviceargs = json_decode(base64_decode($serviceargs_serial), true);

            $urlparts = parse_url($url);
            $hostaddress = $urlparts["host"];

            // save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["hostname"] = $hostname;
            $meta_arr["servicename"] = $servicename;
            $meta_arr["ip"] = $ip;
            $meta_arr["url"] = $url;
            $meta_arr["ssl"] = $ssl;
            $meta_arr["port"] = $port;
            $meta_arr["username"] = $username;
            $meta_arr["password"] = $password;
            $meta_arr["services"] = $services;
            $meta_arr["serivceargs"] = $serviceargs;
            save_configwizard_object_meta($wizard_name, $hostname, "", $meta_arr);

            $objs = array();

            if (!host_exists($hostname)) {

                // Choose the port for check_tcp
                $tmp_port = $port;
                if ($port == "") {
                    $tmp_port = "80";
                    if ($ssl == "on") {
                        $tmp_port = "443";
                    }
                }

                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_website_host",
                    "host_name" => $hostname,
                    "address" => $hostaddress,
                    "icon_image" => "www_server.png",
                    "statusmap_image" => "www_server.png",
                    "check_command" => "check_tcp!$tmp_port!",
                    "_xiwizard" => $wizard_name,
                );
            }

            $pluginopts = "";

            $vhost = $urlparts["host"];

            $pluginopts .= " -f ok"; // on redirect, follow (OK status)

            if (!empty($ip)) {
                $pluginopts .= " -I " . $ip; // ip address
            }

            $urlpath = grab_array_var($urlparts, "path", "");

            // Need to add query (after ?) and fragment (after #) back on -JO

            if (!empty($urlparts["query"])) {
                $urlpath .= "?" . $urlparts["query"];
            }

            if (!empty($urlparts["fragment"])) {
                $urlpath .= "#" . $urlparts["fragment"];
            }

            if ($urlpath == "")
                $urlpath = "/";
            $pluginopts .= " -u '" . $urlpath . "'";

            if ($ssl == "on")
                $pluginopts .= " -S --sni";
            if ($port != "")
                $pluginopts .= " -p " . $port;
            if ($username != "")
                $pluginopts .= " -a '" . $username . ":" . $password . "'";


            // see which services we should monitor
            foreach ($services as $svc => $svcstate) {

                //echo "PROCESSING: $svc -> $svcstate<BR>\n";

                switch ($svc) {

                    case "http":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => $servicename . " " . $serviceargs["httpservicename"],
                            "use" => "xiwizard_website_http_service",
                            "check_command" => "check_xi_service_http!" . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "httpcontent":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => $servicename . " " . $serviceargs["httpcontentservicename"],
                            "use" => "xiwizard_website_http_content_service",
                            //"check_command" => "check_xi_service_http_content!".$serviceargs["httpcontentstr"],
                            "check_command" => "check_xi_service_http!-s \"" . $serviceargs["httpcontentstr"] . "\" " . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "httpregex":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => $servicename . " " . $serviceargs["httpregexservicename"],
                            "use" => "xiwizard_website_http_content_service",
                            "check_command" => "check_xi_service_http!-r \"" . $serviceargs["httpregexstr"] . "\" " . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;


                    default:
                        break;
                }
            }

            //echo "OBJECTS:<BR>";
            //print_r($objs);
            //exit();

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:
            break;
    }

    return $output;
}

/**
 * @param $url
 *
 * @return mixed|string
 */
function websiteurl_configwizard_url_to_name($url)
{

    $urlparts = parse_url($url);
    $path = grab_array_var($urlparts, "path", "");

    $path = str_replace("/", "_", $path);
    $path = str_replace("\\", "_", $path);
    $path = str_replace("?", "_", $path);
    $path = str_replace(";", "_", $path);
    $path = str_replace("&", "_", $path);
    $path = str_replace(":", "_", $path);

    if ($path == "")
        $path = "_";

    return $path;
}
